/*
 * Contact (model)
 * A contact on the contact list
 *
 * Copyright (C) 2001 Barnaby Gray <barnaby@beedesign.co.uk>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

#ifndef CONTACT_H
#define CONTACT_H

#include <list>
#include <string>

#include <sigc++/signal_system.h>

#include <libicq2000/constants.h>
#include <libicq2000/ref_ptr.h>

#include <libicq2000/Capabilities.h>

using std::list;
using std::string;

namespace ICQ2000 {

  // -- Status Codes Flags --
  const unsigned short STATUS_FLAG_ONLINE = 0x0000;
  const unsigned short STATUS_FLAG_AWAY = 0x0001;
  const unsigned short STATUS_FLAG_DND = 0x0002;
  const unsigned short STATUS_FLAG_NA = 0x0004;
  const unsigned short STATUS_FLAG_OCCUPIED = 0x0010;
  const unsigned short STATUS_FLAG_FREEFORCHAT = 0x0020;
  const unsigned short STATUS_FLAG_INVISIBLE = 0x0100;

  class MessageEvent;
  class StatusChangeEvent;
  class UserInfoChangeEvent;

  class Contact {
   public:
    // reference count
    unsigned int count;

    // Inner classes for various sections of Contact details

    class MainHomeInfo {
      string cellular, normalised_cellular;
      // cellular private - access must be through
      // get/setMobileNo for consistency
    
      void normaliseMobileNo();

    public:
      MainHomeInfo();

      string alias, firstname, lastname, email, city, state, phone, fax, street, zip;
      unsigned short country;
      signed char timezone;

      string getCountry() const;
      string getMobileNo() const;
      void setMobileNo(const string& s);

      string getNormalisedMobileNo() const;
    };

    class HomepageInfo {
    public:
      HomepageInfo();

      unsigned char age, sex;
      string homepage;
      unsigned short birth_year;
      unsigned char birth_month, birth_day, lang1, lang2, lang3;

      string getBirthDate() const;
      string getLanguage(int l) const;
    };

    class EmailInfo {
    private:
      list<string> email_list;

    public:
      EmailInfo();

      void addEmailAddress(const string&);
    };
  
    class WorkInfo {
    public:
      WorkInfo();
    
      string city, state, street, zip;
      unsigned short country;
      string company_name, company_dept, company_position, company_web;
    };

    class BackgroundInfo {
    public:
      typedef std::pair<unsigned short, string> School;
      list<School> schools;   // school names

      BackgroundInfo();

      void addSchool(unsigned short cat, const string& s);
    };

    class PersonalInterestInfo {
    public:
      typedef std::pair<unsigned short, string> Interest;
      list<Interest> interests;

      PersonalInterestInfo();
    
      void addInterest(unsigned short cat, const string& s);
    };

  private:
    void Init();
    bool m_icqcontact;
    bool m_virtualcontact;

    // static fields
    unsigned int m_uin;

    // dynamic fields - updated when they come online
    unsigned char m_tcp_version;
    Status m_status;
    bool m_invisible;
    bool m_authreq;
    bool m_direct;
    unsigned int m_ext_ip, m_lan_ip;
    unsigned short m_ext_port, m_lan_port;
    Capabilities m_capabilities;
    unsigned int m_signon_time, m_last_online_time, m_last_status_change_time;
    unsigned int m_last_message_time, m_last_away_msg_check_time;

    static unsigned int imag_uin;
    
    // other fields
    unsigned short m_seqnum;

    // detailed fields
    MainHomeInfo m_main_home_info;
    HomepageInfo m_homepage_info;
    EmailInfo m_email_info;
    WorkInfo m_work_info;
    PersonalInterestInfo m_personal_interest_info;
    BackgroundInfo m_background_info;
    string m_about;

  public:
    Contact();

    Contact(unsigned int uin);
    Contact(const string& a);

    unsigned int getUIN() const;
    void setUIN(unsigned int uin);
    string getStringUIN() const;
    string getMobileNo() const;
    string getNormalisedMobileNo() const;
    string getAlias() const;
    string getFirstName() const;
    string getLastName() const;
    string getEmail() const;

    string getNameAlias() const;

    Status getStatus() const;
    string getStatusStr() const;
    bool isInvisible() const;
    bool getAuthReq() const;

    unsigned int getExtIP() const;
    unsigned int getLanIP() const;
    unsigned short getExtPort() const;
    unsigned short getLanPort() const;
    unsigned char getTCPVersion() const;
    bool get_accept_adv_msgs() const;
    Capabilities get_capabilities() const;

    unsigned int get_signon_time() const;
    unsigned int get_last_online_time() const;
    unsigned int get_last_status_change_time() const;
    unsigned int get_last_message_time() const;
    unsigned int get_last_away_msg_check_time() const;

    void setMobileNo(const string& mn);
    void setAlias(const string& al);
    void setFirstName(const string& fn);
    void setLastName(const string& ln);
    void setEmail(const string& em);
    void setAuthReq(bool b);

    bool getDirect() const;
    void setDirect(bool b);

    void setStatus(Status st, bool i);
    void setStatus(Status st);
    void setInvisible(bool i);
    void setExtIP(unsigned int ip);
    void setLanIP(unsigned int ip);
    void setExtPort(unsigned short port);
    void setLanPort(unsigned short port);
    void setTCPVersion(unsigned char v);
    void set_capabilities(const Capabilities& c);

    void set_signon_time(unsigned int t);
    void set_last_online_time(unsigned int t);
    void set_last_status_change_time(unsigned int t);
    void set_last_message_time(unsigned int t);
    void set_last_away_msg_check_time(unsigned int t);

    void setMainHomeInfo(const MainHomeInfo& m);
    void setHomepageInfo(const HomepageInfo& s);
    void setEmailInfo(const EmailInfo &e);
    void setWorkInfo(const WorkInfo &w);
    void setInterestInfo(const PersonalInterestInfo& p);
    void setBackgroundInfo(const BackgroundInfo& b);
    void setAboutInfo(const string& about);

    MainHomeInfo& getMainHomeInfo();
    HomepageInfo& getHomepageInfo();
    EmailInfo& getEmailInfo();
    WorkInfo& getWorkInfo();
    BackgroundInfo& getBackgroundInfo();
    PersonalInterestInfo& getPersonalInterestInfo();
    const string& getAboutInfo() const;

    bool isICQContact() const;
    bool isVirtualContact() const;

    bool isSMSable() const;

    unsigned short nextSeqNum();

    SigC::Signal1<void,StatusChangeEvent*> status_change_signal;
    SigC::Signal1<void,UserInfoChangeEvent*> userinfo_change_signal;

    void userinfo_change_emit();
    void userinfo_change_emit(bool is_transient_detail);

    static string UINtoString(unsigned int uin);
    static unsigned int StringtoUIN(const string& s);
    
    static unsigned short MapStatusToICQStatus(Status st, bool inv);
    static Status MapICQStatusToStatus(unsigned short st);
    static bool MapICQStatusToInvisible(unsigned short st);

    static unsigned int nextImaginaryUIN();
  };

  typedef ref_ptr<Contact> ContactRef;
}

#endif
