/* This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */


#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/socket.h>

#include <glib.h>

#include <string.h>

#include "icq.h"
#include "libicq.h"
#include "receive.h"
#include "tcp.h"

const guint32 LOCALHOST = 0x0100007F;

extern gint Verbose;
extern guint32 our_ip;
extern guint32 our_port;
extern gint tcp_sok;
extern guint16 seq_num;
extern Contact_Member Contacts[];
extern gint Num_Contacts;
extern guint32 Current_Status;

/* #define DEBUG */

void packet_print(guint8* packet, gint size)
{
  int cx;

#ifdef DEBUG
  printf( "\nPacket:\n" );

  for( cx = 0; cx < size; cx ++ )
  {
    if( cx % 16 == 0 && cx )
      printf( "  \n");
    printf("%02x ", packet[cx] );
  }
  printf( "\n" );
#endif
}


/* might need fcntl(F_SETFL), or ioctl(FIONBIO) */
/* Posix.1g says fcntl */

#ifdef O_NONBLOCK

int set_nonblock(int fd)
{
  int flags = fcntl(fd, F_GETFL, 0);
  if(flags == -1)
    return -1;
  return fcntl(fd, F_SETFL, flags | O_NONBLOCK);
}

int set_block(int fd)
{
  int flags = fcntl(fd, F_GETFL, 0);
  if(flags == -1)
    return -1;
  return fcntl(fd, F_SETFL, flags ^ O_NONBLOCK);
}

#else

int set_nonblock(int fd)
{
  int yes = 1;
  return ioctl(fd, FIONBIO, &yes);
}

int set_block(int fd)
{
  int yes = 0;
  return ioctl(fd, FIONBIO, &yes);
}

#endif

gint tcp_error_message(gint error)
{
   switch(error)
   {
      case EBADF:
         if(Verbose & ICQ_VERB_INFO) 
            printf("\nTCP_Connect(): Bad descriptor");
         return -1;

      case EFAULT:
         if(Verbose & ICQ_VERB_INFO)
            printf("\nTCP_Connect(): The socket structure address is outside your address space");
         return -1;

      case ENOTSOCK:
         if(Verbose & ICQ_VERB_INFO)
            printf("\nTCP_Connect(): The descriptor is not associated with a socket");
         return -1;

      case EISCONN:
         if(Verbose & ICQ_VERB_INFO)
            printf("\nTCP_Connect(): The socket is already connected");
         return -2;

      case ECONNREFUSED:
         if(Verbose & ICQ_VERB_INFO)
            printf("\nTCP_Connect(): Connection refused");
         return -1;

      case ETIMEDOUT:
         if(Verbose & ICQ_VERB_INFO)
            printf("\nTCP_Connect(): Timeout while attempting connection");
         return -1;

      case ENETUNREACH:
         if(Verbose & ICQ_VERB_INFO)
            printf("\nTCP_Connect(): Network is unreachable");
         return -1;

      case EADDRINUSE:
         if(Verbose & ICQ_VERB_INFO)
            printf("\nTCP_Connect(): Address is already in use");
         return -1;

      case EINPROGRESS:
         if(Verbose & ICQ_VERB_INFO)
            printf("\nTCP_Connect(): Connection in progress");
         return -2;

      case EALREADY:
         if(Verbose & ICQ_VERB_INFO)
            printf("\nTCP_Connect(): A previous connection attempt has not been completed");
         return -1;
   }
   
   return -1;
}

gint TCP_Ack(gint sock, guint16 cmd, gint seq)
{
   char buffer[1024];
   guint16 intsize;
   char* sent_message;

   typedef struct
   {
      guint8 uin1[4];
      guint8 version[2];
      guint8 command[2];
      guint8 zero[2];
      guint8 uin2[4];
      guint8 cmd[2];
      guint8 message_length[2];
   } tcp_head;

   typedef struct
   {
      guint8 ip[4];
      guint8 ip_real[4];
      guint8 port[4];
      guint8 junk;
      guint8 status[4];
      guint8 seq[4];
   } tcp_tail;

   tcp_head pack_head;
   tcp_tail pack_tail;

#ifdef DEBUG
   fprintf(stderr, "\nTCP> TCP_Ack(%d, %04X, %d)", sock, cmd, seq);
#endif

   sent_message = "";
/*
   if( Current_Status != STATUS_ONLINE && Current_Status != STATUS_FREE_CHAT )
      sent_message = Away_Message;
*/
   
   DW_2_Chars( pack_head.uin1, UIN );
   Word_2_Chars( pack_head.version, 0x0003 );
   Word_2_Chars( pack_head.command, ICQ_CMDxTCP_ACK );
   Word_2_Chars( pack_head.zero, 0x0000 );
   DW_2_Chars( pack_head.uin2, UIN );
   DW_2_Chars( pack_head.cmd, cmd );
   DW_2_Chars( pack_head.message_length, strlen( sent_message ) + 1 );
   
   DW_2_Chars( pack_tail.ip, our_ip );
   DW_2_Chars( pack_tail.ip_real, LOCALHOST );
   DW_2_Chars( pack_tail.port, our_port );
   pack_tail.junk = 0x04;
   DW_2_Chars( pack_tail.seq, seq );

   switch(Current_Status)
   {
      case STATUS_ONLINE:
         DW_2_Chars( pack_tail.status, ICQ_ACKxTCP_ONLINE );
         break;
      case STATUS_AWAY:
         DW_2_Chars( pack_tail.status, ICQ_ACKxTCP_AWAY );
         break;
      case STATUS_DND:
         DW_2_Chars( pack_tail.status, ICQ_ACKxTCP_DND );
         break;
      case STATUS_OCCUPIED:
         DW_2_Chars( pack_tail.status, ICQ_ACKxTCP_OCC );
         break;
      case STATUS_NA:
         DW_2_Chars( pack_tail.status, ICQ_ACKxTCP_NA );
         break;
      case STATUS_INVISIBLE:
         DW_2_Chars( pack_tail.status, ICQ_ACKxTCP_REFUSE );
         break;
      }

   if(sock != -1)
   {
      intsize = sizeof( tcp_head ) + sizeof( tcp_tail ) + strlen( sent_message ) + 1;
      memcpy(&buffer[0], &intsize, 2 );
      memcpy(&buffer[2], &pack_head, sizeof( pack_head ) );
      memcpy(&buffer[2 + sizeof(pack_head)], sent_message, strlen(sent_message) + 1);
      memcpy(&buffer[2 + sizeof(pack_head) + strlen(sent_message) + 1], &pack_tail, sizeof(pack_tail));
      write(sock, buffer, intsize + 2);
      packet_print(buffer, intsize + 2);
   }
   else
   {
      return -1;
   }
   
   return 1;
}

void TCP_SendHelloPacket(gint sock)
{

   struct
   {
      guint8 size[2];
      guint8 command;
      guint8 version[4];
      guint8 szero[4];
      guint8 uin[4];
      guint8 ip[4];
      guint8 real_ip[4];
      guint8 four;
      guint8 port[4];
   } hello_packet;

#ifdef DEBUG
   fprintf(stderr, "\nTCP> TCP_SendHelloPacket(%d)", sock);
#endif

   Word_2_Chars( hello_packet.size, 0x001A );
   hello_packet.command = 0xFF;
   DW_2_Chars( hello_packet.version, 0x00000003 );
   DW_2_Chars( hello_packet.szero, 0x00000000 );
   DW_2_Chars( hello_packet.uin, UIN );
   DW_2_Chars( hello_packet.ip, LOCALHOST );
   DW_2_Chars( hello_packet.real_ip, LOCALHOST );
   hello_packet.four = 0x04;
   DW_2_Chars( hello_packet.port, (guint32)our_port );

   write(sock, &hello_packet, sizeof(hello_packet));
}


void TCP_ProcessPacket(guint8* packet, gint packet_length, gint sock)
{
   CLIENT_MESSAGE c_mesg;
   int chat_sock;
   int cx;

   typedef struct
   {
      guint32 uin1;
      guint16 version;
      guint16 command;
      guint16 zero;
      guint32 uin2;
      guint16 cmd;
      guint16 message_length;
   } tcp_head;

   typedef struct
   {
      guint32 ip_sender;
      guint32 ip_local;
      guint32 port;
      guint8 junk;
      guint32 status;
      guint32 chat_port;
      guint32 seq;
   } tcp_tail;

   char *message, *data, *tmp;
   char *away_message;
   tcp_head pack_head;
   tcp_tail pack_tail;

   gint cindex;

   guint32 i;

#ifdef DEBUG
   fprintf(stderr, "\nTCP> TCP_ProcessPacket(%04X, %d, %d)", packet, packet_length, sock);
#endif

   if(packet[0] == 0xFF) /* 0xFF means it's just a "Hello" packet; ignore */
      return;

   memcpy(&pack_head.uin1, packet, 4);
   memcpy(&pack_head.version, (packet + 4), 2);
   memcpy(&pack_head.command, (packet + 6), 2);
   memcpy(&pack_head.zero, (packet + 8), 2);
   memcpy(&pack_head.uin2, (packet + 10), 4);
   memcpy(&pack_head.cmd, (packet + 14), 2); 
   memcpy(&pack_head.message_length, (packet + 16), 2);
  
   message = (char *)g_malloc(pack_head.message_length);
   memcpy(message, (packet + 18), pack_head.message_length);
  
   memcpy(&pack_tail.ip_sender, (packet + 18 + pack_head.message_length), 4);
   memcpy(&pack_tail.ip_local, (packet + 18 + pack_head.message_length + 4), 4);
   memcpy(&pack_tail.port, (packet + 18 + pack_head.message_length + 8), 4);
   memcpy(&pack_tail.status, (packet + 18 + pack_head.message_length + 13), 4);
   memcpy(&pack_tail.seq, (packet + packet_length - 4), 4);
   memcpy(&pack_tail.chat_port, (packet + packet_length - 8), 4);

   i = pack_tail.ip_sender;
   pack_tail.ip_sender = ((i << 24)  |  ((i & 0xff00) << 8)  |  ((i & 0xff0000) >> 8)  |  (i >> 24));
   i = pack_tail.ip_local;
   pack_tail.ip_local = ((i << 24)  |  ((i & 0xff00) << 8)  |  ((i & 0xff0000) >> 8)  |  (i >> 24));

   if(pack_head.command == ICQ_CMDxTCP_START)
   {
      switch( pack_head.cmd )
      {
         case ICQ_CMDxTCP_MSG:
            c_mesg.uin = pack_head.uin1;
            c_mesg.year = 0;
            c_mesg.month = 0;
            c_mesg.day = 0;
            c_mesg.hour = 0;
            c_mesg.minute = 0;
            c_mesg.type = MSG_MESS;
            c_mesg.len = strlen(message) + 1;
            c_mesg.msg = message;

if (Verbose & ICQ_VERB_INFO)
   printf("\nTCP_ProcessPacket(): Received message through tcp");

            TCP_Ack(sock, pack_head.cmd, pack_tail.seq);

            if(event[EVENT_MESSAGE] != NULL)
               (*event[EVENT_MESSAGE])(&c_mesg);

            break;

         case ICQ_CMDxTCP_URL:
            c_mesg.uin = pack_head.uin1;
            c_mesg.year = 0;
            c_mesg.month = 0;
            c_mesg.day = 0;
            c_mesg.hour = 0;
            c_mesg.minute = 0;
            c_mesg.type = URL_MESS;
            c_mesg.len = strlen(message) + 1;
            
            data = message;
            tmp = (char *)strchr(data, '\xfe');
            if(!tmp) return;
            *tmp = 0;
            c_mesg.msg = data;
      
            data = ++tmp;
            c_mesg.url = data;
if (Verbose & ICQ_VERB_INFO)
   printf("\nTCP_ProcessPacket(): Received URL through tcp");

            TCP_Ack(sock, pack_head.cmd, pack_tail.seq);

            if(event[EVENT_MESSAGE] != NULL)
               (*event[EVENT_MESSAGE])(&c_mesg);

         case ICQ_CMDxTCP_READxAWAYxMSG:
         case ICQ_CMDxTCP_READxOCCxMSG:
         case ICQ_CMDxTCP_READxDNDxMSG:
         case ICQ_CMDxTCP_READxNAxMSG:
            for(cindex = 0; cindex < Num_Contacts; cindex++)
               if(Contacts[ cindex ].uin == pack_head.uin2) break;

            if(Current_Status != STATUS_ONLINE && Current_Status != STATUS_FREE_CHAT && 
                  cindex != Num_Contacts)
               TCP_Ack(sock, ICQ_CMDxTCP_READxAWAYxMSG, pack_tail.seq);

                   break;

         case ICQ_CMDxTCP_CHAT:
            if(Verbose & ICQ_VERB_INFO) printf("\nReceived chat request");
/*            cindex = Do_Chat( 0, message, pack_head.uin1, data, pack_tail.seq );
            if(Current_Status == STATUS_FREE_CHAT)
            TCP_AcceptChat(sock, cindex, pack_tail.seq);  */
            break;

         case ICQ_CMDxTCP_FILE:
            if(Verbose & ICQ_VERB_INFO) printf("\nReceived file transfer request");
            break;      
      }
   }

   if(pack_head.command == ICQ_CMDxTCP_ACK)
   {
      switch(pack_head.cmd)
      {
         case ICQ_CMDxTCP_MSG:
if (Verbose & ICQ_VERB_INFO)
   printf("\nTCP_ProcessPacket(): Message sent successfully - seq = %d", pack_tail.seq);
break;

         case ICQ_CMDxTCP_URL:
if (Verbose & ICQ_VERB_INFO)
   printf("\nTCP_ProcessPacket(): URL sent successfully");
break;
         
         case ICQ_CMDxTCP_READxAWAYxMSG:
         case ICQ_CMDxTCP_READxOCCxMSG:
         case ICQ_CMDxTCP_READxDNDxMSG:
         case ICQ_CMDxTCP_READxNAxMSG:
            for( cx = 0; cx < Num_Contacts; cx ++ )
               if(Contacts[ cx ].uin == pack_head.uin2) break;
         
            if(pack_tail.status == ICQ_ACKxTCP_AWAY ||
               pack_tail.status == ICQ_ACKxTCP_NA   ||
               pack_tail.status == ICQ_ACKxTCP_DND  ||
               pack_tail.status == ICQ_ACKxTCP_OCC )
            {
               Rec_AwayMessage(Contacts[cx].uin, message);
            }
            break;
               
         case ICQ_CMDxTCP_CHAT:
/*            if(pack_tail.chat_port > 0)
               chat_sock = TCP_ConnectChat(pack_tail.chat_port, pack_head.uin1);  */
                  
            break;
               
         case ICQ_CMDxTCP_FILE:
            if (Verbose & ICQ_VERB_INFO)
               printf( "Received file transfer ack\n" );
            break;
      }
   }

   if(pack_head.command == ICQ_CMDxTCP_CANCEL)
   {
      switch ( pack_head.cmd )
      {
         case ICQ_CMDxTCP_CHAT:
            if (Verbose & ICQ_VERB_INFO)
               printf( "Chat request cancelled\n" );
            break;
      
         case ICQ_CMDxTCP_FILE:
            if (Verbose & ICQ_VERB_INFO)
               printf( "File transfer cancelled\n" );
            break;
      }
   }

   g_free(message);
} 


gint TCP_ReadPacket(gint sock)
{
   guint32 uin;
   gint cindex;
   guint16 packet_size;
   gint real_size, addr_size;
   guint8* packet;
   struct sockaddr_in addr;

#ifdef DEBUG
   fprintf(stderr, "\nTCP> TCP_ReadPacket(%d)", sock);
#endif

   if(sock == tcp_sok)
   {
     addr_size = sizeof(struct sockaddr_in);
     sock = accept(sock, &addr, &addr_size);
     set_nonblock(sock);
   }

   for(cindex = 0; cindex < Num_Contacts; cindex++)
      if(Contacts[cindex].sok == sock) break;

   if(recv(sock, &packet_size, 2, MSG_PEEK) <= 0)
   {
      if(errno == EWOULDBLOCK) return 1;

      if(cindex != Num_Contacts)
      {
        Contacts[cindex].sok = 0;
        Contacts[cindex].connected = 0;
      }

      close(sock);
      return TRUE;
   }

   packet = (guint8*)g_malloc(packet_size + 2);
   real_size = recv(sock, packet, packet_size + 2, MSG_PEEK);
 
   if(real_size < packet_size)
   {
     if(real_size >= 0 || (real_size == -1 && errno == EWOULDBLOCK))
     {
       return TRUE;
     }

     if(cindex != Num_Contacts)
     {
       Contacts[cindex].sok = 0;
       Contacts[cindex].connected = 0;
     }

     close(sock);
     return TRUE;
   }

   recv(sock, packet, packet_size + 2, 0);

   memcpy(&uin, (packet + 11), 4);
 
   for(cindex = 0; cindex < Num_Contacts; cindex++)
   {
      if(Contacts[cindex].uin == uin)
      {
         Contacts[cindex].sok = sock;
         Contacts[cindex].connected = 1;
         break;
      }
   }

   if(cindex == Num_Contacts)
   {
      Contacts[Num_Contacts].uin = uin;
      Contacts[Num_Contacts].status = STATUS_NOT_IN_LIST;
      Contacts[Num_Contacts].last_time = -1L;
      Contacts[Num_Contacts].current_ip = -1L;
      Contacts[Num_Contacts].port = 0;
      Contacts[Num_Contacts].sok = sock;
      Contacts[Num_Contacts].connected = 1;
      sprintf(Contacts[Num_Contacts].nick, "%ld", uin);
      Num_Contacts++;
   }

   if(packet_size < 1024)
      TCP_ProcessPacket(packet + 2, packet_size, sock);

   g_free(packet);
   return TRUE;
}

gint TCP_Connect(guint32 ip, guint32 port)
{
   struct sockaddr_in local, remote;
   int sizeofSockaddr = sizeof( struct sockaddr );
   int rc, sock;

#ifdef DEBUG
   fprintf(stderr, "\nTCP> TCP_Connect(%d)", sock);
#endif

   if(ip == 0) return -1;

   memset(&local, 0, sizeof(local));   
   local.sin_family = AF_INET;
   local.sin_port = htons(0);
   local.sin_addr.s_addr = htonl(INADDR_ANY);

   memset(&remote, 0, sizeof(remote));
   remote.sin_family = AF_INET;
   remote.sin_port = htons(port);
   remote.sin_addr.s_addr = htonl(ip);

   sock = socket(AF_INET, SOCK_STREAM, 0);
   if(sock == -1) return -1;

   set_nonblock(sock);

   if((bind(sock, (struct sockaddr*)&local, sizeof(struct sockaddr))) == -1)
      return -1;

   getsockname(sock, (struct sockaddr*)&local, &sizeofSockaddr);

   rc = connect(sock, (struct sockaddr*)&remote, sizeof(remote));

   if(rc >= 0)
      if (Verbose & ICQ_VERB_INFO)
        fprintf(stderr, "TCP_Connect(): connect() completed immediately\n");
   else if(errno == EINPROGRESS)
      if (Verbose & ICQ_VERB_INFO)
        fprintf(stderr, "TCP_Connect(): connect() in progress...\n");
   else
      tcp_error_message(errno);
   
   return sock;
}

gint TCP_SendMessage(guint32 uin, char* msg)
{
   int cx;
   int sock;
   unsigned short intsize;
   char buffer[1024];

   typedef struct
   {
      guint8 uin_a[4];  // UIN
      guint8 version[2];  // VERSION
      guint8 cmd[2];  // TCP_START
      guint8 zero[2];  // 0x0000
      guint8 uin_b[4];  // UIN
      guint8 command[2];  // TCP_MSG
      guint8 msg_length[2];
   } tcp_head;

   typedef struct
   {
      guint8 ip[4];
      guint8 real_ip[4];
      guint8 port[4];
      guint8 four;
      guint8 zero[4];
      guint8 seq[4];
   } tcp_tail;

   struct
   {
      tcp_head head;
      char *body;
      tcp_tail tail;
   } packet;

#ifdef DEBUG
   fprintf(stderr, "\nTCP> TCP_SendMessage(%04X, %s)\n", uin, msg);
#endif
   
   DW_2_Chars(packet.head.uin_a, UIN);
   Word_2_Chars(packet.head.version, 0x0003);  /* ICQ_VER ??? */
   Word_2_Chars(packet.head.cmd, ICQ_CMDxTCP_START);
   Word_2_Chars(packet.head.zero, 0x0000);
   DW_2_Chars(packet.head.uin_b, UIN);
   Word_2_Chars(packet.head.command, ICQ_CMDxTCP_MSG);
   Word_2_Chars(packet.head.msg_length, (strlen(msg) + 1));

   packet.body = msg;  /* Mike 1999/02/13: Memory issues ... hmmm ... */

   DW_2_Chars(packet.tail.ip, our_ip);
   DW_2_Chars(packet.tail.real_ip, our_ip);
   DW_2_Chars(packet.tail.port, our_port);
   packet.tail.four = 0x04;
   DW_2_Chars(packet.tail.zero, 0x00100000);
   DW_2_Chars(packet.tail.seq, seq_num++);

   for(cx = 0; cx < Num_Contacts; cx++)
      if(Contacts[cx].uin == uin) break;
   
   if(cx == Num_Contacts) return 0;

   sock = Contacts[cx].sok;

   if(sock != -1)
   {
#ifdef DEBUG
      fprintf(stderr, "\nTCP Connection established.");
#endif

      intsize = sizeof(tcp_head) + sizeof(tcp_tail) + strlen(msg) + 1;

      memcpy(&buffer[0], &intsize, 2);
      memcpy(&buffer[2], &packet.head, sizeof(packet.head));
      memcpy(&buffer[2 + sizeof(packet.head)], packet.body, strlen(packet.body) + 1);
      memcpy(&buffer[2 + sizeof(packet.head) + strlen(packet.body) + 1], &packet.tail, sizeof(packet.tail));
      write(sock, buffer, intsize + 2);
      packet_print(buffer, intsize + 2);
      return 1;
   }

#ifdef DEBUG
  fprintf(stderr, "\nTCP Connection failed.\n");
#endif

   return 0;
}

gint TCP_SendURL(guint32 uin, char* url, char *text)
{
   int cx;
   int sock;
   unsigned short intsize;
   char buffer[1024], msg[1024];

   typedef struct
   {
      guint8 uin_a[4];
      guint8 version[2];
      guint8 cmd[2];
      guint8 zero[2];
      guint8 uin_b[4];
      guint8 command[2];
      guint8 msg_length[2];
   } tcp_head;

   typedef struct
   {
      guint8 ip[4];
      guint8 real_ip[4];
      guint8 port[4];
      guint8 four;
      guint8 zero[4];
      guint8 seq[4];
   } tcp_tail;

   struct
   {
      tcp_head head;
      char *body;
      tcp_tail tail;
   } packet;

#ifdef DEBUG
   fprintf(stderr, "\nTCP> TCP_SendURL(%04X, %s, %s)", uin, url, text);
#endif

   if(!url) url = "";
   if(!text) text = "";
   
   strcpy(msg, text);
   strcat(msg, "\xFE");
   strcat(msg, url);

   DW_2_Chars(packet.head.uin_a, UIN);
   Word_2_Chars(packet.head.version, 0x0003);
   Word_2_Chars(packet.head.cmd, ICQ_CMDxTCP_START);
   Word_2_Chars(packet.head.zero, 0x0000);
   DW_2_Chars(packet.head.uin_b, UIN);
   Word_2_Chars(packet.head.command, ICQ_CMDxTCP_URL);
   Word_2_Chars(packet.head.msg_length, (strlen(msg) + 1));

   packet.body = msg;

   DW_2_Chars(packet.tail.ip, our_ip);
   DW_2_Chars(packet.tail.real_ip, our_ip);
   DW_2_Chars(packet.tail.port, our_port);
   packet.tail.four = 0x04;
   DW_2_Chars(packet.tail.zero, 0x00100000);
   DW_2_Chars(packet.tail.seq, seq_num++);

   for(cx = 0; cx < Num_Contacts; cx++)
      if(Contacts[cx].uin == uin) break;

   if(cx == Num_Contacts) return 0;

   sock = Contacts[cx].sok;

   if(sock != -1)
   {
      intsize = sizeof(tcp_head) + sizeof(tcp_tail) + strlen(msg) + 1;

      memcpy(&buffer[0], &intsize, 2);
      memcpy(&buffer[2], &packet.head, sizeof(packet.head));
      memcpy(&buffer[2 + sizeof(packet.head)], packet.body, strlen(packet.body) + 1);
      memcpy(&buffer[2 + sizeof(packet.head) + strlen(packet.body) + 1], &packet.tail, sizeof(packet.tail));
      write(sock, buffer, intsize + 2);
      packet_print(buffer, intsize + 2);
      return 1;
   }

   return 0;
}

gint TCP_GetAwayMessage(guint32 uin)
{
   int cx;
   int sock;
   unsigned short intsize;
   char buffer[1024];
   int status_type;

   typedef struct
   {
      guint8 uin_a[4];
      guint8 version[2];
      guint8 cmd[2];
      guint8 zero[2];
      guint8 uin_b[4];
      guint8 command[2];
      guint8 msg_length[2];
   } tcp_head;

   typedef struct
   {
      guint8 ip[4];
      guint8 real_ip[4];
      guint8 port[4];
      guint8 four;
      guint8 zero[4];
      guint8 seq[4];
   } tcp_tail;

   struct
   {
      tcp_head head;
      char *body;
      tcp_tail tail;
   } packet;
   
#ifdef DEBUG
   fprintf(stderr, "\nTCP> TCP_GetAwayMessage(%04X)", uin);
#endif

   for(cx = 0; cx < Num_Contacts; cx++)
      if(Contacts[ cx ].uin == uin) break;

   if(cx == Num_Contacts) return 0;


   switch(Contacts[cx].status & 0xffff)
   {
      case STATUS_AWAY:
         status_type = ICQ_CMDxTCP_READxAWAYxMSG;
         break;
      case STATUS_NA:
         status_type = ICQ_CMDxTCP_READxNAxMSG;
         break;
      case STATUS_OCCUPIED:
         status_type = ICQ_CMDxTCP_READxOCCxMSG;
         break;
      case STATUS_DND:
         status_type = ICQ_CMDxTCP_READxDNDxMSG;
         break;
      default:
         status_type = ICQ_CMDxTCP_READxAWAYxMSG;
         break;
   }

   DW_2_Chars( packet.head.uin_a, UIN );
   Word_2_Chars( packet.head.version, 0x0003 );
   Word_2_Chars( packet.head.cmd, ICQ_CMDxTCP_START );
   Word_2_Chars( packet.head.zero, 0x0000 );
   DW_2_Chars( packet.head.uin_b, UIN );
   Word_2_Chars( packet.head.command, status_type );
   Word_2_Chars( packet.head.msg_length, 0x0001 );

   packet.body = "";

   DW_2_Chars( packet.tail.ip, our_ip );
   DW_2_Chars( packet.tail.real_ip, LOCALHOST );
   DW_2_Chars( packet.tail.port, our_port );
   packet.tail.four = 0x04;
   DW_2_Chars( packet.tail.zero, 0x00001000 );
   DW_2_Chars( packet.tail.seq, seq_num ++ );

   sock = Contacts[cx].sok;

   if( sock != -1 )
   {
      intsize = sizeof( tcp_head ) + sizeof( tcp_tail ) + 1;

      memcpy( &buffer[0], &intsize, 2 );
      memcpy( &buffer[2], &packet.head, sizeof( packet.head ) );
      memcpy( &buffer[2 + sizeof( packet.head )], packet.body, strlen( packet.body ) + 1 );
      memcpy( &buffer[2 + sizeof( packet.head ) + strlen( packet.body ) + 1 ], &packet.tail, sizeof( packet.tail ) );
      write( sock, buffer, intsize + 2 );
      packet_print( buffer, intsize + 2);
   }
   else
   {
      return 0;
   }

   return 1;
}

/*
int TCPSendChatRequest( guint32 uin, char *msg, struct sokandlb *data )
{
   int cx;
   int sock;
   unsigned short intsize;
   char buffer[1024];

   typedef struct
   {
      guint8 uin_a[4];
      guint8 version[2];
      guint8 cmd[2];
      guint8 zero[2];
      guint8 uin_b[4];
      guint8 command[2];
      guint8 msg_length[2];
   } tcp_head;

   typedef struct
   {
      guint8 ip[4];
      guint8 real_ip[4];
      guint8 port[4];
      guint8 trail1[4];
      guint8 trail2[4];
      guint8 trail3[4];
      guint8 trail4[4];
      guint8 seq[4];
   } tcp_tail;

   struct
   {
      tcp_head head;
      char *body;
      tcp_tail tail;
   } packet;

#ifdef DEBUG
   fprintf(stderr, "\nTCP> TCP_SendChatRequest(%04X)", uin);
#endif

   DW_2_Chars( packet.head.uin_a, UIN );
   Word_2_Chars( packet.head.version, 0x0003 );
   Word_2_Chars( packet.head.cmd, ICQ_CMDxTCP_START );
   Word_2_Chars( packet.head.zero, 0x0000 );
   DW_2_Chars( packet.head.uin_b, UIN );
   Word_2_Chars( packet.head.command, ICQ_CMDxTCP_CHAT );
   Word_2_Chars( packet.head.msg_length, ( strlen( msg ) + 1 ) );

   packet.body = msg;

   DW_2_Chars( packet.tail.ip, our_ip );
   DW_2_Chars( packet.tail.real_ip, LOCALHOST );
   DW_2_Chars( packet.tail.port, our_port );
   DW_2_Chars( packet.tail.trail1, 0x10000004 );
   DW_2_Chars( packet.tail.trail2, 0x00000100 );
   DW_2_Chars( packet.tail.trail3, 0x00000000 );
   DW_2_Chars( packet.tail.trail4, 0x00000000 );
   DW_2_Chars( packet.tail.seq, seq_num ++ );

   for( cx = 0; cx < Num_Contacts; cx ++ )
   {
      if( Contacts[ cx ].uin == uin )
         break;
   }
   if( cx == Num_Contacts )
      return 0;

   sock = TCP_Connect( Contacts[ cx ].current_ip, Contacts[ cx ].port, cx, data );

   if( sock != -1 )
   {
      intsize = sizeof( tcp_head ) + sizeof( tcp_tail ) + strlen( msg ) + 1;

      memcpy( &buffer[0], &intsize, 2 );
      memcpy( &buffer[2], &packet.head, sizeof( packet.head ) );
      memcpy( &buffer[2 + sizeof( packet.head )], packet.body,
              strlen( packet.body ) + 1 );
      memcpy( &buffer[2 + sizeof( packet.head ) + strlen( packet.body ) + 1 ],
              &packet.tail, sizeof( packet.tail ) );
      write( sock, buffer, intsize + 2 );
      packet_print( buffer, intsize + 2,
                    PACKET_TYPE_TCP | PACKET_DIRECTION_SEND );
   }
   else
   {
      return -1;
   }

   return 1;
}
*/
