/* $Id: input.c,v 1.3 1998/12/08 21:27:29 marcus Exp $
******************************************************************************

   Xwin inputlib - use an existing X window as a LibGII input source

   Copyright (C) 1998 Marcus Sundberg [marcus@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <stdlib.h>
#include <X11/Xlib.h>

#include <ggi/internal/gii-dl.h>
#include <ggi/input/xwin.h>
#include "../x/xev.h"

#define RELPTR_KEYS { GIIK_Ctrl, GIIK_Alt, 'm' }
#define RELPTR_KEYINUSE (1 | (1<<1) | (1<<2))
#define RELPTR_NUMKEYS  3

typedef struct {
	Display *disp;
	Window   win;
	Cursor   cursor;
	int      width, height;
	int      oldx, oldy;
	int	 alwaysrel;
	int	 relptr;
	uint32	 relptr_keymask;
} xwin_priv;

#define XWIN_PRIV(inp)	((xwin_priv*)((inp)->priv))


/* These functions are only called from one place, and broken out of the
   code for readability. We inline them so we don't loose any performance- */

static inline Cursor make_cursor(Display *disp, Window win)
{
	char emptybm[] = {0};
	Pixmap crsrpix;
	XColor nocol;
	Cursor mycrsr;

	crsrpix = XCreateBitmapFromData(disp, win, emptybm, 1, 1);
	mycrsr = XCreatePixmapCursor(disp, crsrpix, crsrpix,
				     &nocol, &nocol, 0, 0);
	XFreePixmap(disp, crsrpix);

	return mycrsr;
}

static inline void handle_relptr(xwin_priv *priv)
{
	if (priv->relptr) {
		XUngrabPointer(priv->disp, CurrentTime);
		XUndefineCursor(priv->disp, priv->win);
		priv->relptr = 0;
		DPRINT_EVENTS("GII_xwin_eventpoll: Using absolute pointerevents\n");
	} else {
		XGrabPointer(priv->disp, priv->win, 1, 0,
			     GrabModeAsync, GrabModeAsync,
			     priv->win, priv->cursor, CurrentTime);
		XWarpPointer(priv->disp, None, priv->win, 0, 0, 0, 0,
			     priv->width/2, priv->height/2);
		priv->relptr = 1;
		priv->oldx = priv->width/2;
		priv->oldy = priv->height/2;
		DPRINT_EVENTS("GII_xwin_eventpoll: Using relative pointerevents\n");
	}
}

static inline void center_pointer(xwin_priv *priv)
{
	XEvent event;
	event.type = MotionNotify;
	event.xmotion.display = priv->disp;
	event.xmotion.window = priv->win;
	event.xmotion.x = priv->width/2;
	event.xmotion.y = priv->height/2;
	XSendEvent(priv->disp, priv->win, False, PointerMotionMask, &event);
	XWarpPointer(priv->disp, None, priv->win, 0, 0, 0, 0,
		     priv->width/2, priv->height/2);
}

static inline void update_winparam(xwin_priv *priv)
{
	if (!priv->alwaysrel) {
		Window dummywin;
		unsigned int w, h;
		int dummy;

		priv->cursor = make_cursor(priv->disp, priv->win);
		XGetGeometry(priv->disp, priv->win, &dummywin, &dummy, &dummy,
			     &w, &h, &dummy, &dummy);
		priv->width = w;
		priv->height = h;
		priv->oldx = w/2;
		priv->oldy = h/2;
	}
}

static gii_event_mask GII_xwin_eventpoll(gii_input *inp)
{
	xwin_priv  *priv = XWIN_PRIV(inp);
	int n, i;
	int rc = 0;
	gii_event releasecache;
	Time      releasetime = 0;
	int       havecached = 0;
	
	DPRINT_EVENTS("GII_xwin_eventpoll(%p) called\n", inp);
	n = XEventsQueued(priv->disp, QueuedAfterReading);

	while (n) {
		XEvent xev;
		gii_event giiev;
		
		n--;

		XNextEvent(priv->disp, &xev);
		_giiEventBlank(&giiev);

		switch(xev.type) {
		case KeyPress:
			giiev.any.size = sizeof(gii_key_event);
			giiev.any.type = evKeyPress;
			giiev.key.button = xev.xkey.keycode;

			if (havecached) {
				if (xev.xkey.time == releasetime) {
					giiev.any.type = evKeyRepeat;
					rc |= emKeyRepeat;
				} else {
					_giiEvQueueAdd(inp, &releasecache);
					rc |= emKeyRelease;
					rc |= emKeyPress;
				}
				havecached = 0;
			} else {
				rc |= emKeyPress;
			}
			giiev.key.sym = _gii_xev_ev2sym((XKeyEvent*)&xev);
			giiev.key.label = _gii_xev_ev2label((XKeyEvent*)&xev);
			giiev.key.effect =
				_gii_xev_state2effect(xev.xkey.state);

			if (!priv->alwaysrel) {
				uint32 relsyms[RELPTR_NUMKEYS] = RELPTR_KEYS;
				for (i=0; i < RELPTR_NUMKEYS; i++) {
					if (giiev.key.sym == relsyms[i]) {
						priv->relptr_keymask &= ~(1<<i);
						break;
					}
				}
				if (priv->relptr_keymask == 0) {
					handle_relptr(priv);
				}
			}
			DPRINT_EVENTS("GII_xwin_eventpoll: KeyPress\n");
			break;

		case KeyRelease:
			if (havecached) {
				_giiEvQueueAdd(inp, &releasecache);
				rc |= emKeyRelease;
			}
			_giiEventBlank(&releasecache);
			releasecache.any.origin = inp->origin;
			releasecache.any.type = evKeyRelease;
			releasecache.any.size = sizeof(gii_key_event);
			releasecache.key.sym
				= _gii_xev_ev2sym((XKeyEvent*)&xev);
			releasecache.key.label
				= _gii_xev_ev2label((XKeyEvent*)&xev);
			releasecache.key.button = xev.xkey.keycode;
			releasecache.key.effect =
				_gii_xev_state2effect(xev.xkey.state);

			havecached = 1;
			releasetime = xev.xkey.time;

			if (!priv->alwaysrel) {
				uint32 relsyms[RELPTR_NUMKEYS] = RELPTR_KEYS;
				for (i=0; i < RELPTR_NUMKEYS; i++) {
					if (releasecache.key.sym
					    == relsyms[i]) {
						priv->relptr_keymask |= (1<<i);
						break;
					}
				}
			}
			DPRINT_EVENTS("GII_x_eventpoll: KeyRelease\n");
			break;

		case ButtonPress:
			giiev.any.type = evPtrButtonPress;
			rc |= emPtrButtonPress;
			DPRINT_EVENTS("GII_xwin_eventpoll: ButtonPress %x\n",
				      xev.xbutton.state); 
			break;

		case ButtonRelease:
			giiev.any.type = evPtrButtonRelease;
			rc |= emPtrButtonRelease;
			DPRINT_EVENTS("GII_xwin_eventpoll: ButtonRelease %x\n",
				      xev.xbutton.state); 
			break;

		case MotionNotify:
			if (priv->relptr) {
				int realmove = 0;
				if (!xev.xmotion.send_event) {
					giiev.pmove.x =
						xev.xmotion.x - priv->oldx;
					giiev.pmove.y =
						xev.xmotion.y - priv->oldy;
					giiev.any.type = evPtrRelative;
					rc |= emPtrRelative;
					realmove = 1;
#undef ABS
#define ABS(a) (((a)<0) ? -(a) : (a) )
					if (ABS(priv->width/2 - xev.xmotion.x)
					    > 3 * priv->width / 8 ||
					    ABS(priv->height/2 - xev.xmotion.y)
					    > 3 * priv->height / 8) {
						center_pointer(priv);
					}
#undef ABS			
				}
				priv->oldx = xev.xmotion.x;
				priv->oldy = xev.xmotion.y;
				if (!realmove)
					goto dont_queue_this_event;
			} else {
				if (priv->alwaysrel) {
					giiev.any.type = evPtrRelative;
					rc |= emPtrRelative;
				} else {
					giiev.any.type = evPtrAbsolute;
					rc |= emPtrAbsolute;
				}
				giiev.pmove.x = xev.xmotion.x;
				giiev.pmove.y = xev.xmotion.y;
			}
			giiev.any.size = sizeof(gii_pmove_event);
			DPRINT_EVENTS("GII_xwin_eventpoll: MouseMove %d,%d\n",
				      giiev.pmove.x, giiev.pmove.y);
			break;

		case EnterNotify:
		case LeaveNotify:
			break;

		case Expose:
			giiev.any.size = sizeof(gii_expose_event);
			giiev.any.type = evExpose;
			giiev.expose.x = xev.xexpose.x;
			giiev.expose.y = xev.xexpose.y;
			giiev.expose.w = xev.xexpose.width;
			giiev.expose.h = xev.xexpose.height;
			break;

		default: 
			DPRINT_EVENTS("GII_xwin_eventpoll: Other Event (%d)\n",
				      xev.type);
			break;
			
		}
		switch(giiev.any.type) {
		case evPtrButtonPress:
		case evPtrButtonRelease:
			giiev.any.size = sizeof(gii_pbutton_event);
			giiev.pbutton.button =
				_gii_xev_buttontrans(xev.xbutton.button);
			break;
		}
		giiev.any.origin = inp->origin;
		if (giiev.any.type) _giiEvQueueAdd(inp, &giiev);
		dont_queue_this_event:
		/* "ANSI C forbids label at end of compound statement"
		   Well, this makes GCC happy at least... */
		while(0){};
	}
	if (havecached) {
		_giiEvQueueAdd(inp, &releasecache);
		rc |= emKeyRelease;
	}

	return rc;
}


static int GII_xwin_sendevent(gii_input *inp, gii_event *ev)
{
	xwin_priv  *priv = XWIN_PRIV(inp);	
	
	if (ev->any.type != evCommand) {
		return GGI_EEVUNKNOWN;
	}

	if (ev->cmd.code == GII_CMDCODE_XWINSETPARAM) {
		gii_xwin_cmddata_setparam *data
			= (gii_xwin_cmddata_setparam *)	ev->cmd.data;
		priv->win = data->win;
		priv->alwaysrel = data->ptralwaysrel;
		
		update_winparam(priv);

		return 0;
	}

	return GGI_EEVUNKNOWN;
}


static int GII_xwin_close(gii_input *inp)
{
	xwin_priv  *priv = XWIN_PRIV(inp);
	
	free(priv);
	
	DPRINT_MISC("GII_xwin_close(%p) called\n", inp);

	return 0;
}


int GIIdlinit(gii_input *inp, const char *args, void *argptr)
{
	gii_inputxwin_arg *xwinarg = argptr;
	xwin_priv  *priv;

	DPRINT_MISC("GIIdlinit(%p) called for input-xwin\n", inp);

	if (xwinarg == NULL || xwinarg->disp == NULL) {
		return GGI_EARGREQ;
	}
	
	if ((priv = malloc(sizeof(xwin_priv))) == NULL) {
		return GGI_ENOMEM;
	}
	
	priv->disp = xwinarg->disp;
	priv->win = xwinarg->win;

	priv->alwaysrel = xwinarg->ptralwaysrel;
	priv->relptr = 0;
	priv->relptr_keymask = RELPTR_KEYINUSE;
	
	if (!xwinarg->wait) {
		update_winparam(priv);
	}
	
	inp->priv = priv;
	
	inp->GIIeventpoll = GII_xwin_eventpoll;
	inp->GIIsendevent = GII_xwin_sendevent;
	inp->GIIclose = GII_xwin_close;

	inp->targetcan = emKey | emPointer | emExpose;
	inp->curreventmask = emKey | emPointer | emExpose;

	inp->maxfd = ConnectionNumber(priv->disp) + 1;
	FD_SET(ConnectionNumber(priv->disp), &inp->fdset);

	return 0;
}
