/* $Id: input.c,v 1.12 1998/11/28 22:46:24 ajapted Exp $
******************************************************************************

   Linux_joy: input

   Copyright (C) 1998 Andrew Apted     [andrew@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <termios.h>

#include <sys/time.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <linux/joystick.h>

#include <ggi/internal/gii.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif


#define MAX_NR_JAXES		8
#define MAX_NR_JBUTTONS		32 


typedef struct joystick_hook
{
	int fd;

	unsigned char num_axes;
	unsigned char num_buttons;

	int axes[MAX_NR_JAXES];
	char buttons[MAX_NR_JBUTTONS];

} JoystickHook;

#define JOYSTICK_HOOK(inp)	((JoystickHook *) inp->priv)


/* ---------------------------------------------------------------------- */


static int GII_joystick_init(gii_input *inp, char *filename)
{
	JoystickHook *jj;
	int version;
	int fd;
	char name[128];

	/* open the device file */

	fd = open(filename, O_RDONLY);

	if (fd < 0) {
		perror("Linux_joy: Couldn't open joystick device");
		return GGI_ENODEVICE;
	}

	/* get version and get name */
	
	if (ioctl(fd, JSIOCGVERSION, &version) < 0) {
		perror("Linux_joy: Couldn't read version:");
		version=0;
	}

	DPRINT("Linux_joy: Joystick driver version %d.%d.%d\n",
		(version >> 16) & 0xff, (version >> 8) & 0xff,
		version & 0xff);

	if (version < 0x010000) {
		fprintf(stderr, "Linux_joy: Sorry, only driver versions "
			">= 1.0.0 supported.\n");
		close(fd);
		return GGI_ENODEVICE;
	}

	if (ioctl(fd, JSIOCGNAME(sizeof(name)), name) < 0) {
		strcpy(name, "Unknown");
	}

	DPRINT("Linux_joy: Joystick driver name `%s'.\n", name);

	/* allocate joystick hook */

	if ((jj = malloc(sizeof(JoystickHook))) == NULL) {
		close(fd);
		return GGI_ENOMEM;
	}
	
	
	/* get number of axes and buttons */

	if (ioctl(fd, JSIOCGAXES, &jj->num_axes) ||
	    ioctl(fd, JSIOCGBUTTONS, &jj->num_buttons)) {
		perror("Linux_joy: error getting axes/buttons");
		close(fd);
		free(jj);
		return GGI_ENODEVICE;
	}

	DPRINT("Linux_joy: Joystick has %d axes.\n", jj->num_axes);
	DPRINT("Linux_joy: Joystick has %d buttons.\n", jj->num_buttons);

	if (jj->num_axes > MAX_NR_JAXES) {
		jj->num_axes = MAX_NR_JAXES;
	}
	if (jj->num_buttons > MAX_NR_JBUTTONS) {
		jj->num_buttons = MAX_NR_JBUTTONS;
	}
	
	jj->fd = fd;
	inp->priv = jj;

	DPRINT("Linux_joy: init OK.\n");

	return 0;
}

static void GII_joystick_exit(gii_input *inp)
{
	JoystickHook *jj = JOYSTICK_HOOK(inp);

	close(jj->fd);
	jj->fd = -1;

	free(jj);
	inp->priv = NULL;

	DPRINT("Linux_joy: exit OK.\n");
}

static gii_event_mask GII_joystick_handle_data(gii_input *inp)
{
	JoystickHook *jj = JOYSTICK_HOOK(inp);

	struct js_event js;

	gii_event ev;
	gii_event_mask result=0;

	int i;


	/* read the joystick packet */

	if (read(jj->fd, &js, sizeof(js)) != sizeof(js)) {
		perror("Linux_joy: Error reading joystick");
		return 0;
	}

	switch (js.type & ~JS_EVENT_INIT) {

	  case JS_EVENT_AXIS:

		if (js.number > jj->num_axes)
			break;
		if (jj->axes[js.number] == js.value)
			break;

		jj->axes[js.number] = js.value;

		DPRINT_EVENTS("JOY-AXIS[%d] -> %d.\n", js.number,
				js.value);

		_giiEventBlank(&ev);

		ev.any.type   = evValAbsolute;
		ev.any.size   = sizeof(gii_val_event);
		ev.any.origin = inp->origin;
		ev.any.target = 0;
		ev.val.first  = 0;
		ev.val.count  = jj->num_axes;

		for (i=0; i < ev.val.count; i++) {
			ev.val.value[i] = jj->axes[i];
		}

		_giiEvQueueAdd(inp, &ev);

		result |= emValAbsolute;
		break;
	
	  case JS_EVENT_BUTTON:

		if (js.number > jj->num_buttons)
			break;
		if (jj->buttons[js.number] == js.value)
			break;

		jj->buttons[js.number] = js.value;

		DPRINT_EVENTS("JOY-BUTTON[%d] -> %d.\n", js.number,
				js.value);

		_giiEventBlank(&ev);

		ev.any.type   = js.value ? evKeyPress : evKeyRelease;
		ev.any.size   = sizeof(gii_key_event);
		ev.any.origin = inp->origin;
		ev.any.target = 0;
		ev.key.effect = 0;
		ev.key.button = 1 + js.number;
		ev.key.sym    = GIIK_VOID;
		ev.key.label  = GIIK_VOID;

		_giiEvQueueAdd(inp, &ev);

		result |= (1 << ev.any.type);
		break;

	  default:
		DPRINT_EVENTS("JOY: unknown event from driver "
				"(0x%02x)\n", js.type);
		break;
	}
	
	return result;
}

static gii_event_mask GII_joystick_poll(gii_input *inp)
{
	JoystickHook *jj = JOYSTICK_HOOK(inp);
	
	gii_event_mask result = 0;


	DPRINT_MISC("linux_joy: poll(%p)\n", inp);
	
	for (;;) {

		fd_set readset;

		struct timeval t = {0,0};
		int rc;

		FD_ZERO(&readset);
		FD_SET(jj->fd, &readset);

		/* FIXME !!! doesn't handle -EINTR */
		rc = select(inp->maxfd, &readset, NULL, NULL, &t);

		if (rc <= 0)
			break;

		result |= GII_joystick_handle_data(inp);
	}

	return result;
}


/* ---------------------------------------------------------------------- */


static gii_cmddata_getdevinfo linux_joy_devinfo =
{
	"Linux Joystick",		/* long device name */
	"ljoy",				/* shorthand */
	emKey |	emValuator,		/* can_generate */
	0, 0				/* filled in later */
};

static gii_cmddata_getvalinfo linux_joy_valinfo[3] =
{
    {	0,				/* valuator number */
    	"Side to side",			/* long valuator name */
	"x",				/* shorthand */
	{ -32767, 0, +32767 },		/* range */
	GII_PT_LENGTH,			/* phystype */
	0, 1, 1, -8			/* SI constants (approx.) */
    },
    {	1,				/* valuator number */
    	"Forward and back",		/* long valuator name */
	"y",				/* shorthand */
	{ -32767, 0, +32767 },		/* range */
	GII_PT_LENGTH,			/* phystype */
	0, 1, 1, -8			/* SI constants (approx.) */
    },

    /* This last one is used for all extra axes */

    {	2,				/* changed later */
    	"Axis ",			/* appended to later */
	"n",				/* appended to later */
	{ -32767, 0, +32767 },		/* range */
	GII_PT_LENGTH,			/* phystype */
	0, 1, 1, -8			/* SI constants (approx.) */
    },
};


static int GII_linux_joy_senddevinfo(gii_input *inp)
{
	JoystickHook *jj = JOYSTICK_HOOK(inp);
	
	gii_cmddata_getdevinfo *DI;
	
	gii_event ev;


	_giiEventBlank(&ev);
	
	ev.any.size   = sizeof(gii_cmd_nodata_event) +
		        sizeof(gii_cmddata_getdevinfo);
	ev.any.type   = evCommand;
	ev.any.origin = inp->origin;
	ev.cmd.code   = GII_CMDCODE_GETDEVINFO;

	DI = (gii_cmddata_getdevinfo *) &ev.cmd.data;

	*DI = linux_joy_devinfo;
	
	DI->num_axes    = jj->num_axes;
	DI->num_buttons = jj->num_buttons;

	return _giiEvQueueAdd(inp, &ev);
}

static int GII_linux_joy_sendvalinfo(gii_input *inp, int val)
{
	JoystickHook *jj = JOYSTICK_HOOK(inp);
	
	gii_cmddata_getvalinfo *VI;

	gii_event ev;


	if (val >= jj->num_axes) return -1;
	
	_giiEventBlank(&ev);
	
	ev.any.size   = sizeof(gii_cmd_nodata_event) +
		         sizeof(gii_cmddata_getvalinfo);
	ev.any.type   = evCommand;
	ev.any.origin = inp->origin;
	ev.cmd.code   = GII_CMDCODE_GETVALINFO;


	VI = (gii_cmddata_getvalinfo *) &ev.cmd.data;

	*VI = linux_joy_valinfo[(val < 2) ? val : 2];
	
	if (val >= 2) {
		char buf[20];
	
		sprintf(buf, "%d", val);

		VI->number = val;
		strcat(VI->longname,  buf);
		strcat(VI->shortname, buf);
	}

	return _giiEvQueueAdd(inp, &ev);
}

static int GII_linux_joy_sendevent(gii_input *inp, gii_event *ev)
{
	JoystickHook *jj = JOYSTICK_HOOK(inp);
	
	if ((ev->any.target != inp->origin) &&
	    (ev->any.target != GII_EV_TARGET_ALL)) {
		/* not for us */
		return -1;
	}

	if (ev->any.type != evCommand) {
		return -1;
	}

	if (ev->cmd.code == GII_CMDCODE_GETDEVINFO) {
		return GII_linux_joy_senddevinfo(inp);
	}

	if (ev->cmd.code == GII_CMDCODE_GETVALINFO) {
	
		int i;
		gii_cmddata_getvalinfo *vi;
		
		vi = (gii_cmddata_getvalinfo *) ev->cmd.data;
		
		if (vi->number == GII_VAL_QUERY_ALL) {
			for (i=0; i < jj->num_axes; i++) {
				GII_linux_joy_sendvalinfo(inp, i);
			}
			return 0;
		}

		return GII_linux_joy_sendvalinfo(inp, vi->number);
	}

	return -1;  /* Unknown command */
}


/* ---------------------------------------------------------------------- */


static int GII_linux_joy_close(gii_input *inp)
{
	DPRINT_MISC("Linux_joy close\n");

	if (JOYSTICK_HOOK(inp)) {
		GII_joystick_exit(inp);
	}

	return 0;
}

int GIIdlinit(gii_input *inp, const char *args, void *argptr)
{
	int ret;
	char *filename = "/dev/js0";

	DPRINT_MISC("linux_joy starting.(args=\"%s\",argptr=%p)\n",args,argptr);

	/* Initialize */

	if (args && *args) {
		filename = (char *) args;
	}

	if ((ret = GII_joystick_init(inp, filename)) < 0) {
		return ret;
	}
	
	/* We leave these on the default handlers
	 *	inp->GIIseteventmask = _GIIstdseteventmask;
	 *	inp->GIIgeteventmask = _GIIstdgeteventmask;
	 *	inp->GIIgetselectfdset = _GIIstdgetselectfd;
	 */
	inp->GIIeventpoll = GII_joystick_poll;
	inp->GIIclose = GII_linux_joy_close;
	inp->GIIsendevent = GII_linux_joy_sendevent;

	inp->targetcan = emKey | emValuator;
	inp->GIIseteventmask(inp, emKey | emValuator);

	inp->maxfd = JOYSTICK_HOOK(inp)->fd + 1;
	FD_SET(JOYSTICK_HOOK(inp)->fd, &inp->fdset);

	/* Send initial cmdDevInfo event */
	GII_linux_joy_senddevinfo(inp);
	
	DPRINT_MISC("linux_joy fully up\n");

	return 0;
}
