/* $Id: filter.c,v 1.6 1998/12/15 03:04:46 marcus Exp $
******************************************************************************

   Filter-mouse - generic mouse event generator.

   Copyright (C) 1998 Andreas Beck      [becka@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <stdlib.h>
#include <string.h>

#include <ggi/internal/gii-dl.h>

enum maptype { 
	MAP_NONE, 
	MAP_KEY_TO_BUTTON,
	MAP_KEY_TO_REL,
	MAP_KEY_TO_ABS,
};

typedef struct mapping_entry {

	struct mapping_entry *next;

	uint32	device;

	enum maptype type;
	uint32	number;		/* key/axis */
	uint32	modifier_mask;	/* all modifiers in mask */
	uint32	modifier_value;	/* must match value */
	uint32	to;		/* key/axis */
	sint32	speed;		/* ticks to go */
} mapping_entry;

static mapping_entry test[]= {
	{
		NULL,
		0,
		MAP_KEY_TO_BUTTON,
		
		GIIK_Enter,
		(1<<GII_KM_ALT),
		(1<<GII_KM_ALT),

		GII_PBUTTON_LEFT,
		0
	},
	{
		NULL,
		0,
		MAP_KEY_TO_REL,

		GIIK_Left,
		(1<<GII_KM_ALT),
		(1<<GII_KM_ALT),

		0,
		-1
	},
	{
		NULL,
		0,
		MAP_KEY_TO_REL,

		GIIK_Right,
		(1<<GII_KM_ALT),
		(1<<GII_KM_ALT),

		0,
		1
	},
	{
		NULL,
		0,
		MAP_KEY_TO_REL,

		GIIK_Up,
		(1<<GII_KM_ALT),
		(1<<GII_KM_ALT),

		1,
		-1
	},
	{
		NULL,
		0,
		MAP_KEY_TO_REL,

		GIIK_Down,
		(1<<GII_KM_ALT),
		(1<<GII_KM_ALT),

		1,
		1
	}
};

typedef struct fmouse_state {
	struct gii_input *inp;
	mapping_entry *entry;
} fmouse_state;

static void fmouse_send_pbutton(gii_input *inp, uint8 type, uint32 nr)
{
	gii_event ev;

	_giiEventBlank(&ev);
	ev.pbutton.type = type;
	ev.pbutton.size = sizeof(gii_pbutton_event);
	ev.pbutton.origin = inp->origin;
	ev.pbutton.target = GII_EV_TARGET_ALL;
	ev.pbutton.button = nr;
	_giiEvQueueAdd(inp, &ev);
}

static void fmouse_send_pmove(gii_input *inp, uint8 type, uint32 nr, sint32 speed)
{
	gii_event ev;

	_giiEventBlank(&ev);
	ev.pmove.type = type;
	ev.pmove.size = sizeof(gii_pmove_event);
	ev.pmove.origin = inp->origin;
	ev.pmove.target = GII_EV_TARGET_ALL;
	ev.pmove.x = ev.pmove.y = ev.pmove.z = ev.pmove.wheel =0;
	switch(nr)
	{ case 0:
	  default:
		ev.pmove.x = speed;break;
	  case 1:
		ev.pmove.y = speed;break;
	  case 2:
		ev.pmove.z = speed;break;
	  case 3:
		ev.pmove.wheel = speed;break;
	}

	_giiEvQueueAdd(inp, &ev);
}

static void fmouse_check_trans(gii_input *inp,gii_event *event)
{
	fmouse_state   *state=inp->priv;
	mapping_entry *entry;
	
	entry=state->entry;
	while(entry)
	{
		/* Simple generic check. Is the source o.k. ? */
		if (entry->device!=0 && entry->device!=event->any.origin) 
			continue;

		switch(entry->type)
		{
			case MAP_KEY_TO_BUTTON:
				if ( ( event->any.type==evKeyPress ||
				       event->any.type==evKeyRelease ) &&
				     event->key.sym==entry->number &&
				     ( event->key.effect & entry->modifier_mask ) 
				       == entry->modifier_value ) {
					fmouse_send_pbutton(inp, event->any.type==evKeyRelease ?
								evPtrButtonRelease : evPtrButtonPress,entry->to);
					DPRINT_MISC("Have Key-to-button.\n");
				}
				break;
			case MAP_KEY_TO_REL:
				if ( ( event->any.type==evKeyPress ||
				       event->any.type==evKeyRepeat ) &&
				     event->key.sym==entry->number &&
				     ( event->key.effect & entry->modifier_mask ) 
				       == entry->modifier_value ) {
					fmouse_send_pmove(inp, evPtrRelative, entry->to, entry->speed);
					DPRINT_MISC("Have Key-to-Rel Press.\n");
				}
				break;
			case MAP_KEY_TO_ABS:
				if ( ( event->any.type==evKeyPress ||
				       event->any.type==evKeyRepeat ) &&
				     event->key.sym==entry->number &&
				     ( event->key.effect & entry->modifier_mask ) 
				       == entry->modifier_value ) {
					fmouse_send_pmove(inp, evPtrAbsolute, entry->to, entry->speed);
					DPRINT_MISC("Have Key-to-Abs Press.\n");
				}
				break;
			default:
				DPRINT_MISC("filter-mouse: illegal mapping.\n");
				break;
		}
		entry=entry->next;
	}
}

static gii_event_mask GII_fmouse_poll(struct gii_input *inp)
{
	gii_event	ev;
	gii_event_mask	rc,rc2;
	struct timeval	time={0,0};

	fmouse_state   *state=inp->priv;

	DPRINT_MISC("filter-mouse: poll(%p);\n",inp);
	
	rc2=0;
	/* As long as the input we were derived from has something ... */
	while(giiEventPoll(state->inp,emAll,&time))
	{
		/* Get it. */
		giiEventRead(state->inp,&ev,emAll);

		/* Check if it triggers something. */
		fmouse_check_trans(inp,&ev);
		
		/* Check if we want it. */
		if ( !( inp->curreventmask & (rc=(1<<ev.any.type)) ) )
			continue;		/* No. Drop it. */

		rc2|=rc;			/* Remember what it was. */
		_giiEvQueueAdd(inp,&ev);	/* Post it */
	}

	return rc2;
	return 0;
}

static gii_cmddata_getdevinfo stdin_devinfo={
	"Mouse event generator",	/* long device name */
	"fmou",				/* shorthand */
	emPointer,			/* can_generate */
	0,				/* no buttons */
	0				/* no valuators */
};

static int GII_fmouse_close(gii_input *inp)
{
	fmouse_state   *state=inp->priv;

	giiClose(state->inp);	/* Get rid of old inputlib */

	free(state);
	DPRINT_MISC("filter-mouse: close %p\n",inp);
	return 0;
}

static int GII_fmouse_seteventmask(struct gii_input *inp, gii_event_mask evm)
{
	fmouse_state   *state=inp->priv;

	inp->curreventmask=( evm & inp->targetcan );

	/* Pass it on ! */
	giiSetEventMask(state->inp,evm);

	return GGI_OK;
}

int GIIdlinit(gii_input *inp, const char *args, void *argptr)
{
	gii_event diev;
	fmouse_state   *state;

	DPRINT_MISC("filter-mouse starting.(args=\"%s\",input=%p)\n",args,argptr);

	if (NULL==argptr)
	{
		DPRINT_MISC("filter-mouse: filter libs require an argptr.\n");
		return -1;
	}

        if (NULL==(state=inp->priv=malloc(sizeof(fmouse_state))))
	{
		DPRINT_MISC("filter-mouse: no memory.\n");
		return -1;
	}

	state->inp=argptr;
	test[0].next=&test[1];
	test[1].next=&test[2];
	test[2].next=&test[3];
	test[3].next=&test[4];
	state->entry=&test[0];	/* TEST */
	
	/* We leave these on the default handlers
	 *	inp->GIIgeteventmask=_GIIstdgeteventmask;
	 *	inp->GIIgetselectfdset=_GIIstdgetselectfd;
	 */

	/* They are already set, so we can as well use them instead of
	 * accessing the curreventmask member directly.
	 */
	inp->targetcan=emAll;
	inp->GIIseteventmask=GII_fmouse_seteventmask;
	inp->GIIeventpoll=GII_fmouse_poll;
	inp->GIIclose	 =GII_fmouse_close;

	inp->GIIseteventmask(inp,giiGetEventMask(state->inp)|emPointer);

	inp->maxfd=0;		/* We got to poll ... We might have polling children ... */
	/* Hmm - we should probably copy the fdset ... 
	   and add a "dummy" polled source to make sure ... FIXME */

	_giiEventBlank(&diev);
	diev.cmd.size=sizeof(gii_cmd_nodata_event)+sizeof(gii_cmddata_getdevinfo);
	diev.cmd.type=evCommand;
	diev.cmd.origin=inp->origin;
	diev.cmd.code=GII_CMDCODE_GETDEVINFO;
	memcpy(diev.cmd.data,&stdin_devinfo,sizeof(gii_cmddata_getdevinfo));
	_giiEvQueueAdd(inp,&diev);

	DPRINT_MISC("filter-mouse fully up\n");

	return 0;
}
