/*
 * demo.c - (c) 1998 Andreas Beck   becka@ggi-project.org
 *
 * This is a demonstration of LibGII's functions and can be used as a 
 * reference programming example.
 *
 *   This software is placed in the public domain and can be used freely
 *   for any purpose. It comes without any kind of warranty, either
 *   expressed or implied, including, but not limited to the implied
 *   warranties of merchantability or fitness for a particular purpose.
 *   Use it at your own risk. the author is not responsible for any damage
 *   or consequences raised by use or inability to use this program.
 */

/* Include the necessary headers used for e.g. error-reporting.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* Include the LibGII declarations.
 */
#include <ggi/gii.h>

void test_showevent(gii_event *ev)
{
	int y;
	
	printf("o:%8x ",ev->any.origin);
	switch(ev->any.type)
	{
		case evKeyPress:
			printf("KeyPress    ");goto showkey;
		case evKeyRepeat:
			printf("KeyRepeat   ");goto showkey;
		case evKeyRelease:
			printf("KeyRelease  ");
			showkey:
			printf("sym=%4x label=%4x button=%4x\n",
				ev->key.sym,ev->key.label,ev->key.button);
			break;
		case evValAbsolute:
			printf("ValAbsolute ");goto showval;
		case evValRelative:
			printf("ValRelative ");
			showval:
			for(y=0;y<ev->val.count;y++)
				printf("[%2d]=%d",ev->val.first+y,ev->val.value[y]);
			printf("\n");
			break;
		case evPtrAbsolute:
			printf("PtrAbsolute ");goto showptr;
		case evPtrRelative:
			printf("PtrRelative ");
			showptr:
			printf("x=%4x y=%4x z=%4x w=%4x\n",
				ev->pmove.x,ev->pmove.y,ev->pmove.z,ev->pmove.wheel);
			break;
		default:
			printf("event: type=%d\n",ev->any.type);
			for(y=0;y<ev->any.size;y++)
				printf("%02x  ",((unsigned char *)ev)[y]);
			printf("\n");
	}
}

void test_gii_poll(gii_input_t inp, struct timeval *tval)
{
	gii_event_mask	mask;
	gii_event	event;

	mask=giiEventPoll(inp,emAll,tval);
	printf("0x%8x=giiEventPoll(%p,emAll,&{%d,%d});\n", mask, inp,
	       (int)tval->tv_sec, (int)tval->tv_usec);
	if (mask) 
	{
		printf("0x%8x ",mask);
		giiEventRead(inp,&event,emAll);
		test_showevent(&event);
	}
	while(mask)
	{
		struct timeval	tv= {0,0};
		mask=giiEventPoll(inp,emAll,&tv);
		if (mask) 
		{
			printf("0x%8x ",mask);
			giiEventRead(inp,&event,emAll);
			test_showevent(&event);
		}
	}
}

/* The main routine.
 * It will just open a simple gii_input and test some filters.
 */
int main(int argc, char **argv)
{
	/* First we define a bunch of variables we will access throughout the
	 * main() function. Most of them are pretty meaningless loop-counters
	 * and helper-variables.
	 */
	int x;
	struct timeval tval;

 	gii_input_t inp;
	
	/* Initialize the GII library. This must be called before any other 
	 * GII function. 
	 */
	if (giiInit() != 0) {
		fprintf(stderr, "%s: unable to initialize libgii, exiting.\n",
			argv[0]);
		exit(1);
	}

	/* Open some input.
	 */
	if ((inp=giiOpen("input-stdin",NULL)) == NULL) {
		fprintf(stderr,
			"%s: unable to open stdin-source, exiting.\n",
			argv[0]);
		giiExit();
		exit(1);
	}
	printf("Doing test of input-stdin.\n");
	for(x=0;x<5;x++)
	{
		tval.tv_sec = 0;
		tval.tv_usec = 100000*x;
		test_gii_poll(inp,&tval);
	}
	printf("Now adding save filter.\n");
	if ((inp=giiOpen("filter-save:test",inp)) == NULL) {
		fprintf(stderr,
			"%s: unable to open filter-save, exiting.\n",
			argv[0]);
		giiExit();
		exit(1);
	}
	for(x=0;x<10;x++)
	{
		tval.tv_sec = 0;
		tval.tv_usec = 100000*x;
		test_gii_poll(inp,&tval);
	}
	printf("Now adding mouse filter.\n");
	if ((inp=giiOpen("filter-mouse",inp)) == NULL) {
		fprintf(stderr,
			"%s: unable to open filter-mouse, exiting.\n",
			argv[0]);
		giiExit();
		exit(1);
	}
	for(x=0;x<10;x++)
	{
		tval.tv_sec = 0;
		tval.tv_usec = 100000*x;
		test_gii_poll(inp,&tval);
	}
	giiClose(inp);

	/* Now close down LibGII. */
	giiExit();	

	/* Terminate the program.*/
	return 0;
}
