/* $Id: mode.c,v 1.1 2000/12/17 23:34:36 marcus Exp $
******************************************************************************

   FreeBSD vgl(3) target: mode management

   Copyright (C) 2000 Alcove - Nicolas Souchu <nsouch@freebsd.org>

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <sys/ioccom.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <ggi/internal/ggi-dl.h>
#include <ggi/display/vgl.h>

#include "../common/pixfmt-setup.inc"

void _GGI_vgl_freedbs(ggi_visual *);

int GGI_vgl_flush(ggi_visual *vis, int x, int y, int w, int h, int tryflag)
{
	return 0;
}
	
int GGI_vgl_getapi(ggi_visual *vis, int num, char *apiname, char *arguments)
{
	switch(num) {
		case 0:
			strcpy(apiname, "display-vgl");
			strcpy(arguments, "");
			return 0;
		case 1:
			strcpy(apiname, "generic-stubs");
			strcpy(arguments, "");
			return 0;
		case 2:
			strcpy(apiname, "generic-color");
			strcpy(arguments, "");
			return 0;
		default:
			break;
	}
			
	return -1;
}

int GGI_vgl_setmode(ggi_visual *vis, ggi_mode *tm)
{ 
	struct vgl_priv *priv = VGL_PRIV(vis);
	struct video_info modeinfo;
	unsigned long modenum = 0;
	char sugname[256];
	char args[256];
	int err = 0;
	int id;

	err = GGI_vgl_checkmode(vis, tm);
	if (err) return err;

	/* reset the modeinfo structure as expected by query_mode */
	memset(&modeinfo, 0, sizeof(modeinfo));
	
	switch(tm->graphtype) {
		case GT_1BIT : modeinfo.vi_depth = 1; break;
		case GT_4BIT : modeinfo.vi_depth = 4; break;
		case GT_8BIT : modeinfo.vi_depth = 8; break;
		case GT_15BIT: modeinfo.vi_depth = 15; break;
		case GT_16BIT: modeinfo.vi_depth = 16; break;
		case GT_24BIT: modeinfo.vi_depth = 24; break;
		case GT_32BIT: modeinfo.vi_depth = 32; break;
		default: return -1;
	}

	modeinfo.vi_width = tm->visible.x;
	modeinfo.vi_height = tm->visible.y;

	/* XXX should be added to libvgl */
	if (ioctl(0, FBIO_FINDMODE, &modeinfo))
		return -1;

	GGIDPRINT("Setting VGLlib mode %d (0x%x)\n",
			modeinfo.vi_mode, modeinfo.vi_mode);

	/* Terminate any current mode before initialising another */
	if (priv->vgl_init_done)
		VGLEnd();

	/* XXX should be in VGL */
	if ((modeinfo.vi_mode >= M_B40x25) && (modeinfo.vi_mode <= M_VGA_M90x60))
		modenum = _IO('S', modeinfo.vi_mode);
	if ((modeinfo.vi_mode >= M_TEXT_80x25) && (modeinfo.vi_mode <= M_TEXT_132x60))
		modenum = _IO('S', modeinfo.vi_mode);
	if ((modeinfo.vi_mode >= M_VESA_CG640x400) &&
		(modeinfo.vi_mode <= M_VESA_FULL_1280))
		modenum = _IO('V', modeinfo.vi_mode - M_VESA_BASE);

	if ((err = VGLInit(modenum)) != 0) {
		GGIDPRINT("display-vgl: setting mode 0x%x failed with error %d\n",
			modeinfo.vi_mode, err);
		return GGI_EFATAL;
	}

	priv->vgl_init_done = 1;

	/* Palette */
	if (vis->palette) {
		free(vis->palette);
		vis->palette = NULL;
	}
	if (priv->savepalette) {
		free(priv->savepalette);
		priv->savepalette = NULL;
	}
	if (GT_SCHEME(tm->graphtype) == GT_PALETTE) {
		int len = 1 << GT_DEPTH(tm->graphtype);

		vis->palette = malloc(len * sizeof(ggi_color));
		if (vis->palette == NULL) return GGI_EFATAL;
		priv->savepalette = malloc(sizeof(int) * (len*3));
		if (priv->savepalette == NULL) return GGI_EFATAL;

		/* Set an initial palette */
		ggiSetColorfulPalette(vis);
	}

	/* Pixel format */
	memset(LIBGGI_PIXFMT(vis), 0, sizeof(ggi_pixelformat));
	setup_pixfmt(LIBGGI_PIXFMT(vis), tm->graphtype);
	_ggi_build_pixfmt(LIBGGI_PIXFMT(vis));

	memcpy(LIBGGI_MODE(vis),tm,sizeof(ggi_mode));

	_ggiZapMode(vis, 0);

	for(id = 1; 0 == GGI_vgl_getapi(vis, id, sugname, args); id++) {
		if (_ggiOpenDL(vis, sugname, args, NULL)) {
			fprintf(stderr,"display-vgl: Can't open the %s (%s) library.\n",
				sugname, args);
			return GGI_EFATAL;
		} else {
			GGIDPRINT("Success in loading %s (%s)\n", sugname, args);
		}
	}

	vis->opdraw->putpixel		= GGI_vgl_putpixel;
	vis->opdraw->putpixel_nc	= GGI_vgl_putpixel_nc;
	vis->opdraw->getpixel		= GGI_vgl_getpixel;
	vis->opdraw->drawpixel		= GGI_vgl_drawpixel;
	vis->opdraw->drawpixel_nc	= GGI_vgl_drawpixel_nc;
	vis->opdraw->drawhline		= GGI_vgl_drawhline;
	vis->opdraw->drawhline_nc	= GGI_vgl_drawhline_nc;
	vis->opdraw->drawvline		= GGI_vgl_drawvline;
	vis->opdraw->drawvline_nc	= GGI_vgl_drawvline_nc;
	vis->opdraw->drawbox		= GGI_vgl_drawbox;
	vis->opdraw->drawline		= GGI_vgl_drawline;
	vis->opdraw->puthline		= GGI_vgl_puthline;
	vis->opdraw->putbox		= GGI_vgl_putbox;

	if (GT_SCHEME(tm->graphtype) == GT_PALETTE) {
		vis->opcolor->setpalvec = GGI_vgl_setpalvec;
	}

	ggiIndicateChange(vis, GGI_CHG_APILIST);

	return 0;
}

#define WANT_MODELIST
#include "../common/modelist.inc"

/**********************************/
/* check any mode (text/graphics) */
/**********************************/
int GGI_vgl_checkmode(ggi_visual *vis,ggi_mode *tm)
{
	int ret, err = 0;

	if (vis==NULL || tm==NULL)
		return -1;
	
	if (tm->visible.x == GGI_AUTO)
		tm->visible.x = tm->virt.x;
	if (tm->visible.y == GGI_AUTO)
		tm->visible.y = tm->virt.y;
	
	if (tm->graphtype == GGI_AUTO) {
		err=_GGIcheckautobpp(vis, tm, VGL_PRIV(vis)->availmodes);
	} else if ((ret =
		    _GGIcheckonebpp(vis, tm, VGL_PRIV(vis)->availmodes))
		   != 0) {
		err = -1;
		if (ret == 1)
			_GGIgethighbpp(vis, tm, VGL_PRIV(vis)->availmodes);
	}
	
	if(tm->virt.x==GGI_AUTO) tm->virt.x = tm->visible.x;
	if(tm->virt.y==GGI_AUTO) tm->virt.y = tm->visible.y;

	/* Force virtual to equal visible */
	if(tm->virt.x != tm->visible.x) {
		tm->virt.x = tm->visible.x;
		err = -1;
	}
	if (tm->virt.y != tm->visible.y) {
		tm->virt.y = tm->visible.y;
		err = -1;
	}
 
	/* Multiple frames are not implemented yet... */
	if (tm->frames != 1 && tm->frames != GGI_AUTO) {
		err = -1;
	}
	tm->frames = 1;

	if ((tm->dpp.x != 1 && tm->dpp.x != GGI_AUTO) ||
	    (tm->dpp.y != 1 && tm->dpp.y != GGI_AUTO)) {
		err = -1;
	}
	tm->dpp.x = tm->dpp.y = 1;

	if (tm->size.x != GGI_AUTO || tm->size.y != GGI_AUTO) {
		err = -1;
	}
	tm->size.x = tm->size.y = GGI_AUTO;

	return err;
}

/************************/
/* get the current mode */
/************************/
int GGI_vgl_getmode(ggi_visual *vis,ggi_mode *tm)
{
	GGIDPRINT("In GGIgetmode(%p,%p)\n",vis,tm);
	if (vis==NULL)
		return -1;

	memcpy(tm,LIBGGI_MODE(vis),sizeof(ggi_mode));
	return 0;
}

/*************************/
/* set the current flags */
/*************************/
int GGI_vgl_setflags(ggi_visual *vis,ggi_flags flags)
{
	LIBGGI_FLAGS(vis)=flags;
	return 0;
}
