/* $Id: visual.c,v 1.31 1998/12/18 21:36:24 marcus Exp $
******************************************************************************

   XF86DGA display target.

   Copyright (C) 1997-1998 Steve Cheng	[steve@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <ggi/internal/ggi-dl.h>
#include <ggi/display/xf86dga.h>

#include <stdlib.h>
#include <unistd.h>

#ifdef HAVE_SIGNAL_H
#include <signal.h>
#endif

void _GGI_xf86dga_freedbs(ggi_visual *vis)
{
	int i;
	int first = LIBGGI_APPLIST(vis)->first_targetbuf;
	int last = LIBGGI_APPLIST(vis)->last_targetbuf;
	
	if (first < 0) {
		return;
	}
	for (i = (last - first); i >= 0; i--) {
		_ggi_db_free(LIBGGI_APPBUFS(vis)[i+first]);
		_ggi_db_del_buffer(LIBGGI_APPLIST(vis), i+first);
	}
	LIBGGI_APPLIST(vis)->first_targetbuf = -1;
}


static int _GGI_xf86dga_getbpp(struct Xhooks *priv)
{
	XImage *bppcheckimage;
	unsigned int bits_per_pixel = 0;
	
	if ((bppcheckimage = XGetImage(priv->display,
				       RootWindow(priv->display, priv->screen),
				       0, 0, 1, 1, AllPlanes, ZPixmap))
	    != NULL) {
		bits_per_pixel = bppcheckimage->bits_per_pixel;
		XDestroyImage(bppcheckimage);
	}
	return bits_per_pixel;
}

/* DGA-installed signal handlers.  I don't know it does with them, but they 
   don't handle all the error conditions. Thus we also need to fork
   to prevent dead apps from locking up the console. */
static int dga_signals[] = {
	SIGSEGV, SIGBUS, SIGHUP, SIGFPE};
static struct sigaction old_signals[sizeof(dga_signals)];

int GGIdlinit(ggi_visual *vis,const char *args,void *argptr)
{
	struct Xhooks *priv;

	/* Just dummy variables */
	int x, y;
	unsigned z;
	Window root;

	if (geteuid()) {
		fprintf(stderr, "Need root privs to do XF86DGA.\n");
	        return GGI_DL_ERROR;
	}

	GIVE_UP_ROOT;

	priv = _ggi_malloc(sizeof(struct Xhooks));

	if ((priv->XLibLock = ggLockCreate()) == NULL) {
		free(priv);
		return GGI_DL_ERROR;
	}

	DPRINT("XF86DGA starting.\n");

	LIBGGI_GC(vis) = _ggi_malloc(sizeof(ggi_gc));
	
	priv->cmap=0;
	priv->cmap2=0;
	priv->nocols=0;

	DPRINT("XF86DGA wants display %s.\n",args);
	priv->display = XOpenDisplay(args);
	if (priv->display == NULL)
		return GGI_DL_ERROR;

	DPRINT("XF86DGA has display %s.\n",args);

	priv->screen = DefaultScreen(priv->display);

	XF86DGAQueryVersion(priv->display, &x, &y);
	DPRINT("XF86DGA version %d.%d\n", x, y);
	if(x < 1) {
		fprintf(stderr, "Your XF86DGA is too old (%d.%d).\n", x, y);
		return GGI_DL_ERROR;
	}

	XF86VidModeQueryVersion(priv->display, &x, &y);
	DPRINT("XF86VidMode version %d.%d\n", x, y);
	
	XF86DGAQueryDirectVideo(priv->display, priv->screen, &x);
	if (!(x && XF86DGADirectPresent)) {
		fprintf(stderr, "No direct video capability available!\n");
		return GGI_DL_ERROR;
	}

	GET_ROOT;
	XF86DGAGetVideo(priv->display, priv->screen, (char**) &priv->fb,
		&priv->stride, &priv->bank_size, &priv->mem_size);
	PERM_GIVE_UP_ROOT;

	DPRINT("fb: %p, stride: %d, bank_size: %d, mem_size: %d\n",
	       priv->fb, priv->stride, priv->bank_size, priv->mem_size);

	if (priv->bank_size != priv->mem_size*1024) {
		fprintf(stderr,"DGA-target: Sorry, banked framebuffer layout not supported.\n");
		return GGI_DL_ERROR;
	}

	/* Get virtual dimensions */
	XGetGeometry(priv->display,
		RootWindow(priv->display, priv->screen),
		&root, &x, &y,
		&priv->width, &priv->height,
		(unsigned int*)&z,
		&priv->depth);
	
	priv->size = _GGI_xf86dga_getbpp(priv);

	DPRINT("Virtwidth: %d, depth: %d, size: %d\n", 
		priv->width, priv->depth, priv->size);

	/* Get XF86VidMode modelines */
	XF86VidModeGetAllModeLines(priv->display, priv->screen,
		&priv->num_modes, &priv->dgamodes);

	priv->modes = _ggi_malloc((priv->num_modes+1)*sizeof(ggi_modelistmode));
	
	for(x = 0; x<priv->num_modes; x++) {
		priv->modes[x].x = priv->dgamodes[x]->hdisplay;
		priv->modes[x].y = priv->dgamodes[x]->vdisplay;
		priv->modes[x].bpp = priv->depth;
		priv->modes[x].gt = GT_CONSTRUCT(priv->depth,
			(priv->depth <= 8) ? GT_PALETTE : GT_TRUECOLOR,
			priv->size);
		DPRINT("Found mode: %dx%d\n",
			priv->modes[x].x,
			priv->modes[x].y);
	}
	priv->modes[priv->num_modes].bpp=0;

	XF86DGAForkApp(priv->screen);

	/* Has mode management */
	vis->opdisplay->getmode=GGI_xf86dga_getmode;
	vis->opdisplay->setmode=GGI_xf86dga_setmode;
	vis->opdisplay->checkmode=GGI_xf86dga_checkmode;
	vis->opdisplay->getapi=GGI_xf86dga_getapi;
	vis->opdisplay->setflags=GGI_xf86dga_setflags;

	{
		gii_inputxwin_arg args;
		gii_input *inp;
		
		args.disp = priv->display;
		args.ptralwaysrel = 1;
		args.wait = 1;
		
		if ((inp = giiOpen("xwin", &args, NULL)) == NULL) {
			DPRINT_MISC("Unable to open xwin inputlib\n");
			GGIdlcleanup(vis);
			return GGI_DL_ERROR;
		}

		priv->inp = inp;
		/* Now join the new event source in. */
		vis->input = giiJoinInputs(vis->input, inp);
        }

	for(x = 0; x < sizeof(dga_signals); x++)
		sigaction(dga_signals[x], NULL, old_signals + x);

	LIBGGI_PRIVATE(vis) = priv;

	return GGI_DL_OPDISPLAY;
}

int GGIdlcleanup(ggi_visual *vis)
{
	struct Xhooks *priv;
	int i;

	_GGI_xf86dga_freedbs(vis);

	if ((priv=LIBGGI_PRIVATE(vis)) != NULL)
	{
		XF86DGADirectVideo(priv->display, priv->screen, 0);

		XUngrabPointer(priv->display, CurrentTime);
		XUngrabKeyboard(priv->display, CurrentTime);

		/* Get rid of DGA's signals */
		for(i = 0; i<sizeof(dga_signals); i++)
			sigaction(dga_signals[i], old_signals+i, NULL);

#ifndef ALWAYS_SEGFAULT_ON_EXIT
		/* prevent-segfault hack */
		if(!_ggiDebugState)
			signal(SIGSEGV, _exit);
#endif

		if (priv->cmap)
			XFreeColormap(priv->display,priv->cmap);
		if (priv->cmap2)
			XFreeColormap(priv->display,priv->cmap2);

		if (priv->dgamodes) {
			XF86VidModeSwitchToMode(priv->display, priv->screen,
				priv->dgamodes[0]);

			/* Free the modelines */
			for (i = 0; i < priv->num_modes; i++)
				if (priv->dgamodes[i]->privsize > 0)
					XFree(priv->dgamodes[i]->private);

			XFree(priv->dgamodes);
		}
		if (priv->modes)
			free(priv->modes);
		
		if (priv->display)
			XCloseDisplay(priv->display);

		ggLockDestroy(priv->XLibLock);

		free(LIBGGI_PRIVATE(vis));
	}

	if(vis->palette)
		free(vis->palette);
	free(LIBGGI_GC(vis));

	return 0;
}

#include <ggi/internal/ggidlinit.h>
