/* $Id: mode.c,v 1.34 1998/12/24 00:59:52 marcus Exp $
******************************************************************************

   Mode management for XF86DGA

   Copyright (C) 1997,1998 Steve Cheng   [steve@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <ggi/internal/ggi-dl.h>
#include <ggi/display/xf86dga.h>

static int GGI_xf86dga_setdisplayframe(ggi_visual *vis, int num)
{
	struct Xhooks *priv=LIBGGI_PRIVATE(vis);
	ggi_directbuffer *db = _ggi_db_find_frame(vis, num);

	if (db == NULL) {
		return -1;
	}

	vis->d_frame_num = num;

	XF86DGASetViewPort(priv->display, priv->screen,
			   vis->origin_x, LIBGGI_MODE(vis)->virt.y 
			   * vis->d_frame_num + vis->origin_y);

	return 0;
}


static int GGI_xf86dga_setorigin(ggi_visual *vis,int x,int y)
{
	struct Xhooks *priv=LIBGGI_PRIVATE(vis);
	ggi_mode *mode=LIBGGI_MODE(vis);

	if (x<0 || x > mode->virt.x - mode->visible.x ||
	    y<0 || y > mode->virt.y - mode->visible.y)
	     return -1;

	XF86DGASetViewPort(priv->display, priv->screen,
			   x, mode->virt.y*vis->d_frame_num + y);

	vis->origin_x=x;
	vis->origin_y=y;
	
	return 0;
}

/*
** Finds the appropriate XF86VidMode modeline for x/y.
*/
static int _GGI_xf86dga_findmode(ggi_visual *vis, int visible_x, int visible_y)
{
	struct Xhooks *priv=LIBGGI_PRIVATE(vis);
	int i;

	/* Find a suitable XF86VidMode and return it */
	for(i = 0; i<priv->num_modes; i++)
		if(	visible_x == priv->modes[i].x &&
			visible_y == priv->modes[i].y)
			{ return i; }

	/* mode not found */
	return -1;
}


int GGI_xf86dga_getapi(ggi_visual *vis,int num, char *apiname ,char *arguments)
{
	switch(num) {
		case 0: strcpy(apiname, "display-dga");
			strcpy(arguments, "");
			return 0;
		case 1: strcpy(apiname, "generic-stubs");
			strcpy(arguments, "");
			return 0;
		case 2: strcpy(apiname, "generic-color");
			strcpy(arguments, "");
			return 0;
		case 3:
			sprintf(apiname, "generic-linear-%d", GT_SIZE(LIBGGI_MODE(vis)->graphtype));
			strcpy(arguments,"");
			return 0;
	}
	return -1;
}


static int _GGI_xf86dga_maskgetshift(int mask)
{
	int shift;
	for (shift = 16; mask != 0; mask >>= 1) {
		shift--;
	}
	return shift;
}


int GGI_xf86dga_setmode(ggi_visual *vis,ggi_mode *tm)
{ 
	struct Xhooks *priv;
	XVisualInfo    vinfo;
	int flags, i, err, id;
	char sugname[256],args[256];


	/* First, check if the mode can be set */
	if(GGI_xf86dga_checkmode(vis, tm))
		return -1;

	priv=LIBGGI_PRIVATE(vis);

	_GGI_xf86dga_freedbs(vis);

	/* Set XF86VidMode */
	XF86VidModeSwitchToMode(priv->display, priv->screen,
		priv->dgamodes[_GGI_xf86dga_findmode(vis, tm->visible.x,
						     tm->visible.y)]);

/* 
 * Not needed. We already have the information.
 *
 * However, for the paranoid, the fields in priv can be checked 
 * against the information here, and panic if they don't match.
 */
#if 0
	XF86DGAGetViewPortSize(priv->display, priv->screen, 
		(int*) &tm->visible.x, (int*) &tm->visible.y);
	DPRINT("viewport: %dx%d\n", tm->visible.x, tm->visible.y);
#endif	
	
	/* Get all the events */
	XGrabKeyboard(priv->display, DefaultRootWindow(priv->display), 
		True, GrabModeAsync, GrabModeAsync, CurrentTime);

	XGrabPointer(priv->display, DefaultRootWindow(priv->display),
		True, PointerMotionMask | ButtonPressMask | ButtonReleaseMask,
		GrabModeAsync, GrabModeAsync, None, None, CurrentTime);

	if (priv->cmap)
		XFreeColormap(priv->display,priv->cmap);
	if (priv->cmap2)
		XFreeColormap(priv->display,priv->cmap2);

	XMatchVisualInfo(priv->display, priv->screen,
			 GT_DEPTH(tm->graphtype),
			 GT_SCHEME(tm->graphtype) == GT_PALETTE ? PseudoColor
			 : TrueColor, &vinfo);

	if (GT_SCHEME(tm->graphtype) == GT_PALETTE) {
		priv->cmap = XCreateColormap(priv->display,
				DefaultRootWindow(priv->display),
				vinfo.visual, AllocAll);
		priv->cmap2 = XCreateColormap(priv->display,
				DefaultRootWindow(priv->display),
				vinfo.visual, AllocAll);

		DPRINT("%d-bit visual: X-lib colormap allocated %x.\n",
		       GT_DEPTH(tm->graphtype), priv->cmap);
	} else {
		priv->cmap=0;
		priv->cmap2=0;
	}

	DPRINT("Ready, now taking crash course!\n"); 

	flags = XF86DGADirectGraphics | XF86DGADirectMouse;
	XF86DGADirectVideo(priv->display, priv->screen, flags);

	_ggiZapMode(vis,~GGI_DL_OPDISPLAY);

	if (GT_SCHEME(tm->graphtype) == GT_PALETTE) {
		XLIB_PRIV(vis)->nocols
			= 1<< GT_DEPTH(tm->graphtype);
		XLIB_PRIV(vis)->activecmap=0;
		
		if(vis->palette)
			free(vis->palette);
		vis->palette = _ggi_malloc(sizeof(ggi_color) * XLIB_PRIV(vis)->nocols);
	}

	priv->pixperframe = ((GT_SIZE(tm->graphtype)*priv->stride + 7)/8)
		* tm->virt.y;
	vis->d_frame_num = 0;
	
	/* Set up directbuffers */
	for(i = 0; i < tm->frames; i++) {
		LIBGGI_APPLIST(vis)->last_targetbuf = _ggi_db_add_buffer(LIBGGI_APPLIST(vis), _ggi_db_get_new());
		LIBGGI_APPBUFS(vis)[i]->frame = i;
		LIBGGI_APPBUFS(vis)[i]->type
			= GGI_DB_NORMAL | GGI_DB_SIMPLE_PLB;
		LIBGGI_APPBUFS(vis)[i]->read = LIBGGI_APPBUFS(vis)[i]->write
			= (unsigned char*)(priv->fb) + priv->pixperframe * i;
		LIBGGI_APPBUFS(vis)[i]->layout = blPixelLinearBuffer;
		LIBGGI_APPBUFS(vis)[i]->buffer.plb.stride
			= (GT_SIZE(tm->graphtype) * priv->stride + 7)/8;
		LIBGGI_APPBUFS(vis)[i]->buffer.plb.pixelformat
			= LIBGGI_PIXFMT(vis);
		DPRINT_MODE("DB: %d, addr: %p, stride: %d\n", i, 
			    LIBGGI_APPBUFS(vis)[i]->read, 
			    LIBGGI_APPBUFS(vis)[i]->buffer.plb.stride);
	}
	LIBGGI_APPLIST(vis)->first_targetbuf = LIBGGI_APPLIST(vis)->last_targetbuf - (tm->frames-1);
	
	vis->origin_x=0;
	vis->origin_y=0;

	/* Fill in ggi_pixelformat */
	memset(LIBGGI_PIXFMT(vis), 0, sizeof(ggi_pixelformat));
	LIBGGI_PIXFMT(vis)->red_mask = vinfo.red_mask;
	LIBGGI_PIXFMT(vis)->green_mask = vinfo.green_mask;
	LIBGGI_PIXFMT(vis)->blue_mask=vinfo.blue_mask;
	LIBGGI_PIXFMT(vis)->depth = GT_DEPTH(tm->graphtype);
	LIBGGI_PIXFMT(vis)->size = GT_SIZE(tm->graphtype);

	if (GT_SCHEME(tm->graphtype) == GT_PALETTE)
		LIBGGI_PIXFMT(vis)->clut_mask
			= (1 << GT_DEPTH(tm->graphtype)) - 1;

	_ggi_build_pixfmt(LIBGGI_PIXFMT(vis));

	memcpy(LIBGGI_MODE(vis),tm,sizeof(ggi_mode));

	for(id=1;0==GGI_xf86dga_getapi(vis,id,sugname,args);id++) {
		err=(_ggiOpenDL(vis, sugname, args, NULL)==NULL);
		if (err) {
			fprintf(stderr,"display-dga: Can't open the %s (%s) library.\n",
				sugname,args);
			/* In our special case, fail is always fatal. */
			return err;
		} else {
			DPRINT_MODE("Success in loading %s (%s)\n",sugname,args);
		}
	}

	vis->opdraw->setorigin=GGI_xf86dga_setorigin;
	vis->opdraw->setdisplayframe=GGI_xf86dga_setdisplayframe;

	/* color */
	if (GT_SCHEME(tm->graphtype) == GT_PALETTE)
		vis->opcolor->setpalvec=GGI_xf86dga_setpalvec;
#if 0	/* generic-color does this for us */
	vis->opcolor->mapcolor=GGI_xf86dga_mapcolor;
	vis->opcolor->unmappixel=GGI_xf86dga_unmappixel;
#endif

	ggiIndicateChange(vis, GGI_CHG_APILIST);

	/* Tell inputlib about the new window */
	{
		gii_event ev;
		gii_xwin_cmddata_setparam *data
			= (gii_xwin_cmddata_setparam *) &ev.cmd.data;
		ev.cmd.size = sizeof(gii_cmd_event);
		ev.cmd.type = evCommand;
		ev.cmd.target = priv->inp->origin;
		ev.cmd.code = GII_CMDCODE_XWINSETPARAM;
		data->win = DefaultRootWindow(priv->display);
		data->ptralwaysrel = 1;

		giiEventSend(priv->inp, &ev);
	}

	return 0;
}

#define WANT_CHECKONEBPP
#define _GGIcheckonebpp _GGI_xf86dga_checkonebpp
#include "../common/modelist.inc"

/**********************************/
/* check any mode (text/graphics) */
/**********************************/
int GGI_xf86dga_checkmode(ggi_visual *vis,ggi_mode *tm)
{
	struct Xhooks *priv;
	int err = 0;	/* Flag: if the mode being checked is not found */

	if (vis==NULL) {
		DPRINT("Visual==NULL\n");
		return -1;
	}

	priv=LIBGGI_PRIVATE(vis);

	if (tm->visible.x == GGI_AUTO) tm->visible.x = priv->width;
	if (tm->visible.y == GGI_AUTO) tm->visible.y = tm->virt.y;

	if ((tm->dpp.x!=1 && tm->dpp.x!=GGI_AUTO) ||
	    (tm->dpp.y!=1 && tm->dpp.y!=GGI_AUTO)) {
		err = -1;
	}
	tm->dpp.x = 1;
	tm->dpp.y = 1;

	if (GT_DEPTH(tm->graphtype) != priv->depth 
	    || GT_SIZE(tm->graphtype) != priv->size) {
		if (tm->graphtype != GT_AUTO)
			err = -1;
		tm->graphtype = GT_CONSTRUCT(priv->depth,
		      (priv->depth <= 8) ? GT_PALETTE : GT_TRUECOLOR,
		      priv->size);
	}
	
	/* Try to find a suitable XF86DGA mode. */
#if 1
	if (_GGI_xf86dga_checkonebpp(vis, tm, XLIB_PRIV(vis)->modes))
		err = -1;
#else
	if(_GGI_xf86dga_findmode(vis, tm->visible.x, tm->visible.y)==-1) {
		int sug_x, sug_y, new_x, new_y, i;

		/* If that cannot be done, flag it as an error */
		err = -1;

		/* Then try to find another mode with
		   the next higher resolution */

		sug_x = sug_y = 0x7FFFFFFF;	/* Just the maximum */

		for(i = 0; i<priv->num_modes; i++) {
			new_x = priv->modes[i].x;
			new_y = priv->modes[i].y;
			if(	new_x >= tm->visible.x &&
				new_y >= tm->visible.y &&
				new_x <= sug_x &&
				new_y <= sug_y)
			{
				sug_x = new_x;
				sug_y = new_y;
			}
		}
		
		/* If the above failed, then use
		   the highest resolution possible */
		if(	sug_x == 0x7FFFFFFF &&
			sug_y == 0x7FFFFFFF)
		{
			sug_x = sug_y = 0;

			for(i = 0; i<priv->num_modes; i++) {
				new_x = priv->modes[i].x;
				new_y = priv->modes[i].y;
				if(new_x >= sug_x && new_y >= sug_y) {
					sug_x = new_x;
					sug_y = new_y;
				}
			}
		}
		
		tm->visible.x = sug_x;
		tm->visible.y = sug_y;
	}			
#endif
	if (tm->virt.x==GGI_AUTO) tm->virt.x = priv->width;
	if (tm->virt.y==GGI_AUTO) tm->virt.y = tm->visible.y;

	/* Virtual x-resolution is unchangeable */
	if(tm->virt.x != priv->width) {
		tm->virt.x = priv->width;
		err = -1;
	}
	if (tm->virt.y > priv->height) {
		tm->virt.y = priv->height;
		err = -1;
	} else if (tm->virt.y < tm->visible.y) {
		tm->virt.y = tm->visible.y;
		err = -1;
	}	
	
	if (tm->frames * ((GT_SIZE(tm->graphtype)*priv->stride + 7)/8) 
	    * tm->virt.y > priv->mem_size*1024) {
		tm->frames = priv->mem_size*1024 /
			(((GT_SIZE(tm->graphtype)*priv->stride + 7)/8)
			 * tm->virt.y);
		err = -1;
	}
	if (tm->frames < 1) {
		if (tm->frames != GGI_AUTO) {
			err = -1;
		}
		tm->frames = 1;
	}

	return err;	
}

/************************/
/* get the current mode */
/************************/
int GGI_xf86dga_getmode(ggi_visual *vis,ggi_mode *tm)
{
	DPRINT("In GGI_xf86dga_getmode(%p,%p)\n",vis,tm);
	if (vis==NULL)
		return -1;
	
	if (LIBGGI_MODE(vis)==NULL)
		return -1;

	memcpy(tm,LIBGGI_MODE(vis),sizeof(ggi_mode));

	return 0;
}

/*************************/
/* set the current flags */
/*************************/
int GGI_xf86dga_setflags(ggi_visual *vis,ggi_flags flags)
{
	LIBGGI_FLAGS(vis)=flags;
	return 0;
}
