/* $Id: visual.c,v 1.35 1998/12/18 21:36:23 marcus Exp $
******************************************************************************

   Initializing tiles

   Copyright (C) 1998 Steve Cheng    [steve@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <ggi/display/tile.h>

void _GGI_tile_freedbs(ggi_visual *vis) {
	int i;
        
	for (i=LIBGGI_APPLIST(vis)->num-1; i >= 0; i--) {
		free(LIBGGI_APPBUFS(vis)[i]->write);
		_ggi_db_free(LIBGGI_APPBUFS(vis)[i]);
		_ggi_db_del_buffer(LIBGGI_APPLIST(vis), i);
	}
}


#define ARGUMENT_FORMAT  "display-tile:\n\
    The argument format is `offx,offy,sizex,sizey,(subdisplay):...',\n\
    where offx and offy are the tile's offset from the main display,\n\
    sizex and sizey are the size of the tile,\n\
    subdisplay is the display string to open for the tile,\n\
    and ... is more tiles following the same format as above...\n"

int GGIdlinit(ggi_visual *vis,const char *args,void *argptr)
{
	struct TileHooks *tilehook;
	char target[1024];
	int sx, sy, vx, vy, n, i=0;

	if (! args) {
		fprintf(stderr, "display-tile needs the real targets as arguments.\n");
		fprintf(stderr, ARGUMENT_FORMAT);
		return -1;
	}

	LIBGGI_GC(vis) = _ggi_malloc(sizeof(ggi_gc));

	LIBGGI_PRIVATE(vis) = tilehook = _ggi_malloc(sizeof(struct TileHooks));
	tilehook->opmansync = _ggi_malloc(sizeof(_ggi_opmansync));

	tilehook->buf=NULL;
	tilehook->use_db=1;

	/* parse each visual */
	for (;;) {

		sx = sy = vx = vy = 0;

		while (*args && isspace((int)*args)) args++;

		if (! *args) break;

		if (strncmp(args, "-usedb:", 7) == 0) {
			DPRINT_MISC("display-tile: Enabling DB\n");
			tilehook->use_db = 1;
			args += 7; continue;
		}
		if (strncmp(args, "-nodb:", 6) == 0) {
			DPRINT_MISC("display-tile: Disabling DB\n");
			tilehook->use_db = 0;
			args += 6; continue;
		}

		/* Avoid overflowing tables. If someone ever needs more than
		   256 we'll do dynamic allocation instead. */
		if (i==MAX_VISUALS) {
			ggiPanic("display-tile: FIXME: visual limit reached!\n");
		}

		if ((sscanf(args, "%d , %d , %d , %d %n", &sx, &sy,
			    &vx, &vy, &n) != 4) || 
		    (args += n, *args++ != ',')) {

			fprintf(stderr, ARGUMENT_FORMAT);
			free(tilehook);
			return GGI_DL_ERROR;
		}
		
		/* Empty tile! */
		if(vx<=0 || vy<=0 || sx<0 || sy<0) {
			fprintf(stderr, "display-tile: erroneous coordinates for tile #%d!\n", i);
			free(tilehook);
			return GGI_DL_ERROR;
		}

		tilehook->vis_origins[i].x = sx;
		tilehook->vis_origins[i].y = sy;
		tilehook->vis_sizes[i].x = vx;
		tilehook->vis_sizes[i].y = vy;

		args = ggParseTarget((char *)args, target, 1024);

		if (! args) {
			fprintf(stderr,"display-tile: parsetarget error.\n");
			free(tilehook);
			return GGI_DL_ERROR;
		}

		if (*target == 0) {
			strcpy(target, "auto");
		}
		
		DPRINT_MISC("display-tile: visual #%d is %s (%d,%d)[%dx%d]\n",
			i, target, sx, sy, vx, vy);

		if (! (tilehook->vislist[i]=ggiOpen(target,NULL)) ) {
			fprintf(stderr,"display-tile: Opening of target %s failed.\n", target);
			free(tilehook);
			return GGI_DL_ERROR;
		}

		/* Add giiInputs, if we have them. */
		if (tilehook->vislist[i]->input)
		{
			vis->input=giiJoinInputs(vis->input,tilehook->vislist[i]->input);
			tilehook->vislist[i]->input=NULL;	/* Destroy old reference */
		}

		if(tilehook->use_db) {
			/* Don't need SYNC mode, we do it ourselves */
			ggiSetFlags(tilehook->vislist[i], GGIFLAG_ASYNC);
		}

		/* check for ':' separator */
		
		while (*args && isspace((int)*args)) args++;
		
		if (*args && (*args != ':')) {
			fprintf(stderr, "display-tile: expecting ':' between targets.\n");
			free(tilehook);
			return GGI_DL_ERROR;
		}

		if (*args == ':') args++;
		i++;
	}

	tilehook->numvis=i;

	if (tilehook->numvis == 0) {
		fprintf(stderr, "display-tile needs the real targets as arguments.\n");
		free(tilehook);
		return -1;
	}

	if(tilehook->use_db) {
		if(!_ggiAddDL(vis, "helper-mansync", (char*) tilehook->opmansync, NULL, 0)) {
			fprintf(stderr, "display-tile: Cannot load required helper-mansync! (for DB mode)\n");
			return GGI_DL_ERROR;
		}

		MANSYNC_init(vis);
	}

	/* Has mode management */
	vis->opdisplay->getmode=GGI_tile_getmode;
	vis->opdisplay->setmode=GGI_tile_setmode;
	vis->opdisplay->checkmode=GGI_tile_checkmode;
	vis->opdisplay->getapi=GGI_tile_getapi;
	vis->opdisplay->setflags=GGI_tile_setflags;
	
	if(tilehook->use_db)
		vis->opdisplay->flush=GGI_tile_flush_db;
	else
		vis->opdisplay->flush=GGI_tile_flush;

	return GGI_DL_OPDISPLAY;
}

int GGIdlcleanup(ggi_visual *vis)
{
	struct TileHooks *tilehook = LIBGGI_PRIVATE(vis);
	int i;

	if(tilehook->use_db) {
		MANSYNC_deinit(vis);

		_GGI_tile_freedbs(vis);
	}

	if(tilehook->buf)
		free(tilehook->buf);

	for(i = 0; i<tilehook->numvis; i++)
		ggiClose(tilehook->vislist[i]);

	free(tilehook->opmansync);
	free(tilehook);
	free(LIBGGI_GC(vis));

	return 0;
}


