/* $Id: monotext.c,v 1.19 1998/12/18 21:36:23 marcus Exp $
******************************************************************************

   Display-monotext

   Copyright (C) 1998 Andrew Apted    [andrew@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#define MAINTAINER  "Andrew Apted <andrew@ggi-project.org>"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include <ggi/display/monotext.h>
#include "font_data.h"


/**************************************************
 ***
 ***  Various Defines
 ***
 **************************************************/

/**************************************************
 ***
 ***  Internal functions 
 ***
 **************************************************/


static uint8 ascii_template[256*4*4];

static uint8 greyblock_to_ascii[65536];


static inline int count_bits(uint8 *ch, int x, int y, int w, int h)
{
	int cx, cy;
	int result=0;

	for (cy = y; cy < (y+h); cy++)
	for (cx = x; cx < (x+w); cx++) {

		if (ch[cy] & (1 << (7-cx))) {
			result++;
		}
	}

	return result;
}

static void setup_templates(ggi_coord accuracy)
{
	int box_w = 8 / accuracy.x;
	int box_h = 8 / accuracy.y;

	int a, x, y;

	DPRINT("Calculating character templates...\n");

	for (a=32; a <= 126; a++)
	for (y=0; y < accuracy.y; y++)
	for (x=0; x < accuracy.x; x++) {

		ascii_template[(a-32)*16 + y*accuracy.x + x] =
			count_bits(& font_data[(a-32)*8], x*box_w, y*box_h,
				   box_w, box_h) * 255 / (box_w*box_h);
	}
}

static inline int sum_of_squared_diffs(uint8 *ta, uint8 *tb, 
					ggi_coord accuracy)
{
	int n, diff;
	int result = 0;

	for (n=0; n < (accuracy.x * accuracy.y); n++) {

		diff = (int) *ta++ - (int) *tb++;
		
		result += diff * diff;
	}

	return result;
}

static int find_closest_char(uint8 *template, ggi_coord accuracy)
{
	int a, val;

	int min_result = 0x70000000;
	int min_char = 32;

	for (a=32; a <= 126; a++) {

		val = sum_of_squared_diffs(template, 
				& ascii_template[(a-32)*16], accuracy);

		if (val < min_result) {
			min_char = a;
			min_result = val;
		}
	}

	return min_char;
}

static void setup_rgb2grey_table(uint8 *map)
{
	int r, g, b, i;

	DPRINT("Calculating rgb->greyscale table...\n");

	for (r=0; r < 32; r++)
	for (g=0; g < 32; g++)
	for (b=0; b < 32; b++) {

		i = (int) sqrt((double) (r*r*30 + g*g*50 + b*b*20));

		map[(r << 10) | (g << 5) | b] = i * 256 / 311;
	}
}

static void calc_accuracy_1x1(int index, ggi_coord acc)
{
	uint8 template[16];

	template[0] = index;

	greyblock_to_ascii[index] = find_closest_char(template, acc);
}

static void calc_accuracy_1x2(int index, ggi_coord acc)
{
	uint8 template[16];

	template[0] = index >> 8;
	template[1] = index & 0xff;

	greyblock_to_ascii[index] = find_closest_char(template, acc);
}

static void calc_accuracy_2x2(int index, ggi_coord acc)
{
	uint8 template[16];

	template[0] = ((index >> 12) & 0xf) * 255 / 15;
	template[1] = ((index >>  8) & 0xf) * 255 / 15;
	template[2] = ((index >>  4) & 0xf) * 255 / 15;
	template[3] = ((index      ) & 0xf) * 255 / 15;

	greyblock_to_ascii[index] = find_closest_char(template, acc);
}

static void calc_accuracy_2x4(int index, ggi_coord acc)
{
	uint8 template[16];

	template[0] = ((index >> 14) & 0x3) * 255 / 3;
	template[1] = ((index >> 12) & 0x3) * 255 / 3;
	template[2] = ((index >> 10) & 0x3) * 255 / 3;
	template[3] = ((index >>  8) & 0x3) * 255 / 3;

	template[4] = ((index >>  6) & 0x3) * 255 / 3;
	template[5] = ((index >>  4) & 0x3) * 255 / 3;
	template[6] = ((index >>  2) & 0x3) * 255 / 3;
	template[7] = ((index      ) & 0x3) * 255 / 3;

	greyblock_to_ascii[index] = find_closest_char(template, acc);
}

static void calc_accuracy_4x4(int index, ggi_coord acc)
{
	uint8 template[16];

	template[ 0] = (index & 0x8000) ? 255 : 0;
	template[ 1] = (index & 0x4000) ? 255 : 0;
	template[ 2] = (index & 0x2000) ? 255 : 0;
	template[ 3] = (index & 0x1000) ? 255 : 0;

	template[ 4] = (index & 0x0800) ? 255 : 0;
	template[ 5] = (index & 0x0400) ? 255 : 0;
	template[ 6] = (index & 0x0200) ? 255 : 0;
	template[ 7] = (index & 0x0100) ? 255 : 0;

	template[ 8] = (index & 0x0080) ? 255 : 0;
	template[ 9] = (index & 0x0040) ? 255 : 0;
	template[10] = (index & 0x0020) ? 255 : 0;
	template[11] = (index & 0x0010) ? 255 : 0;

	template[12] = (index & 0x0008) ? 255 : 0;
	template[13] = (index & 0x0004) ? 255 : 0;
	template[14] = (index & 0x0002) ? 255 : 0;
	template[15] = (index & 0x0001) ? 255 : 0;

	greyblock_to_ascii[index] = find_closest_char(template, acc);
}

static void blitter_1x1(MonotextHook *mt, void *dest, void *src, int w)
{
	uint8  *s = (uint8 *)  src;
	uint16 *d = (uint16 *) dest;

	for (; w > 0; w--, s++) {
	
		if (greyblock_to_ascii[*s] == 255) {
			calc_accuracy_1x1(*s, mt->accuracy);
		}

		*d++ = 0x0700 | greyblock_to_ascii[*s]; 
	}
}

static void blitter_1x2(MonotextHook *mt, void *dest, void *src, int w)
{
	uint8  *s = (uint8 *)  src;
	uint16 *d = (uint16 *) dest;

	int stride = mt->size.x * mt->accuracy.x * mt->squish.x;
	int index;

	for (; w > 0; w--, s++) {
	
		index = (s[0] << 8) | s[stride];
		
		if (greyblock_to_ascii[index] == 255) {
			calc_accuracy_1x2(index, mt->accuracy);
		}

		*d++ = 0x0700 | greyblock_to_ascii[index]; 
	}
}

static void blitter_2x2(MonotextHook *mt, void *dest, void *src, int w)
{
	uint8  *s = (uint8 *)  src;
	uint16 *d = (uint16 *) dest;

	int stride = mt->size.x * mt->accuracy.x * mt->squish.x;
	int index;

	for (; w > 1; w -= 2, s += 2) {
	
		index = ((s[stride*0 + 0] & 0xf0) << 8) |
		        ((s[stride*0 + 1] & 0xf0) << 4) |
		        ((s[stride*1 + 0] & 0xf0)     ) |
		        ((s[stride*1 + 1] & 0xf0) >> 4);
		
		if (greyblock_to_ascii[index] == 255) {
			calc_accuracy_2x2(index, mt->accuracy);
		}

		*d++ = 0x0700 | greyblock_to_ascii[index];
	}
}

static void blitter_2x4(MonotextHook *mt, void *dest, void *src, int w)
{
	uint8  *s = (uint8 *)  src;
	uint16 *d = (uint16 *) dest;

	int stride = mt->size.x * mt->accuracy.x * mt->squish.x;
	int index;

	for (; w > 1; w -= 2, s += 2) {
	
		index = ((s[stride*0 + 0] & 0xc0) << 8) |
		        ((s[stride*0 + 1] & 0xc0) << 6) |
		        ((s[stride*1 + 0] & 0xc0) << 4) |
		        ((s[stride*1 + 1] & 0xc0) << 2) |
	
		        ((s[stride*2 + 0] & 0xc0)     ) |
		        ((s[stride*2 + 1] & 0xc0) >> 2) |
		        ((s[stride*3 + 0] & 0xc0) >> 4) |
		        ((s[stride*3 + 1] & 0xc0) >> 6);
		
		if (greyblock_to_ascii[index] == 255) {
			calc_accuracy_2x4(index, mt->accuracy);
		}

		*d++ = 0x0700 | greyblock_to_ascii[index];
	}
}

static void blitter_4x4(MonotextHook *mt, void *dest, void *src, int w)
{
	uint8  *s = (uint8 *)  src;
	uint16 *d = (uint16 *) dest;

	int stride = mt->size.x * mt->accuracy.x * mt->squish.x;
	int index;

	for (; w > 3; w -= 4, s += 4) {
	
		index = ((s[stride*0 + 0] & 0x80) << 8) |
		        ((s[stride*0 + 1] & 0x80) << 7) |
		        ((s[stride*0 + 2] & 0x80) << 6) |
		        ((s[stride*0 + 3] & 0x80) << 5) |

		        ((s[stride*1 + 0] & 0x80) << 4) |
		        ((s[stride*1 + 1] & 0x80) << 3) |
		        ((s[stride*1 + 2] & 0x80) << 2) |
		        ((s[stride*1 + 3] & 0x80) << 1) |

		        ((s[stride*2 + 0] & 0x80)     ) |
		        ((s[stride*2 + 1] & 0x80) >> 1) |
		        ((s[stride*2 + 2] & 0x80) >> 2) |
		        ((s[stride*2 + 3] & 0x80) >> 3) |

		        ((s[stride*3 + 0] & 0x80) >> 4) |
		        ((s[stride*3 + 1] & 0x80) >> 5) |
		        ((s[stride*3 + 2] & 0x80) >> 6) |
		        ((s[stride*3 + 3] & 0x80) >> 7);

		if (greyblock_to_ascii[index] == 255) {
			calc_accuracy_4x4(index, mt->accuracy);
		}

		*d++ = 0x0700 | greyblock_to_ascii[index];
	}
}


/**************************************************
 ***
 ***  Exported functions 
 ***
 **************************************************/


static uint8 src_buf[8192];
static uint8 dest_buf[8192];


static inline void get_source_lines(ggi_visual *vis, int x, int y, int w,
				   uint8 *src)
{
	MonotextHook *mt = LIBGGI_PRIVATE(vis);

	int stride = mt->size.x * mt->accuracy.x * mt->squish.x;

	int num_w  = w / mt->squish.x;
	int i, j;

	for (j=0; j < mt->accuracy.y; j++, y+=mt->squish.y, src+=stride) {

		ggiGetHLine(vis, x, y, w, src);
		
		for (i=0; i < num_w; i++) {
			src[i] = mt->greymap[src[i * mt->squish.x]];
		}
	}
}

int _ggi_monotextUpdate(ggi_visual *vis, int x, int y, int w, int h)
{
	MonotextHook *mt = LIBGGI_PRIVATE(vis);

	int step_x = mt->accuracy.x * mt->squish.x;
	int step_y = mt->accuracy.y * mt->squish.y;
	
	/* round */

	if (y % step_y) {
		h += (y % step_y);
		y -= (y % step_y);
	}

	if (x % step_x) {
		w += (x % step_x);
		x -= (x % step_x);
	}

	/* do transfer */

	for (; h >= step_y; h -= step_y, y += step_y) {

		get_source_lines(vis, x, y, w, src_buf);

		(* mt->do_blit)(mt, dest_buf, src_buf, w);

		ggiPutHLine(mt->parent, x / step_x, y / step_y, w / step_x, 
			    dest_buf);
	}

	if (! (LIBGGI_FLAGS(vis) & GGIFLAG_ASYNC)) {
		ggiFlush(mt->parent);
	}
	
	return 0;
}

int _ggi_monotextFlush(ggi_visual *vis)
{
	MonotextHook *mt = LIBGGI_PRIVATE(vis);

	int sx = MAX(mt->dirty_tl.x, vis->gc->cliptl.x);
	int sy = MAX(mt->dirty_tl.y, vis->gc->cliptl.y);

	int ex = MIN(mt->dirty_br.x, vis->gc->clipbr.x);
	int ey = MIN(mt->dirty_br.y, vis->gc->clipbr.y);


	/* clear the 'dirty region' */

	mt->dirty_tl.x = mt->size.x;
	mt->dirty_tl.y = mt->size.y;
	mt->dirty_br.x = 0;
	mt->dirty_br.y = 0;


	if ((sx < ex) && (sy < ey)) {
		return _ggi_monotextUpdate(vis, sx, sy, ex-sx, ey-sy);
	}

	return 0;
}

int _ggi_monotextOpen(ggi_visual *vis)
{
	MonotextHook *mt = LIBGGI_PRIVATE(vis);

	ggi_coord child_size;

	mt->size = LIBGGI_MODE(vis)->visible;

	DPRINT("display-monotext: Open (size=%dx%d accuracy=%dx%d "
		"squish=%dx%d)\n", mt->size.x, mt->size.y,
		mt->accuracy.x, mt->accuracy.y, mt->squish.x, mt->squish.y);

	mt->colormap    = _ggi_malloc(256 * sizeof(ggi_color));
	mt->greymap     = _ggi_malloc(256);
	mt->rgb_to_grey = _ggi_malloc(32 * 32 * 32);

	child_size.x = mt->size.x / mt->accuracy.x / mt->squish.x;
	child_size.y = mt->size.y / mt->accuracy.y / mt->squish.y;

	mt->red_gamma = mt->green_gamma = mt->blue_gamma = 1.0;


	/* set the parent mode */

	if (ggiSetTextMode(mt->parent, child_size.x, child_size.y, 
		child_size.x, child_size.y, GGI_AUTO, GGI_AUTO,
		mt->parent_gt) < 0) {

		DPRINT("Couldn't set child graphic mode.\n");
		return -1;
	}


  	/* now calculate the conversion */

	setup_rgb2grey_table(mt->rgb_to_grey);
	setup_templates(mt->accuracy);


	/* setup tables and choose blitter function */

	if ((mt->accuracy.x == 1) && (mt->accuracy.y == 1)) {
		mt->do_blit = &blitter_1x1;
		
	} else if ((mt->accuracy.x == 1) && (mt->accuracy.y == 2)) {
		mt->do_blit = &blitter_1x2;
		
	} else if ((mt->accuracy.x == 2) && (mt->accuracy.y == 2)) {
		mt->do_blit = &blitter_2x2;
		
	} else if ((mt->accuracy.x == 2) && (mt->accuracy.y == 4)) {
		mt->do_blit = &blitter_2x4;
		
	} else if ((mt->accuracy.x == 4) && (mt->accuracy.y == 4)) {
		mt->do_blit = &blitter_4x4;

	} else {
		ggiPanic("display-monotext: INTERNAL ERROR: "
			"ACCURACY %dx%d not supported.\n", 
			mt->accuracy.x, mt->accuracy.y);
		exit(1);
	}

	
	/* Setup the greyblock_to_ascii[] array.  We are using lazy
	 * evaluation, which means that we do the expensive "template ->
	 * character" mapping on-the-fly.  Thus saying goodbye to the
	 * long delays at startup.
	 */

	 memset(greyblock_to_ascii, 255, sizeof(greyblock_to_ascii));


	/* clear the 'dirty region' */

	mt->dirty_tl.x = mt->size.x;
	mt->dirty_tl.y = mt->size.y;
	mt->dirty_br.x = 0;
	mt->dirty_br.y = 0;

	return 0;
}

int _ggi_monotextClose(ggi_visual *vis)
{
	MonotextHook *mt = LIBGGI_PRIVATE(vis);

	if (mt->colormap != NULL) {
		free(mt->colormap);
	}
	if (mt->greymap != NULL) {
		free(mt->greymap);
	}
	if (mt->rgb_to_grey != NULL) {
		free(mt->rgb_to_grey);
	}

	ggiClose(mt->parent);

	return 0;
}
