/* $Id: mode.c,v 1.25 1998/12/24 01:44:36 marcus Exp $
******************************************************************************

   LibGGI GLIDE target - Mode management.

   Copyright (C) 1997-1998 Jon Taylor	[taylorj@ecs.csus.edu]
   Copyright (C) 1998 Marcus Sundberg	[marcus@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <ggi/internal/ggi-dl.h>
#include <ggi/display/glide.h>
#include "../../default/color/color.h"

#include "../Linux_common/vtswitch.inc"


#define DOSHIFT(val, shift)  \
	(((shift) >= 0) ? (val) << (shift) : (val) >> -(shift))

static void
GGI_glide_gcchanged(ggi_visual *vis, int mask)
{
	if ((mask & GGI_GCCHANGED_CLIP)) {
		int 
			tlx = LIBGGI_GC(vis)->cliptl.x,
			tly = LIBGGI_GC(vis)->cliptl.y,
			brx = LIBGGI_GC(vis)->clipbr.x,
			bry = LIBGGI_GC(vis)->clipbr.y;
		if (tlx > 0) tlx--;
		if (tly > 0) tly--;
		if (brx < LIBGGI_VIRTX(vis)) brx--;
		if (bry < LIBGGI_VIRTY(vis)) bry--;

		grClipWindow(tlx, tly, brx, bry);
	}
	if ((mask & GGI_GCCHANGED_FG)) {
		color_hook *ck = (color_hook *) vis->colorpriv;
		GLIDE_PRIV(vis)->fgvertex.r
			= DOSHIFT(LIBGGI_GC(vis)->fg_color
				  & LIBGGI_PIXFMT(vis)->red_mask,
				  8 - ck->red_total);
		GLIDE_PRIV(vis)->fgvertex.g
			= DOSHIFT(LIBGGI_GC(vis)->fg_color
				  & LIBGGI_PIXFMT(vis)->green_mask,
				  8 - ck->green_total);
		GLIDE_PRIV(vis)->fgvertex.b =
			DOSHIFT(LIBGGI_GC(vis)->fg_color
				& LIBGGI_PIXFMT(vis)->blue_mask,
				8 - ck->blue_total);
		GLIDE_PRIV(vis)->fgvertex.a = 255;
	}
	if ((mask & GGI_GCCHANGED_BG)) {
		color_hook *ck = (color_hook *) vis->colorpriv;
		GLIDE_PRIV(vis)->bgvertex.r
			= DOSHIFT(LIBGGI_GC(vis)->bg_color
				  & LIBGGI_PIXFMT(vis)->red_mask,
				  8 - ck->red_total);
		GLIDE_PRIV(vis)->bgvertex.g
			= DOSHIFT(LIBGGI_GC(vis)->bg_color
				  & LIBGGI_PIXFMT(vis)->green_mask,
				  8 - ck->green_total);
		GLIDE_PRIV(vis)->bgvertex.b =
			DOSHIFT(LIBGGI_GC(vis)->bg_color
				& LIBGGI_PIXFMT(vis)->blue_mask,
				8 - ck->blue_total);
		GLIDE_PRIV(vis)->bgvertex.a = 255;
	}
}

int GGI_glide_flush(ggi_visual *vis, int tryflag)
{
	grSstIdle();
	
	return 0;
}

int GGI_glide_getapi(ggi_visual *vis, int num, char *apiname, char *arguments)
{
	switch(num)
	{
	case 0:
		strcpy(apiname, "display-glide");
		strcpy(arguments, "");
		return 0;
	case 1:
		strcpy(apiname, "generic-stubs");
		strcpy(arguments, "");
		return 0;
	case 2:
		strcpy(apiname, "generic-color");
		strcpy(arguments, "");
		return 0;
	}
	
	return -1;
}

		
int GGI_glide_setmode(ggi_visual *vis, ggi_mode *mode)
{
	struct glide_hooks *priv = LIBGGI_PRIVATE(vis);
	char libname[256], libargs[256];
	int id;

	GrScreenResolution_t resolution;
	
	if(GGI_glide_checkmode(vis, mode))
		return -1;
	
	/* Fill in ggi_pixelformat */
	memset(LIBGGI_PIXFMT(vis), 0, sizeof(ggi_pixelformat));
	LIBGGI_PIXFMT(vis)->depth=GT_DEPTH(mode->graphtype);
	LIBGGI_PIXFMT(vis)->size=GT_SIZE(mode->graphtype);

	switch(mode->graphtype)	{
	case GT_15BIT:
		LIBGGI_PIXFMT(vis)->blue_mask  = ((1<<5)-1);
		LIBGGI_PIXFMT(vis)->green_mask = ((1<<5)-1)<<5;
		LIBGGI_PIXFMT(vis)->red_mask   = ((1<<5)-1)<<(5+5);

		priv->src_format = GR_LFB_SRC_FMT_555;
		priv->bytes_per_pixel = 2;
		break;
		
	case GT_16BIT:
		LIBGGI_PIXFMT(vis)->blue_mask  = ((1<<5)-1);
		LIBGGI_PIXFMT(vis)->green_mask = ((1<<6)-1)<<5;
		LIBGGI_PIXFMT(vis)->red_mask   = ((1<<5)-1)<<(5+6);

		priv->src_format = GR_LFB_SRC_FMT_565;
		priv->bytes_per_pixel = 2;
		break;
		
	case GT_24BIT:
		LIBGGI_PIXFMT(vis)->blue_mask  = ((1<<8)-1);
		LIBGGI_PIXFMT(vis)->green_mask = ((1<<8)-1)<<8;
		LIBGGI_PIXFMT(vis)->red_mask   = ((1<<8)-1)<<(8+8);

		priv->src_format = GR_LFB_SRC_FMT_8888;
		priv->bytes_per_pixel = 3;
		break;
		
	case GT_32BIT:
		LIBGGI_PIXFMT(vis)->blue_mask  = ((1<<8)-1);
		LIBGGI_PIXFMT(vis)->green_mask = ((1<<8)-1)<<8;
		LIBGGI_PIXFMT(vis)->red_mask   = ((1<<8)-1)<<(8+8);

		priv->src_format = GR_LFB_SRC_FMT_8888;
		priv->bytes_per_pixel = 4;
		break;
	default:
		LIBGGI_ASSERT(0, "Glide: Illegal mode!");
		return -1;
	}
	_ggi_build_pixfmt(LIBGGI_PIXFMT(vis));
	
	if (mode->visible.x == 640 && mode->visible.y == 480)
	{
		resolution = GR_RESOLUTION_640x480;
	}
	else if (mode->visible.x == 800 && mode->visible.y == 600)
	{
		resolution = GR_RESOLUTION_800x600;
	}
	else if (mode->visible.x == 512 && mode->visible.y == 384)
	{
		resolution = GR_RESOLUTION_512x384;
	}
	else 
	{
		LIBGGI_ASSERT(0, "Invalid Glide resolution!");
		return -1;
	}

	if (priv->setmodesuccess) {
		if (priv->on_linux_cons) vtswitch_close(vis);
		grSstWinClose();
	}
	
	grSstWinOpen(0, resolution, GR_REFRESH_70Hz,
		     GR_COLORFORMAT_ABGR, GR_ORIGIN_UPPER_LEFT, 2, 0);
	vis->d_frame_num = 0;
	vis->r_frame_num = 0;
	vis->w_frame_num = 0;
	priv->readbuf = GR_BUFFER_FRONTBUFFER;
	priv->writebuf = GR_BUFFER_FRONTBUFFER;
	grRenderBuffer(GR_BUFFER_FRONTBUFFER);
	
	memcpy(LIBGGI_MODE(vis), mode, sizeof(ggi_mode));

	_ggiZapMode(vis, ~GGI_DL_OPDISPLAY);
	
	for (id = 1; GGI_glide_getapi(vis, id, libname, libargs) == 0; id++)
	{
		if (_ggiOpenDL(vis, libname, libargs, NULL) == NULL) 
		{
			fprintf(stderr, "display-glide: Error opening the "
				"%s (%s) library\n", libname, libargs);
			return -1;
		}
		DPRINT ("Success in loading %s (%s)\n", libname, libargs);
	}
	
	/* Drawing functions */
		/* framebuffer */
	vis->opdraw->setreadframe=GGI_glide_setreadframe;
	vis->opdraw->setwriteframe=GGI_glide_setwriteframe;
	vis->opdraw->setdisplayframe=GGI_glide_setdisplayframe;
		/* misc */
	vis->opdraw->fillscreen=GGI_glide_fillscreen;
		/* pixels */
	vis->opdraw->drawpixel_nc = GGI_glide_drawpixel;
	vis->opdraw->drawpixel    = GGI_glide_drawpixel;
	vis->opdraw->putpixel_nc  = GGI_glide_putpixel;
	vis->opdraw->putpixel     = GGI_glide_putpixel;
	vis->opdraw->getpixel     = GGI_glide_getpixel;
		/* lines */
	vis->opdraw->drawline     = GGI_glide_drawline;
	vis->opdraw->drawhline_nc = GGI_glide_drawhline;
	vis->opdraw->drawhline    = GGI_glide_drawhline;
	vis->opdraw->drawvline_nc = GGI_glide_drawvline;
	vis->opdraw->drawvline    = GGI_glide_drawvline;
	vis->opdraw->puthline     = GGI_glide_puthline;
	vis->opdraw->putvline     = GGI_glide_putvline;
	vis->opdraw->gethline     = GGI_glide_gethline;
	vis->opdraw->getvline     = GGI_glide_getvline;
		/* boxes */
	vis->opdraw->drawbox=GGI_glide_drawbox;
/*	vis->opdraw->copybox=GGI_glide_copybox;*/
	vis->opdraw->putbox=GGI_glide_putbox;
	vis->opdraw->getbox=GGI_glide_getbox;

	/* GC management */
	vis->opgc->gcchanged = GGI_glide_gcchanged;

	/* Text */
	if (GT_SIZE(mode->graphtype) == 16) {
		vis->opdraw->putc = GGI_glide16_putc;
	} else if (GT_SIZE(mode->graphtype) == 32) {
		vis->opdraw->putc = GGI_glide32_putc;
	}
	
	ggiIndicateChange(vis, GGI_CHG_APILIST);

	if (priv->on_linux_cons) {
		if (vtswitch_open(vis) < 0) {
			return -1;
		}
	}
		
	priv->setmodesuccess = 1;

	return 0;
}

int GGI_glide_checkmode(ggi_visual *vis, ggi_mode *tm)
{
	int err = 0;

	LIBGGI_APPASSERT(vis != NULL, "glide: Visual NULL in GGIcheckmode");
	
	if (tm->frames == GGI_AUTO) {
		tm->frames = 1;
	}
	if (tm->frames < 1) {
		err = -1;
		tm->frames = 1;
	}
	if (tm->frames > 2) {
		err = -1;
		tm->frames = 2;
	}

	if (tm->visible.x == GGI_AUTO) tm->visible.x = 512;
	if (tm->visible.y == GGI_AUTO) tm->visible.y = 384;

	if(!(	(tm->visible.x == 512 && tm->visible.y == 384) ||
		(tm->visible.x == 640 && tm->visible.y == 480) ||
		(tm->visible.x == 800 && tm->visible.y == 600))) {
		tm->visible.x = 512;
		tm->visible.y = 384;
		err = -1;
	}

	if (tm->virt.x == GGI_AUTO) {
		tm->virt.x = tm->visible.x;
	}
	if (tm->virt.y == GGI_AUTO) {
		tm->virt.y = tm->visible.y;
	}	

	if (tm->virt.x < tm->visible.x) {
		tm->virt.x = tm->visible.x;
		err = -1;
	}
	if (tm->virt.y < tm->visible.y) {
		tm->virt.y = tm->visible.y;
		err = -1;
	}

        if(     (tm->dpp.x!=1 && tm->dpp.x!=GGI_AUTO) ||
                (tm->dpp.y!=1 && tm->dpp.y!=GGI_AUTO))
                { err = -1; }
        tm->dpp.x = tm->dpp.y = 1;

	if (GT_SCHEME(tm->graphtype) != GT_TRUECOLOR
	    && GT_SCHEME(tm->graphtype) != GT_AUTO) {
		err = -1;
	}
	GT_SETSCHEME(tm->graphtype, GT_TRUECOLOR);
	if (GT_SUBSCHEME(tm->graphtype) != 0
	    && GT_SUBSCHEME(tm->graphtype) != GT_AUTO) {
		err = -1;
	}
	GT_SETSUBSCHEME(tm->graphtype, 0);

	/* FIXME!
	   Could someone please come up with a nice way to handle
	   SIZE/DEPTH properly...
	*/
	switch (GT_DEPTH(tm->graphtype)) {
	case 15:
	case 16:
		if (GT_SIZE(tm->graphtype) != 16
		    && GT_SIZE(tm->graphtype) != GT_AUTO) {
			err = -1;
		}
		GT_SETSIZE(tm->graphtype, 16);
		break;
	case 24:
		if (GT_SIZE(tm->graphtype) != 24
		    && GT_SIZE(tm->graphtype) != 32
		    && GT_SIZE(tm->graphtype) != GT_AUTO) {
			err = -1;
			GT_SETSIZE(tm->graphtype, 32);
		} else if (GT_SIZE(tm->graphtype) == GT_AUTO) {
			GT_SETSIZE(tm->graphtype, 32);
		}
		break;
	case GT_AUTO:
		break;
	default:
		GT_SETDEPTH(tm->graphtype, GT_AUTO);
		err = -1;
		break;
	}
	switch (GT_SIZE(tm->graphtype)) {
	case 16:
		if (GT_DEPTH(tm->graphtype) != 15
		    && GT_DEPTH(tm->graphtype) != 16
		    && GT_DEPTH(tm->graphtype) != GT_AUTO) {
			err = -1;
			GT_SETDEPTH(tm->graphtype, 16);
		} else if (GT_DEPTH(tm->graphtype) == GT_AUTO) {
			GT_SETDEPTH(tm->graphtype, 16);
		}
		break;
	case 24:
	case 32:
		if (GT_DEPTH(tm->graphtype) != 24
		    && GT_DEPTH(tm->graphtype) != GT_AUTO) {
			err = -1;
		}
		GT_SETDEPTH(tm->graphtype, 24);
		break;
	case GT_AUTO:
		break;
	default:
		GT_SETSIZE(tm->graphtype, GT_AUTO);
		err = -1;
		break;
	}
	if (GT_DEPTH(tm->graphtype) == GT_AUTO) {
		tm->graphtype = GT_16BIT;
	} else if (GT_SIZE(tm->graphtype) == GT_AUTO) {
		GT_SETSIZE(tm->graphtype, (GT_DEPTH(tm->graphtype)+1)&~8);
	}

        return err;
}

/*
** Get the current mode
*/
int GGI_glide_getmode(ggi_visual *vis, ggi_mode *tm)
{
	LIBGGI_APPASSERT(vis != NULL, "glide: GGIgetmode: Visual == NULL");
	
	/* We assume the mode in the visual to be OK 
	*/
	memcpy(tm, LIBGGI_MODE(vis), sizeof(ggi_mode));

	return 0;
}

void handle_switched_away(ggi_visual *vis)
{
	grSstControl(GR_CONTROL_DEACTIVATE);
}


void handle_switched_back(ggi_visual *vis)
{
	grSstControl(GR_CONTROL_ACTIVATE);
}
