/* $Id: mode.inc,v 1.47 1998/12/24 00:59:51 marcus Exp $
******************************************************************************

   Mode management for display-X and display-Xlib.

   Copyright (C) 1998 Marcus Sundberg   [marcus@ggi-project.org]
   Copyright (C) 1998 Steve Cheng       [steve@ggi-project.org]
   Copyright (C) 1998 Andreas Beck      [becka@ggi-project.org]
   Copyright (C) 1997 Jason McMullan    [jmcc@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

/*
 * This file is included by Xlib/mode.c and X/mode.c
 * It contains the following common functions:
 *
 *   static int _GGIgetbpp(ggi_visual *vis);
 *   static ggi_graphtype _GGIautodepth(ggi_visual *vis);
 *   static int _GGIbasiccheck(ggi_visual *vis,ggi_mode *tm,
                               XVisualInfo *vinfo);
 *   static int _GGImaskgetshift(int mask);
 *   int GGIcheckmode(ggi_visual *vis,ggi_mode *tm);
 *   int GGIgetmode(ggi_visual *vis,ggi_mode *tm);
 */  

static int _GGIgetbpp(ggi_visual *vis)
{
	XImage *bppcheckimage;
	unsigned int bits_per_pixel=0;
	
	if ((bppcheckimage=XGetImage(XLIB_PRIV(vis)->display,
				     RootWindow(XLIB_PRIV(vis)->display,
						XLIB_PRIV(vis)->screen),
				     0, 0, 1, 1, AllPlanes, ZPixmap)) != NULL) {
		bits_per_pixel=bppcheckimage->bits_per_pixel;
		XDestroyImage(bppcheckimage);
	}
	return bits_per_pixel;
}

static int _GGIbasiccheck(ggi_visual *vis,ggi_mode *tm, XVisualInfo *vinfo)
{
	int err = 0;
	struct Xhooks *xhook=LIBGGI_PRIVATE(vis);
	unsigned int bits_per_pixel = _GGIgetbpp(vis);
	ggi_graphtype good_gt;
	
	good_gt = GT_CONSTRUCT((bits_per_pixel > 24) ? 24 : bits_per_pixel,
	      (bits_per_pixel <= 8) ? GT_PALETTE : GT_TRUECOLOR, 
	      bits_per_pixel);

	if (tm->graphtype == GT_AUTO) {
		tm->graphtype = good_gt;
	}

	switch(tm->graphtype) {
	case GT_8BIT:
		if ((tm->virt.x & ~3) != tm->virt.x) {
			tm->virt.x=((tm->virt.x+3) & ~3);
			err = -1;
			break;
		}
		if (! XMatchVisualInfo(xhook->display, xhook->screen,
				       8, PseudoColor, vinfo)) {
			err = -1;
		}
		break;
	case GT_15BIT:
		if ((tm->virt.x & ~1) != tm->virt.x) {
			tm->virt.x=((tm->virt.x+1) & ~1);
			err = -1;
			break;
		}
		if (! XMatchVisualInfo(xhook->display, xhook->screen,
				       15, TrueColor, vinfo)) {
			err = -1;
		}
		break;
	case GT_16BIT:
		if((tm->virt.x & ~1) != tm->virt.x) {
			tm->virt.x=((tm->virt.x+1) & ~1);
			err = -1;
		}
		if (! XMatchVisualInfo(xhook->display, xhook->screen,
				       16, TrueColor, vinfo)) {
			err = -1;
		}
		break;
	case GT_24BIT:
		if (! XMatchVisualInfo(xhook->display, xhook->screen,
				       24, TrueColor, vinfo)
		    || bits_per_pixel != 24 ) {
			err = -1;
		}
		break;
	case GT_32BIT:
		if (! XMatchVisualInfo(xhook->display, xhook->screen,
				       24, TrueColor, vinfo)
		    || bits_per_pixel != 32 ) {
			err = -1;
		}
		break;
	default:
		err = -1;
	}
	if (err) {
		tm->graphtype = good_gt;
	}		
		
	return err;
}

static int _GGImaskgetshift(int mask)
{
	int shift;
	for (shift = 16; mask != 0; mask >>= 1) {
		shift--;
	}
	return shift;
}

/**********************************/
/* check any mode (text/graphics) */
/**********************************/
int GGI_X_checkmode(ggi_visual *vis,ggi_mode *tm)
{
	XVisualInfo vinfo;      /* Dummy here */
	struct Xhooks *xhook;
	int err = 0;

	LIBGGI_APPASSERT(vis != NULL, "GGIcheckmode: Visual == NULL");

	xhook=LIBGGI_PRIVATE(vis);

	if(tm->visible.x==GGI_AUTO) 
		tm->visible.x = tm->virt.x==GGI_AUTO? 640 : tm->virt.x;
	if(tm->visible.y==GGI_AUTO)
		tm->visible.y = tm->virt.y==GGI_AUTO? 480 : tm->virt.y;
	if(tm->virt.x==GGI_AUTO) {
		tm->virt.x = (tm->visible.x+3) & ~3;
	}
	if(tm->virt.y==GGI_AUTO)
		tm->virt.y = tm->visible.y;

	if(tm->virt.x < tm->visible.x) {
		tm->virt.x = (tm->visible.x+3) & ~3;
		err = -1;
	}
	if(tm->virt.y < tm->visible.y) {
		tm->virt.y = tm->visible.y;
		err = -1;
	}

	if(tm->frames==GGI_AUTO)
		tm->frames = 1;

#ifdef GGI_X_TARGET
	if(tm->frames>8) {
		tm->frames = 8;
#endif
#ifdef GGI_XLIB_TARGET
	if(tm->frames>1) {
		tm->frames = 1;
#endif
		err = -1;
	}

	
	if(	(tm->dpp.x!=1 && tm->dpp.x!=GGI_AUTO) ||
		(tm->dpp.y!=1 && tm->dpp.y!=GGI_AUTO)) {
		err = -1;
	}
	tm->dpp.x = tm->dpp.y = 1;

	if (_GGIbasiccheck(vis,tm,&vinfo) != 0) err = -1;

	return err;
}

/************************/
/* get the current mode */
/************************/
int GGI_X_getmode(ggi_visual *vis,ggi_mode *tm)
{
	LIBGGI_APPASSERT(vis != NULL, "GGIgetmode: Visual == NULL");

	/* We assume the mode in the visual to be o.k. */
	memcpy(tm,LIBGGI_MODE(vis),sizeof(ggi_mode));

	return 0;
}

int GGI_X_getapi(ggi_visual *vis,int num, char *apiname ,char *arguments)
{
	switch(num) {
		case 0:
#ifdef GGI_X_TARGET
			strcpy(apiname  ,"display-x");
#endif
#ifdef GGI_XLIB_TARGET
			strcpy(apiname  ,"display-xlib");
#endif
			strcpy(arguments,"");
			return 0;
		case 1: strcpy(apiname ,"generic-stubs"); 
			strcpy(arguments,"");
			return 0;
		case 2: strcpy(apiname ,"generic-color"); 
			strcpy(arguments,"");
			return 0;
#ifdef GGI_X_TARGET
                case 3:
			sprintf(apiname, "generic-linear-%d",  GT_SIZE(LIBGGI_MODE(vis)->graphtype));
                	strcpy(arguments,"");
			return 0;
#endif /* GGI_X_TARGET */
	}

	return -1;
}

static int _GGIdomode(ggi_visual *vis)
{
	int err,id;
	char sugname[256],args[256];
	
	_ggiZapMode(vis,~GGI_DL_OPDISPLAY);

	for(id=1;0==GGI_X_getapi(vis,id,sugname,args);id++) {
		err=(_ggiOpenDL(vis,sugname,args,NULL)==NULL);
		if (err) {
#ifdef GGI_X_TARGET
			fprintf(stderr, "display-x: Can't open the %s (%s) library.\n",				
				sugname,args);
#endif
#ifdef GGI_XLIB_TARGET
			fprintf(stderr,"display-xlib: Can't open the %s (%s) library.\n",
				sugname,args);
#endif
			/* In our special case, fail is always fatal. */
			return err;
		} else {
			DPRINT_LIBS("X: GGIsetmode: success in loading %s (%s)\n",sugname,args);
		}
	}

	/* color */
	/*vis->opcolor->mapcolor=GGImapcolor;*/
	/*vis->opcolor->unmappixel=GGIunmappixel;*/
	if(GT_SCHEME(LIBGGI_GT(vis))==GT_PALETTE)
		vis->opcolor->setpalvec=GGI_X_setpalvec;

#ifdef GGI_X_TARGET
	/* X can pan on a larger bitmap ;-) */
	vis->opdraw->setorigin=GGI_X_setorigin;

	/* X can also page flip ;-) */
	vis->opdraw->setdisplayframe=GGI_X_setdisplayframe;
#endif
#ifdef GGI_XLIB_TARGET
	vis->opgc->gcchanged=GGI_Xlib_gcchanged;
	 
		/* pixels */
	vis->opdraw->drawpixel_nc=GGI_Xlib_drawpixel;
	vis->opdraw->drawpixel=GGI_Xlib_drawpixel;
	vis->opdraw->putpixel_nc=GGI_Xlib_putpixel;
	vis->opdraw->putpixel=GGI_Xlib_putpixel;
	vis->opdraw->getpixel=GGI_Xlib_getpixel;

		/* lines */
	vis->opdraw->drawline=GGI_Xlib_drawline;
	vis->opdraw->drawhline_nc=GGI_Xlib_drawhline;
	vis->opdraw->drawhline=GGI_Xlib_drawhline;
	vis->opdraw->puthline=GGI_Xlib_puthline;
	vis->opdraw->drawvline_nc=GGI_Xlib_drawvline;
	vis->opdraw->drawvline=GGI_Xlib_drawvline;
	vis->opdraw->putvline=GGI_Xlib_putvline;
	 
		/* boxes */
	vis->opdraw->drawbox=GGI_Xlib_drawbox;
	vis->opdraw->putbox=GGI_Xlib_putbox;
	vis->opdraw->copybox=GGI_Xlib_copybox;
	vis->opdraw->getbox=GGI_Xlib_getbox;
	vis->opdraw->fillscreen=GGI_Xlib_fillscreen;
		/* text */
/*	Not operational yet. */
/*	vis->opdraw->putc=GGI_Xlib_putc; */
#endif /* GGI_XLIB_TARGET */
        
	ggiIndicateChange(vis, GGI_CHG_APILIST);

	return 0;
}

int GGI_X_setmode(ggi_visual *vis,ggi_mode *tm)
{
	struct Xhooks *xhook=LIBGGI_PRIVATE(vis);
	XSizeHints hint;
	unsigned int fg, bg;
	XVisualInfo vinfo;
	int rc;
#ifdef GGI_X_TARGET
	void *fbaddr;
#endif
	int i;
	
	if ((rc=GGI_X_checkmode(vis,tm))) return rc;

	ggLock(xhook->XLibLock);

#ifdef GGI_X_TARGET
	MANSYNC_ignore(vis);
#endif /* GGI_X_TARGET */

	_GGIbasiccheck(vis,tm,&vinfo);
	
	/* Fill in ggi_pixelformat shifts */
	memset(LIBGGI_PIXFMT(vis), 0, sizeof(ggi_pixelformat));
	xhook->red_shift=_GGImaskgetshift(LIBGGI_PIXFMT(vis)->red_mask=vinfo.red_mask);
	xhook->green_shift=_GGImaskgetshift(LIBGGI_PIXFMT(vis)->green_mask=vinfo.green_mask);
	xhook->blue_shift=_GGImaskgetshift(LIBGGI_PIXFMT(vis)->blue_mask=vinfo.blue_mask);

	/* we count on how much to shift _down_ a 32 bit value.*/
	LIBGGI_PIXFMT(vis)->red_shift  =16+xhook->red_shift;	
	LIBGGI_PIXFMT(vis)->green_shift=16+xhook->green_shift;	
	LIBGGI_PIXFMT(vis)->blue_shift =16+xhook->blue_shift;	
	LIBGGI_PIXFMT(vis)->depth=GT_DEPTH(tm->graphtype);
	LIBGGI_PIXFMT(vis)->size=GT_SIZE(tm->graphtype);
	LIBGGI_PIXFMT(vis)->clut_mask = 0;
	if (GT_SCHEME(tm->graphtype) == GT_PALETTE) {
		for (i=0; i < GT_DEPTH(tm->graphtype); i++)
			LIBGGI_PIXFMT(vis)->clut_mask
				= (LIBGGI_PIXFMT(vis)->clut_mask << 1) | 1;
	}
	LIBGGI_PIXFMT(vis)->stdformat=0;
	/* FIXME ! check prerequisites right ! */
	switch(tm->graphtype)
	{
		case GT_CONSTRUCT(8, GT_PALETTE, 8):
			LIBGGI_PIXFMT(vis)->stdformat=GGI_DB_STD_8a8i8;
			break;
		case GT_CONSTRUCT(15, GT_TRUECOLOR, 16):
			LIBGGI_PIXFMT(vis)->stdformat=GGI_DB_STD_16a16r5g5b5A1;
			break;
		case GT_CONSTRUCT(16, GT_TRUECOLOR, 16):
			LIBGGI_PIXFMT(vis)->stdformat=GGI_DB_STD_16a16r5g6b5;
			break;
		case GT_CONSTRUCT(24, GT_TRUECOLOR, 24):
			LIBGGI_PIXFMT(vis)->stdformat=GGI_DB_STD_24a8b8g8r8;
			break;
		case GT_CONSTRUCT(24, GT_TRUECOLOR, 32):
			LIBGGI_PIXFMT(vis)->stdformat=GGI_DB_STD_32a32b8g8r8A8;
			break;
	}
	_ggi_build_pixfmt(LIBGGI_PIXFMT(vis));

	
	DPRINT_MODE("X: Red,   shift: %ld, mask: 0x%x\n", xhook->red_shift,
		    LIBGGI_PIXFMT(vis)->red_mask);
	DPRINT_MODE("X: Green, shift: %ld, mask: 0x%x\n", xhook->green_shift,
		    LIBGGI_PIXFMT(vis)->green_mask);
	DPRINT_MODE("X: Blue,  shift: %ld, mask: 0x%x\n", xhook->blue_shift,
		    LIBGGI_PIXFMT(vis)->blue_mask);
	
#ifdef GGI_X_TARGET
	xhook->xoff=xhook->yoff=0;
	xhook->ysplit=tm->visible.y;
#endif /* GGI_X_TARGET */
	xhook->viswidth=tm->visible.x;
	xhook->visheight=tm->visible.y;

	/* Fill in hint structure. Minimal window until we know better. */
	hint.x      = hint.y      = 0;
	hint.width  = tm->visible.x;
	hint.height = tm->visible.y;
	hint.flags  = PSize | PMinSize | PMaxSize;
	hint.min_width = tm->visible.x;
	hint.min_height = tm->visible.y;
	hint.max_width = tm->visible.x;
	hint.max_height = tm->visible.y;
  
	/* Get some colors */
	bg = WhitePixel (xhook->display, xhook->screen);
	fg = BlackPixel (xhook->display, xhook->screen);
	DPRINT_MODE("X-lib has colors.\n");
  
	if (xhook->window) 
		XDestroyWindow(xhook->display,xhook->window);

#ifdef GGI_X_TARGET
	_GGI_X_freedbs(vis, xhook);
#endif /* GGI_X_TARGET */

	xhook->window = XCreateSimpleWindow (xhook->display,
				 RootWindow (xhook->display,xhook->screen),
				 hint.x, hint.y,
				 hint.width, hint.height,
				 2, fg, bg);
  
	DPRINT_MODE("X-lib has window 0x%x\n", xhook->window);

	/* Tell other applications about this window */
	XSetStandardProperties (xhook->display, xhook->window, "GGI-on-X", "GGI-on-X", None, NULL, 0, &hint);

	if (!xhook->gc)
	  	xhook->gc = XCreateGC(xhook->display, xhook->window, 0, 0);
#ifdef GGI_XLIB_TARGET
	if (!xhook->tempgc)
		xhook->tempgc = XCreateGC(xhook->display, xhook->window, 0, 0);
	DPRINT_MODE("X-lib GCs allocated.\n");
#endif
#ifdef GGI_X_TARGET
	DPRINT_MODE("X-lib GC allocated.\n");

#ifdef HAVE_SHM
	if (xhook->have_shm) {
		for(i = 0; i<tm->frames; i++) {

			DPRINT_MODE("X: Creating shm image #%d\n", i);

			xhook->ximage = XShmCreateImage(
				xhook->display, vinfo.visual, vinfo.depth,
				ZPixmap, 
				0, 
				&(xhook->shminfo[i]),
				tm->virt.x,tm->virt.y);

#ifdef LITTLE_ENDIAN_ARCHITECTURE
			xhook->ximage->byte_order = LSBFirst;
			xhook->ximage->bitmap_bit_order = LSBFirst;
#else
			xhook->ximage->byte_order = MSBFirst;
			xhook->ximage->bitmap_bit_order = MSBFirst;
#endif

			xhook->shminfo[i].shmid = 
				shmget(IPC_PRIVATE, (unsigned int) 
					tm->virt.x*tm->virt.y*vinfo.depth,
					IPC_CREAT | 0777);

			fbaddr = xhook->shminfo[i].shmaddr
				= xhook->ximage->data =
				(char *) shmat(xhook->shminfo[i].shmid,0,0);
			xhook->shminfo[i].readOnly=False;
	
			DPRINT_MODE("X: shmat success at %p.\n", fbaddr);

			/* Check if XShm is available */
			/* FIXME: This is a nasty hack to do global locking */
			while (curvis != NULL) {
				ggUSleep(10000);
			}
			curvis = vis;

			xhook->shmerror = 0;
			xhook->oldshmerrorhandler
				= XSetErrorHandler(shmerrorhandler);
			XShmAttach(xhook->display, &(xhook->shminfo[i]));

			XSync(xhook->display,0);
			XSetErrorHandler(xhook->oldshmerrorhandler);

			if (xhook->shmerror) {
				if (fbaddr!=NULL) {
					shmdt(fbaddr);
				}
				if (xhook->ximage) {
					XDestroyImage(xhook->ximage);
					xhook->ximage=NULL;
				}
				xhook->shminfo[i].shmid = -1;
				xhook->have_shm = 0;
				DPRINT_MODE("X: XShmAttach failed - trying without shm\n");
				break;
			} else {
				shmctl(xhook->shminfo[i].shmid, IPC_RMID, 0);
				DPRINT_MODE("X: ShmImage #%d allocated\n", i);
				xhook->ximage_list[i] = xhook->ximage;
			}
			curvis = NULL;
		}
	}

	if (! xhook->have_shm)
#endif	/* HAVE_SHM */
	{
		for(i = 0; i < tm->frames; i++) {
			fbaddr = (char *) _ggi_malloc(tm->virt.x*tm->virt.y
						      * vinfo.depth);
			DPRINT_MODE("X: Buffer allocated %p\n", fbaddr);

			xhook->ximage_list[i] = XCreateImage(
				xhook->display, vinfo.visual, vinfo.depth,
				ZPixmap, /* format */
				0, 	 /* offset */
				fbaddr,	/* data */
				tm->virt.x,tm->virt.y, /* size */
				8,	/* bitmap_pad*/
				0);
			DPRINT_MODE("X: Image #%d allocated\n", i);
		}
	}
	/* Set up directbuffers */
	for(i = 0; i < tm->frames; i++) {
		LIBGGI_APPLIST(vis)->last_targetbuf
			= _ggi_db_add_buffer(LIBGGI_APPLIST(vis), 
					     _ggi_db_get_new());
		LIBGGI_APPBUFS(vis)[i]->frame = i;
		LIBGGI_APPBUFS(vis)[i]->type
			= GGI_DB_NORMAL | GGI_DB_SIMPLE_PLB;
		LIBGGI_APPBUFS(vis)[i]->read = LIBGGI_APPBUFS(vis)[i]->write
			= xhook->ximage_list[i]->data;
		LIBGGI_APPBUFS(vis)[i]->layout = blPixelLinearBuffer;
		LIBGGI_APPBUFS(vis)[i]->buffer.plb.stride
			= xhook->ximage_list[i]->bytes_per_line;
		LIBGGI_APPBUFS(vis)[i]->buffer.plb.pixelformat
			= LIBGGI_PIXFMT(vis);
	}
	LIBGGI_APPLIST(vis)->first_targetbuf
		= LIBGGI_APPLIST(vis)->last_targetbuf - (tm->frames-1);
	xhook->ximage = xhook->ximage_list[0];

#endif /* GGI_X_TARGET */

	XResizeWindow(xhook->display, xhook->window,
		      tm->visible.x, tm->visible.y);
	DPRINT_MODE("X: Resize\n");

	DPRINT_MODE("X: About to map\n");
	/* Map window. */
	XSelectInput (xhook->display, xhook->window, ExposureMask);
	XMapRaised(xhook->display, xhook->window);
	XFlush(xhook->display);
		/* Wait for window to become mapped */
	{
		XEvent        event;
		XNextEvent (xhook->display, &event);
	}
	DPRINT_MODE("X: Window Mapped\n");

	/* Select input events to listen for */
	XSelectInput(xhook->display, xhook->window, 
		     KeyPressMask | KeyReleaseMask |
		     ButtonPressMask | ButtonReleaseMask |
		     EnterWindowMask | LeaveWindowMask |
		     ExposureMask | PointerMotionMask);

#ifdef GGI_XLIB_TARGET
	/* Turn on backing store */
	{
		XSetWindowAttributes attrib;
		attrib.backing_store=Always;
		XChangeWindowAttributes(xhook->display, xhook->window,
					CWBackingStore, &attrib);
	}
#endif /* GGI_XLIB_TARGET */

	if (xhook->cmap)
		XFreeColormap(xhook->display,xhook->cmap);
	if (vis->palette)
		free(vis->palette);

	/* FIXME: X/Xlib target should support any other modes X supports
	   besides GT_8BIT, etc. */
	if (GT_SCHEME(tm->graphtype)==GT_PALETTE) {
		Colormap defcmap;
		XColor colour_cell;

		xhook->nocols = 1 << GT_DEPTH(tm->graphtype);

		vis->palette = _ggi_malloc(sizeof(ggi_color) * xhook->nocols);

		/* get the deafult colour map (i.e. the shared one) 
		   (minimize colour flashing between windows) 
		   (Thanks to Wouter for this code.) */
		defcmap = DefaultColormap(xhook->display, xhook->screen);

		/* Create X window with AllocAll */
		xhook->cmap = XCreateColormap(xhook->display, xhook->window, 
						vinfo.visual, AllocAll);

		/* Fill the colourmap with the original colours */
		for (i=0; i<xhook->nocols; i++) {
			colour_cell.pixel = i;
			colour_cell.flags = DoRed | DoGreen | DoBlue;
			XQueryColor(xhook->display, defcmap, &colour_cell);
			XStoreColor(xhook->display, xhook->cmap, &colour_cell);

			vis->palette[i].r = colour_cell.red;
			vis->palette[i].g = colour_cell.green;
			vis->palette[i].b = colour_cell.blue;
		}

		xhook->cmap_first=256;
		xhook->cmap_last=0;
		 
		DPRINT_MODE("X: copied default colormap into private colormap (%x)\n", 
			    xhook->cmap);
	} else {
		xhook->cmap = XCreateColormap(xhook->display, xhook->window,
					      vinfo.visual, AllocNone);
	}

	ggUnlock(xhook->XLibLock);

	GGI_X_flush(vis, 1);
	
	DPRINT_MODE("X: Flush\n");
	
	memcpy(LIBGGI_MODE(vis),tm,sizeof(ggi_mode));


	rc = _GGIdomode(vis);
	
	/* Tell inputlib about the new window */
	{
		gii_event ev;
		gii_xwin_cmddata_setparam *data
			= (gii_xwin_cmddata_setparam *) &ev.cmd.data;
		ev.cmd.size = sizeof(gii_cmd_event);
		ev.cmd.type = evCommand;
		ev.cmd.target = xhook->inp->origin;
		ev.cmd.code = GII_CMDCODE_XWINSETPARAM;
		data->win = xhook->window;
		data->ptralwaysrel = 0;

		giiEventSend(xhook->inp, &ev);
	}
#ifdef GGI_X_TARGET
	MANSYNC_cont(vis);
#endif /* GGI_X_TARGET */

	return rc;
}
