/* $Id: vtswitch.inc,v 1.12 1998/12/22 14:25:22 jmcc Exp $
******************************************************************************

   Linux_common/vtswitch.c
   Linux VT switching code.

   Copyright (C) 1998  Andrew Apted  <andrew.apted@ggi-project.org>

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

   ---------------------------------------------------------------------

   USAGE:

	1. There needs to be two functions implemented as follows:
	handle_switched_away(vis) and handle_switched_back(vis).

	The first one is called when the console is being switched away.
	For example, under fbcon nothing is necessary (since mode
	switching is done by the kernel), but with the SUIDKGI target
	this routine should restore the card to the previous text mode.

	The second one is called when the console is being switched
	back.  For example, with the SUIDKGI target this would restore
	the program's graphics mode.

	NOTE WELL: these are called from signal handlers (SIGUSR1 and
	SIGUSR2 to be precise), so make sure that any code they use is
	async-signal safe.

	2. If vis->selectfd is somehow not the VT file descriptor
	(rather unlikely), then just #define VTSWITCH_VT_FD(vis) to
	something appropriate, and initialize it to -1 for good measure.
	
	3. Define VTSWITCH_PRIV(vis) to be a variable (type: void *)
	where some private information can be stored.  Initialize the
	pointer to NULL for good measure.

	4. Call vtswitch_open(vis) to open the tty, activate VT_PROCESS
	mode, and put the console into KD_GRAPHICS mode.  Returns 0 if
	successful.

	5. Call vtswitch_close(vis) to restore the tty to normal.

******************************************************************************
*/

#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <errno.h>

#include <sys/ioctl.h>

#include <linux/vt.h>
#include <linux/kd.h>


#ifndef VTSWITCH_VT_FD
#error VTSWITCH_VT_FD undefined!!!
#endif


/* Variables for the signal handlers of the horrible VT_PROCESS beasty.
 */

static volatile int __vt_back;
static ggi_visual * __vt_vis;


extern void handle_switched_away(ggi_visual *vis);
extern void handle_switched_back(ggi_visual *vis);


#define SETSIG(sig, fun) \
{ \
	struct sigaction sa; \
\
	sa.sa_handler = fun; \
	sa.sa_flags = 0; \
	sigemptyset(&sa.sa_mask); \
	sigaction(sig, &sa, NULL); \
}


static void release_vt_handler(int n)
{
	SETSIG(SIGUSR1, release_vt_handler);

	__vt_back=0;

	DPRINT("release_vt_handler START\n");

	handle_switched_away(__vt_vis);

	/* now release the VT */

	ioctl(VTSWITCH_VT_FD(__vt_vis), VT_RELDISP, 1);

	/* suspend program until we get switched back */
	
	DPRINT("release_vt_handler SUSPEND\n");

	while (! __vt_back) {    

		/* Note: we rely on the acquire signal interrupting us
		 */
		pause();
	}

	DPRINT("release_vt_handler DONE\n");
}


static void acquire_vt_handler(int n)
{
	SETSIG(SIGUSR2, acquire_vt_handler);

	DPRINT("acquire_vt_handler START\n");

	/* acknowledge the VT */

	ioctl(VTSWITCH_VT_FD(__vt_vis), VT_RELDISP, VT_ACKACQ);

	handle_switched_back(__vt_vis);

	__vt_back=1;

	DPRINT("acquire_vt_handler DONE\n");
}


int vtswitch_open(ggi_visual *vis)
{
	/* do the VT switching magic here */

	char filename[80];

	int qry_fd, our_vt;

	struct vt_stat qry_stat;
	struct vt_mode qry_mode;


	/* query free VT and current VT */

	qry_fd = open("/dev/tty", O_RDWR);

	if (qry_fd < 0) {
		perror("L/vtswitch: open /dev/tty");
		return -1;
	}

	if (ioctl(qry_fd, VT_GETSTATE, &qry_stat) != 0) {
		perror("L/vtswitch: VT_GETSTATE failed");
		fprintf(stderr, "L/vtswitch: (You need to be running "
			"directly on a virtual console).\n");
		close(qry_fd);
		return -1;
	}
	
	our_vt = qry_stat.v_active;

	close(qry_fd);


	/* open new VT */

	sprintf(filename, "/dev/tty%d", our_vt);

	VTSWITCH_VT_FD(vis) = open(filename, O_RDWR | O_NDELAY); 
	
	if (VTSWITCH_VT_FD(vis) < 0) {
		fprintf(stderr, "L/vtswitch: open %s: %s\n",
			filename, strerror(errno));
		return -1;
	}

	DPRINT_MISC("L/vtswitch: Using VT %d.\n", our_vt);
	
	
	/* Disable normal text on the console */

	ioctl(VTSWITCH_VT_FD(vis), KDSETMODE, KD_GRAPHICS);
	

	/* Unfortunately, we need to take control over VT switching like
	 * Xfree and SVGAlib does.  Sigh.
	 */

	ioctl(VTSWITCH_VT_FD(vis), VT_GETMODE, &qry_mode);

	qry_mode.mode = VT_PROCESS;
	qry_mode.relsig = SIGUSR1;
	qry_mode.acqsig = SIGUSR2;
	
	__vt_back = 1;
	__vt_vis  = vis;

	SETSIG(SIGUSR1, release_vt_handler);
	SETSIG(SIGUSR2, acquire_vt_handler);

	if (ioctl(VTSWITCH_VT_FD(vis), VT_SETMODE, &qry_mode) < 0) {
		perror("L/vtswitch: Setting VT mode failed");
		close(VTSWITCH_VT_FD(vis));
		return -1;
	}

	DPRINT("L/vtswitch: open OK.\n");

	return 0;
}


void vtswitch_close(ggi_visual *vis)
{
	struct vt_mode qry_mode;
	

	/* handle VT */

	if (VTSWITCH_VT_FD(vis) >= 0) {
		ioctl(VTSWITCH_VT_FD(vis), KDSETMODE, KD_TEXT);

		if (ioctl(VTSWITCH_VT_FD(vis), VT_GETMODE, &qry_mode) == 0) {
			qry_mode.mode = VT_AUTO;
			ioctl(VTSWITCH_VT_FD(vis), VT_SETMODE, &qry_mode);
		}
		
		signal(SIGUSR1, SIG_DFL);
		signal(SIGUSR2, SIG_DFL);

		close(VTSWITCH_VT_FD(vis));
		
		VTSWITCH_VT_FD(vis) = -1;
	}

	DPRINT("L/vtswitch: close OK.\n");
}
