/*****************************************************************************
 * css.c: Functions for DVD authentification and unscrambling
 *****************************************************************************
 * Copyright (C) 1999-2001 VideoLAN
 * $Id: css.c,v 1.8 2001/08/06 13:27:59 sam Exp $
 *
 * Author: Stphane Borel <stef@via.ecp.fr>
 *
 * based on:
 *  - css-auth by Derek Fawcus <derek@spider.com>
 *  - DVD CSS ioctls example program by Andrew T. Veliath <andrewtv@usa.net>
 *  - The Divide and conquer attack by Frank A. Stevenson <frank@funcom.com>
 *  - DeCSSPlus by Ethan Hawke
 *  - DecVOB
 *  see http://www.lemuria.org/DeCSS/ by Tom Vogt for more information.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************/

/*****************************************************************************
 * Preamble
 *****************************************************************************/
#include "defs.h"

#include <stdio.h>
#include <stdlib.h>

#include <string.h>

#include "config.h"
#include "common.h"

#include "videolan/dvdcss.h"
#include "libdvdcss.h"

#include "csstables.h"
#include "ioctl.h"

/*****************************************************************************
 * Local prototypes
 *****************************************************************************/
static int CSSDecryptDiscKey( dvd_key_t p_disc_key, u8* p_struct_disckey );
static int CSSDecryptTitleKey( dvd_key_t p_disc_key, dvd_key_t p_titlekey );

/*static*/ void CSSCryptKey  ( int i_key_type, int i_varient,
                           u8 const * p_challenge, u8* p_key );
static void CSSDecryptKey( int b_player_key, u8* p_key, u8* p_secret,  
			   u8* p_result);

static int  CSSCracker   ( int i_start, unsigned char * p_crypted,
                           unsigned char * p_decrypted,
                           u8* p_sector_seed, u8* p_key );
static int CSSAttackPattern( u8 p_sec[0x800], int i_pos, u8* p_key );
static int CSSAttackPadding( u8 p_sec[0x800], int i_pos, u8* p_key );


/*****************************************************************************
 * CSSTest : check if the disc is encrypted or not
 *****************************************************************************/
int CSSTest( dvdcss_handle dvdcss )
{
    int i_ret, i_copyright;

    i_ret = ioctl_ReadCopyright( dvdcss->i_fd, 0 /* i_layer */, &i_copyright );

    if( i_ret < 0 )
    {
        /* Since it's the first ioctl we try to issue, we add a notice */
        _dvdcss_error( dvdcss, "css error: ioctl_ReadCopyright failed, "
                       "make sure DVD ioctls were compiled in" );

        return i_ret;
    }

    return i_copyright;
}

/*****************************************************************************
 * CSSInit : DVD authentication.
 *****************************************************************************
 * It simulates the mutual authentication between logical unit and host.
 *****************************************************************************/
int CSSInit( dvdcss_handle dvdcss )
{
    /* Always do the full auth sequence. Some drives seem to 
     * lie and always respond with ASF=1. */
    /* The old DVD roms on Compaq Armada sais that ASF=1 from the start
       and then later fail with 'read of scrambled block without 
       authentication' errors. */
    /* Should we conditionalize this on b_use_titlekey perhaps? or
     * make the retriving of the disc key more lazy, i.e. just before
     * requesting the firs title key? Before the first read operation? 
     * That could work. */
    
    /* So this isn't really necessary except for debuging. */ 
    if( CSSGetASF( dvdcss ) < 0 )
    {
	_dvdcss_debug( dvdcss, "fatal error in CSS Init" );
	// ioctl_InvalidateAgid( dvdcss->i_fd, &dvdcss->css.i_agid ); ??
	return -1;
    }
    
    /* Go through the nessesary steps to get a Bus Key */
    if( CSSGetBusKey( dvdcss ) < 0 )
    {
	_dvdcss_debug( dvdcss, "fatal error in CSS Init" );
	return -1;
    }
    
    /* Request the Disc Key, this should set ASF=1 if successfull */
    if( CSSGetDiscKey( dvdcss ) < 0 )
    {
	_dvdcss_debug( dvdcss, "fatal error in CSS Init" );
	return -1;
    }
    
    /* Test authentication success */
    if( CSSGetASF( dvdcss ) != 1 )
    {
	_dvdcss_debug( dvdcss, "failed to authenticate drive" );
	// ioctl_InvalidateAgid( dvdcss->i_fd, &dvdcss->css.i_agid ); ?? 
	return -1;
    }

    _dvdcss_debug( dvdcss, "fully authenticated, success" );
    return 0;
}

/*static*/ void printHex( u8* data, unsigned int i_count )
{
    unsigned int i;
    for( i = 0; i < i_count; i++ )
    {
	fprintf( stderr, " %02x", data[i] );
    }
    fprintf( stderr, "\n" );
}

/*****************************************************************************
 * CSSGetBusKey : Go through the CSS Authentication process
 *****************************************************************************
 * It simulates the mutual authentication between logical unit and host
 * and stops when a Bus Key has been established. 
 * The Bus Key can then be used to get eihter the Disc Key or a Title Key.
 *****************************************************************************/
int CSSGetBusKey( dvdcss_handle dvdcss )
{
    unsigned char p_buffer[10];
    char          psz_warning[32];
    u8            p_challenge[2*KEY_SIZE];
    dvd_key_t     p_key1;
    dvd_key_t     p_key2;
    dvd_key_t     p_key_check;
    u8            i_varient = 0;
    int           i_ret = -1;
    int           i;
    
    _dvdcss_debug( dvdcss, "requesting AGID" );
    i_ret = ioctl_ReportAgid( dvdcss->i_fd, &dvdcss->css.i_agid );

    /* We might have to reset hung authentication processes in the drive 
       by invalidating the corresponding AGID'.  As long as we haven't got
       an AGID, invalidate one (in sequence) and try again. */
    for( i = 0; i_ret == -1 && i < 4 ; ++i )
      {
	_dvdcss_error( dvdcss, "ioctl_ReportAgid failed" );
	
	sprintf( psz_warning, "invalidating AGID %d", i );
        _dvdcss_debug( dvdcss, psz_warning );
	
	/* This is really _not good_, should be handled by the OS.
	   Invalidating an AGID could make another process fail some
	   where in it's authentication process. */
        dvdcss->css.i_agid = i;
        ioctl_InvalidateAgid( dvdcss->i_fd, &dvdcss->css.i_agid );
        
	_dvdcss_debug( dvdcss, "requesting AGID" );
        i_ret = ioctl_ReportAgid( dvdcss->i_fd, &dvdcss->css.i_agid );
    }

    /* Unable to begin authentication process without an AGID */
    if( i_ret == -1 )
    {
        _dvdcss_error( dvdcss, "ioctl_ReportAgid failed, fatal" );
        return -1;
    }

    /* Setup some challenge */
    for( i = 0 ; i < 10; ++i )
    {
        p_challenge[i] = i;
        //p_challenge[i] = 0;
        //p_challenge[i] = 0xff - i;
    }
    
    //fprintf( stderr, "libdvdcss debug: challenge is" );
    //printHex( p_challenge, 10 );
    
    /* Get challenge from host */
    for( i = 0 ; i < 10 ; ++i )
    {
        p_buffer[9-i] = p_challenge[i];
    }

    /* Send challenge to LU */
    if( ioctl_SendChallenge( dvdcss->i_fd, 
			     &dvdcss->css.i_agid, p_buffer ) < 0 )
    {
        _dvdcss_error( dvdcss, "ioctl_SendChallenge failed" );
        // ioctl_InvalidateAgid( dvdcss->i_fd, &dvdcss->css.i_agid ); ??
	return -1;
    }

    /* Get key1 from LU */
    if( ioctl_ReportKey1( dvdcss->i_fd, &dvdcss->css.i_agid, p_buffer ) < 0)
    {
        _dvdcss_error( dvdcss, "ioctl_ReportKey1 failed" );
        // ioctl_InvalidateAgid( dvdcss->i_fd, &dvdcss->css.i_agid ); ??
        return -1;
    }

    /* Send key1 to host */
    for( i = 0 ; i < KEY_SIZE ; i++ )
    {
        p_key1[i] = p_buffer[4-i];
    }
    
    //fprintf( stderr, "libdvdcss debug: key1 is" );
    //printHex( p_key1, KEY_SIZE );
    
    for( i = 0 ; i < 32 ; ++i )
    {
        CSSCryptKey( 0, i, p_challenge, p_key_check );

	//fprintf( stderr, "libdvdcss debug: variant %d gives", i );
	//printHex( p_key_check, KEY_SIZE );
    
        if( memcmp( p_key_check, p_key1, KEY_SIZE ) == 0 )
        {
            sprintf( psz_warning, "drive authentic, using variant %d", i );
            _dvdcss_debug( dvdcss, psz_warning );
            i_varient = i;
            break;
        }
    }

    if( i == 32 )
    {
        _dvdcss_error( dvdcss, "drive not authentic" );
        // ioctl_InvalidateAgid( dvdcss->i_fd, &dvdcss->css.i_agid ); ??
        return -1;
    }

    /* Get challenge from LU */
    if( ioctl_ReportChallenge( dvdcss->i_fd, 
			       &dvdcss->css.i_agid, p_buffer ) < 0 )
    {
        _dvdcss_error( dvdcss, "ioctl_ReportKeyChallenge failed" );
        // ioctl_InvalidateAgid( dvdcss->i_fd, &dvdcss->css.i_agid ); ??
        return -1;
    }

    /* Send challenge to host */
    for( i = 0 ; i < 10 ; ++i )
    {
        p_challenge[i] = p_buffer[9-i];
    }

    CSSCryptKey( 1, i_varient, p_challenge, p_key2 );

    /* Get key2 from host */
    for( i = 0 ; i < KEY_SIZE ; ++i )
    {
        p_buffer[4-i] = p_key2[i];
    }

    /* Send key2 to LU */
    if( ioctl_SendKey2( dvdcss->i_fd, &dvdcss->css.i_agid, p_buffer ) < 0 )
    {
        _dvdcss_error( dvdcss, "ioctl_SendKey2 failed" );
        // ioctl_InvalidateAgid( dvdcss->i_fd, &dvdcss->css.i_agid ); ??
        return -1;
    }
    
    /* The drive has accepted us as authentic. */
    _dvdcss_debug( dvdcss, "host authentication established" );

    memcpy( p_challenge, p_key1, KEY_SIZE );
    memcpy( p_challenge + KEY_SIZE, p_key2, KEY_SIZE );

    CSSCryptKey( 2, i_varient, p_challenge, dvdcss->css.p_bus_key );

    return 0;
}

/*****************************************************************************
 * CSSGetDiscKey : Finish the CSS Authentication, requesting the Disc Key
 *****************************************************************************
 * CSSGetBusKey should have compleated successfully before calling us.
 * It will set dvdcss->b_use_titlekey=0 if it fails to find a disc key.
 * (Change it to instead return the status and let the caller set this?)
 *****************************************************************************/
int CSSGetDiscKey( dvdcss_handle dvdcss )
{
    unsigned char p_buffer[2048];
    int i;
    
    /* Get encrypted disc key */
    if( ioctl_ReadDiscKey( dvdcss->i_fd, &dvdcss->css.i_agid, p_buffer ) < 0 )
    {
        _dvdcss_error( dvdcss, "ioctl_ReadDiscKey failed" );
        return -1;
    }

    /* Unencrypt disc key using the bus key */
    for( i = 0 ; i < 2048 ; i++ )
    {
        p_buffer[ i ] ^= dvdcss->css.p_bus_key[ 4 - (i % KEY_SIZE) ];
    }
    
    if( CSSDecryptDiscKey( p_buffer, dvdcss->css.p_disc_key ) )
    {
        _dvdcss_debug( dvdcss, "CSSGetDiscKey failed to find a valid key" );
        /* No disc key means that we won't be able to decrypt title keys */
	dvdcss->b_use_titlekey = 0;
    }
    
    /* This should have gotten us to ASF=1 and invaidated the AGID. */
    return 0;
}

/*****************************************************************************
 * CSSGetTitleKey : Finish the CSS Authentication, requesting a Title Key
 *****************************************************************************
 * CSSGetBusKey should have compleated successfully before calling us.
 *****************************************************************************/
int CSSGetTitleKey( dvdcss_handle dvdcss, int i_lba, u8* p_key )
{
    unsigned char p_buffer[KEY_SIZE];
    int i, i_ret = 0;
    
    /* Get encrypted title key */
    if( ioctl_ReportTitleKey( dvdcss->i_fd, 
			      &dvdcss->css.i_agid, i_lba, p_buffer ) < 0 )
    {
        _dvdcss_error( dvdcss, "ioctl_ReportTitleKey failed" );
        i_ret = -1;
    }
    
    /* Test ASF, it will be reset to 0 if we got a Region error */
    switch( CSSGetASF( dvdcss ) )
    {
        case -1:
	    // ioctl_InvalidateAgid( dvdcss->i_fd, &dvdcss->css.i_agid );
            return -1;

        case 1:
	    /* This might be an title that has no key or a region error. */
	    if( i_ret )
	    {
		ioctl_InvalidateAgid( dvdcss->i_fd, &dvdcss->css.i_agid );
		return i_ret;
	    }
	    break;

        case 0:
            _dvdcss_debug( dvdcss, "Lost authentication reqesting Title key" );
	    return i_ret;
    }
    
    /* Unencrypt disc key using the bus key */
    for( i = 0 ; i < KEY_SIZE ; i++ )
    {
        p_buffer[ i ] ^= dvdcss->css.p_bus_key[ 4 - (i % KEY_SIZE) ];
    }
    
    /* All zero key indicates an unencrypted VOBS ? */
    if( p_buffer[0] | p_buffer[1] | p_buffer[2] | p_buffer[3] | p_buffer[4] )
    {
	CSSDecryptTitleKey( dvdcss->css.p_disc_key, p_buffer );
    }
    memcpy( p_key, p_buffer, KEY_SIZE );
    
    fprintf( stderr, "libdvdcss debug: title key is" );
    printHex( p_key, KEY_SIZE );
    
    _dvdcss_debug( dvdcss, "got title key" );
    return 1;
}


/*****************************************************************************
 * CSSGetASF : Get Authentification success flag
 *****************************************************************************
 * Returns :
 *  -1 on ioctl error,
 *  0 if the device needs to be authenticated,
 *  1 either.
 *****************************************************************************/
int CSSGetASF( dvdcss_handle dvdcss )
{
    int i_asf = 0;
  
    if( ioctl_ReportASF( dvdcss->i_fd, &i_asf ) != 0 )
    {
	/* The ioctl process has failed */
	_dvdcss_error( dvdcss, "GetASF fatal error" );
	return -1;
    }	
    
    if( i_asf )
    {
	_dvdcss_debug( dvdcss, "GetASF authenticated (ASF=1)" );
    }
    else
    {
	_dvdcss_debug( dvdcss, "GetASF not authenticated (ASF=0)" );
    }
    
    return i_asf;
}


/*****************************************************************************
 * CSSDescrambleSector
 *****************************************************************************
 * sec : sector to descramble
 * key : title key for this sector
 *****************************************************************************/
int CSSDescrambleSector( dvd_key_t p_key, u8* p_sec )
{
    unsigned int    i_t1, i_t2, i_t3, i_t4, i_t5, i_t6;
    u8*             p_end = p_sec + 0x800;

    /* PES_scrambling_control */
    if( p_sec[0x14] & 0x30)
    {
        i_t1 = (p_key[0] ^ p_sec[0x54]) | 0x100;
        i_t2 = p_key[1] ^ p_sec[0x55];
        i_t3 = (p_key[2] | (p_key[3] << 8) |
               (p_key[4] << 16)) ^ (p_sec[0x56] |
               (p_sec[0x57] << 8) | (p_sec[0x58] << 16));
        i_t4 = i_t3 & 7;
        i_t3 = i_t3 * 2 + 8 - i_t4;
        p_sec += 0x80;
        i_t5 = 0;

        while( p_sec != p_end )
        {
            i_t4 = p_css_tab2[i_t2] ^ p_css_tab3[i_t1];
            i_t2 = i_t1>>1;
            i_t1 = ( ( i_t1 & 1 ) << 8 ) ^ i_t4;
            i_t4 = p_css_tab5[i_t4];
            i_t6 = ((((((( i_t3 >> 3 ) ^ i_t3 ) >> 1 ) ^
                                         i_t3 ) >> 8 ) ^ i_t3 ) >> 5) & 0xff;
            i_t3 = (i_t3 << 8 ) | i_t6;
            i_t6 = p_css_tab4[i_t6];
            i_t5 += i_t6 + i_t4;
            *p_sec = p_css_tab1[*p_sec] ^ ( i_t5 & 0xff );
            p_sec++;
            i_t5 >>= 8;
        }
    }

    return 0;
}



/* Include Frank's Disc key cracker instead of all the player keys? */
dvd_key_t player_keys[ 31 ] = 
{
    { 0x01, 0xaf, 0xe3, 0x12, 0x80 },
    { 0x12, 0x11, 0xca, 0x04, 0x3b },
    { 0x14, 0x0c, 0x9e, 0xd0, 0x09 },
    { 0x14, 0x71, 0x35, 0xba, 0xe2 },
    { 0x1a, 0xa4, 0x33, 0x21, 0xa6 },
    { 0x26, 0xec, 0xc4, 0xa7, 0x4e },
    { 0x2c, 0xb2, 0xc1, 0x09, 0xee },
    { 0x2f, 0x25, 0x9e, 0x96, 0xdd },
    { 0x33, 0x2f, 0x49, 0x6c, 0xe0 },
    { 0x35, 0x5b, 0xc1, 0x31, 0x0f },
    { 0x36, 0x67, 0xb2, 0xe3, 0x85 },
    { 0x39, 0x3d, 0xf1, 0xf1, 0xbd },
    { 0x3b, 0x31, 0x34, 0x0d, 0x91 },
    { 0x45, 0xed, 0x28, 0xeb, 0xd3 },
    { 0x48, 0xb7, 0x6c, 0xce, 0x69 },
    { 0x4b, 0x65, 0x0d, 0xc1, 0xee },
    { 0x4c, 0xbb, 0xf5, 0x5b, 0x23 },
    { 0x51, 0x67, 0x67, 0xc5, 0xe0 },
    { 0x53, 0x94, 0xe1, 0x75, 0xbf },
    { 0x57, 0x2c, 0x8b, 0x31, 0xae },
    { 0x63, 0xdb, 0x4c, 0x5b, 0x4a },
    { 0x7b, 0x1e, 0x5e, 0x2b, 0x57 },
    { 0x85, 0xf3, 0x85, 0xa0, 0xe0 },
    { 0xab, 0x1e, 0xe7, 0x7b, 0x72 },
    { 0xab, 0x36, 0xe3, 0xeb, 0x76 },
    { 0xb1, 0xb8, 0xf9, 0x38, 0x03 },
    { 0xb8, 0x5d, 0xd8, 0x53, 0xbd },
    { 0xbf, 0x92, 0xc3, 0xb0, 0xe2 },
    { 0xcf, 0x1a, 0xb2, 0xf8, 0x0a },
    { 0xec, 0xa0, 0xcf, 0xb3, 0xff },
    { 0xfc, 0x95, 0xa9, 0x87, 0x35 }
};

/*****************************************************************************
 * CSSDecryptDiscKey
 *****************************************************************************
 * Try to decrypt the disc key from every position with every player key.
 * p_struct_disckey: the 2048 byte DVD_STRUCT_DISCKEY data
 * p_disc_key: result, the 5 byte disc key
 *****************************************************************************/
static int CSSDecryptDiscKey( u8* p_struct_disckey, dvd_key_t p_disc_key )
{
    unsigned char p_verify[ 5 ];
    int i, n = 0;
    
    while( n < sizeof(player_keys)/sizeof(dvd_key_t) )
    {
	for( i = 1; i < 409; i++ )
	{
	    /* Check if player key n is the right key for position i */
	    CSSDecryptKey( 0, player_keys[n], p_struct_disckey + 5 * i, 
			   p_disc_key );
	    
	    /* The first part in the struct_disckey block is the 
	       'disc key' encrypted with it self.  Using this we 
	       can check if we decrypted the correct key. */
	    CSSDecryptKey( 0, p_disc_key, p_struct_disckey, p_verify );
	    if( memcmp( p_disc_key, p_verify, 5 ) == 0 )
	    {
		/* Found a possition, player key pair that worked */
		return 0;
	    }
	}
	n++;
    }
    
    /* Tried all combinations of keys and still didn't succeed. */
    return -1;
}

/*****************************************************************************
 * CSSDecryptDiscKey
 *****************************************************************************
 * Decrypt the title key using the disc key.
 * p_disc_key: result, the 5 byte disc key
 * p_titlekey: the encrypted title key, gets overwritten by the decrypted key
 *****************************************************************************/
static int CSSDecryptTitleKey( dvd_key_t p_disc_key, dvd_key_t p_titlekey )
{
    CSSDecryptKey( 1, p_disc_key, p_titlekey, p_titlekey );
    return 0;
}



/******************************************************************************
 * Various pices for the crack engine.
 ******************************************************************************
 * The length of the PES packet is located at 0x12-0x13.
 * The the copyrigth protection bits are located at 0x14 (bits 0x20 and 0x10).
 * The data of the PES packet begins at 0x15 (if there isn't any PTS/DTS)
 * or at 0x?? if there are both PTS and DTS's.
 * The seed value used with the descrambler key is the 5 bytes at 0x54-0x58.
 * The scrabled part of a sector begins at 0x80. 
 *****************************************************************************/

/* Statistics */
static int i_tries = 0, i_success = 0;


#define key_seed &p_sec[0x54]



/* The original Ethan Hawke (DeCSSPlus) attack (modified). 
 * Tries to find a repeating pattern just before the encrypted part starts.
 * Then it guesses that the plain text for fisrt encrypted bytes are
 * a contiuation of that pattern.
 */
static int CSSAttackPattern( u8 p_sec[0x800], int i_pos, u8* p_key )
{
    unsigned int i_best_plen = 0;
    unsigned int i_best_p = 0;
    unsigned int i, j;
    //static int i_tries = 0, i_success = 0;
    
    /* For all cycle length from 2 to 48 */
    /* FIXME: for debuging I also allow 1 (for repeated bytes) */
    for( i = 1 ; i < 0x30 ; i++ )
    {
	/* Find the number of bytes that repeats in cycles. */
	for( j = i + 1;
	     j < 0x80 && ( p_sec[0x7F - (j%i)] == p_sec[0x7F - j] );
	     j++ )
	{
	    /* We have found j repeating bytes with a cycle length i */
	    if( j > i_best_plen )
	    {
		i_best_plen = j;
		i_best_p = i;
	    }
	}
    }
    
    /* We need 10 plain text bytes?, so a make sure that we have 
       at least 20 repeated bytes and that they have cycled at 
       least one time. */ 
    if( ( i_best_plen > 3 ) && ( i_best_plen / i_best_p >= 2) )
    {
        int res;
	
	i_tries++;
	memset( p_key, 0, KEY_SIZE );
	res = CSSCracker( 0,  &p_sec[0x80],
			  &p_sec[0x80 - ( i_best_plen / i_best_p) * i_best_p],
			  key_seed, p_key );
	i_success += ( res >= 0 );
	
	if( 0 )
	{
	    fprintf( stderr, "key is %02x %02x %02x %02x %02x ",
		     p_key[0], p_key[1], p_key[2], p_key[3], p_key[4] );
	    fprintf( stderr, "at block %5d pattern len %3d period %3d %s\n", 
		     i_pos, i_best_plen, i_best_p, (res>=0?"y":"n") );
	}

	return ( res >= 0 );
    }
    
    return 0;
}



/* Encrypted Padding_stream attack.
 
   DVD specifies that there must only be one type of data in every sector.
   Every sector is one pack and so must obviously be 2048 bytes long.
   For the last pice of video data before a VOBU boundary there might not
   be exactly the right amount of data to fill a sector. They one has to 
   pad the pack to 2048 bytes. For just a few bytes this is doen in the
   header but for any large amount you insert a PES packet from the 
   Padding stream. This looks like 0x00 00 01 be xx xx ff ff ...
   where xx xx is the length of the padding stream.
 */
static int CSSAttackPadding( u8 p_sec[0x800], int i_pos, u8* p_key )
{
    unsigned int i_pes_length;
    //unsigned int i;
    //static int i_tries = 0, i_success = 0;

    i_pes_length = (p_sec[0x12]<<8) | p_sec[0x13];
    
    // Coverd by the test below but usfull for debuging
    if( i_pes_length == 0x800 - 0x14 ) return 0;
    
    /* There must be room for at least 4? bytes of padding stream, 
       and it must be encrypted.
       sector size - pack/pes header - padding startcode - padding length */
    if( ( 0x800 - 0x14 - 4 - 2 - i_pes_length < 4 ) ||
	( p_sec[0x14 + i_pes_length + 0] == 0x00 &&
	  p_sec[0x14 + i_pes_length + 1] == 0x00 &&
	  p_sec[0x14 + i_pes_length + 2] == 0x01 ) )
    { 
      fprintf( stderr, "plain %d %02x %02x %02x %02x (type %02x sub %02x)\n", 
	       0x800 - 0x14 - 4 - 2 - i_pes_length,
	       p_sec[0x14 + i_pes_length + 0],
	       p_sec[0x14 + i_pes_length + 1],
	       p_sec[0x14 + i_pes_length + 2],
	       p_sec[0x14 + i_pes_length + 3],
	       p_sec[0x11], p_sec[0x17 + p_sec[0x16]]);
      return 0;
    }
    
    /* If we are here we know that there is a where in the pack a
       encrypted PES header is (startcode + lenght). It's never more 
       than  two packets in the pack, so we 'know' the length. The 
       plaintext at offset (0x14 + i_pes_length) will then be 
       00 00 01 e0/bd/be xx xx, in the case of be the following bytes 
       are also known. */
    
    /* A encrypted SPU PES packet with another encrypted PES packet following.
       Normaly if the following was a padding stream that would be in plain
       text. So it will be another SPU PES packet. */
    if( p_sec[0x11] == 0xbd && 
	p_sec[0x17 + p_sec[0x16]] >= 0x20 &&
	p_sec[0x17 + p_sec[0x16]] <= 0x3f )
    {
        i_tries++;
    }
    
    /* A Video PES packet with another encrypted PES packet following.
       No reason execpt for time stamps to break the data into two packets.
       So it's likely that the following PES packet is a padding stream. */
    if( p_sec[0x11] == 0xe0 )
    { 
        i_tries++;
    }
   
    
    if( 1 )
    {
        //fprintf( stderr, "key is %02x %02x %02x %02x %02x ",
        //	     p_key[0], p_key[1], p_key[2], p_key[3], p_key[4] );
        fprintf( stderr, "at block %5d padding len %4d "
		 "type %02x sub %02x\n",  i_pos, i_pes_length, 
		 p_sec[0x11], p_sec[0x17 + p_sec[0x16]]);
    }
    
    return 0;
}





/*****************************************************************************
 * CSSGetKey : crack title key.
 *****************************************************************************
 * The DVD should have been opened and authenticated before.
 *****************************************************************************/
int CSSGetKey( dvdcss_handle dvdcss, int i_pos, int i_len, 
	       dvd_key_t p_titlekey )
{
    u8          p_buf[0x800];
    u8          p_packstart[4] = { 0x00, 0x00, 0x01, 0xba };
    dvd_key_t   p_key;
    int         i_reads;
    int         i_encrypted;
    //int         i_tries, i_success;    
    boolean_t   b_stop_scanning;
    int         i_blocks_read;
    int         i;

    for( i = 0 ; i < KEY_SIZE ; i++ )
    {
        p_key[i] = 0;
    }

    i_reads = 0;
    i_encrypted = 0;
    i_tries = 0;
    i_success = 0;
    b_stop_scanning = 0;

    do
    {
        i_pos = dvdcss_seek( dvdcss, i_pos );
        i_blocks_read = dvdcss_read( dvdcss, p_buf, 1, DVDCSS_NOFLAGS );
	
	if( !i_blocks_read )
	{
	    _dvdcss_debug( dvdcss, "read returned 0 (end of device?)" );
	    break;
	}
	    
	/* Hack to stop when we find a non MPEG stream block. */
	/* For now allow all blocks that begin with a start code */
	if( memcmp( p_buf, p_packstart, 3 ) )
	{
	    _dvdcss_debug( dvdcss, "non MPEG block found (end of title)" );
	    break;
	}
	
	if( p_buf[0x0d] & 0x07 )
	    fprintf( stderr, "stuffing in pack header\n" ); 
	
        /* PES_scrambling_control does not exist in a system_header or 
	   a padding stream or a private_stream2. (and others). */
        if( p_buf[0x14] & 0x30  && ! ( p_buf[0x11] == 0xbb 
				       || p_buf[0x11] == 0xbe  
				       || p_buf[0x11] == 0xbf ) )
        {
            i_encrypted++;
            
	    i = CSSAttackPattern(p_buf, i_reads, p_key);
	    b_stop_scanning = ( i > 0 );
	    
	    //i = CSSAttackPadding(p_buf, i_reads, p_key);
	    //b_stop_scanning |= ( i > 0 );
	}

        i_pos += i_blocks_read;
	i_len -= i_blocks_read;
	i_reads += i_blocks_read;
	
	/* Stop after 50000 blocks */
	if( !( i_reads % 5000 ) ) 
	  _dvdcss_debug( dvdcss, "still working..." );
	
	/* Stop after 2000 blocks if we haven't seen any encrypted ones */
	if( i_reads >= 2000 && i_encrypted == 0 ) break;
	
    } while( !b_stop_scanning && i_len > 0);

    if( i_len <= 0 )
        _dvdcss_debug( dvdcss, "next title reached" );
    
    {
        char psz_info[128];
	snprintf( psz_info, sizeof(psz_info), 
		  "%d of %d tries in %d scrambled of %d total blocks", 
		  i_success, i_tries, i_encrypted, i_reads );
	_dvdcss_debug( dvdcss, psz_info );
    }
    
    if( i_success > 0 /* b_stop_scanning */ )
    {
        memcpy( p_titlekey, &p_key, sizeof(dvd_key_t) );
        _dvdcss_debug( dvdcss, "vts key initialized" );
        return 0;
    }

    if( i_encrypted == 0 )
    {
        _dvdcss_debug( dvdcss, "file was unscrambled" );
        return 0;
    }

    return -1;
}




/*****************************************************************************
 * CSSCryptKey : shuffles bits and unencrypt keys.
 *****************************************************************************
 * Used during authentication in CSSInit.
 * i_key_type : 0->key1, 1->key2, 2->buskey.
 * i_varient : between 0 and 31.
 *****************************************************************************/
/*static*/ void CSSCryptKey( int i_key_type, int i_varient,
                         u8 const * p_challenge, u8* p_key )
{
    /* Permutation table for challenge */
    u8      pp_perm_challenge[3][10] =
            { { 1, 3, 0, 7, 5, 2, 9, 6, 4, 8 },
              { 6, 1, 9, 3, 8, 5, 7, 4, 0, 2 },
              { 4, 0, 3, 5, 7, 2, 8, 6, 1, 9 } };

    /* Permutation table for varient table for key2 and buskey */
    u8      pp_perm_varient[2][32] =
            { { 0x0a, 0x08, 0x0e, 0x0c, 0x0b, 0x09, 0x0f, 0x0d,
                0x1a, 0x18, 0x1e, 0x1c, 0x1b, 0x19, 0x1f, 0x1d,
                0x02, 0x00, 0x06, 0x04, 0x03, 0x01, 0x07, 0x05,
                0x12, 0x10, 0x16, 0x14, 0x13, 0x11, 0x17, 0x15 },
              { 0x12, 0x1a, 0x16, 0x1e, 0x02, 0x0a, 0x06, 0x0e,
                0x10, 0x18, 0x14, 0x1c, 0x00, 0x08, 0x04, 0x0c,
                0x13, 0x1b, 0x17, 0x1f, 0x03, 0x0b, 0x07, 0x0f,
                0x11, 0x19, 0x15, 0x1d, 0x01, 0x09, 0x05, 0x0d } };

    u8      p_varients[32] =
            {   0xB7, 0x74, 0x85, 0xD0, 0xCC, 0xDB, 0xCA, 0x73,
                0x03, 0xFE, 0x31, 0x03, 0x52, 0xE0, 0xB7, 0x42,
                0x63, 0x16, 0xF2, 0x2A, 0x79, 0x52, 0xFF, 0x1B,
                0x7A, 0x11, 0xCA, 0x1A, 0x9B, 0x40, 0xAD, 0x01 };

    /* The "secret" key */
    u8      p_secret[5] = { 0x55, 0xD6, 0xC4, 0xC5, 0x28 };

    u8      p_bits[30];
    u8      p_scratch[10];
    u8      p_tmp1[5];
    u8      p_tmp2[5];
    u8      i_lfsr0_o;  /* 1 bit used */
    u8      i_lfsr1_o;  /* 1 bit used */
    u32     i_lfsr0;
    u32     i_lfsr1;
    u8      i_css_varient;
    u8      i_cse;
    u8      i_index;
    u8      i_combined;
    u8      i_carry;
    u8      i_val = 0;
    int     i_term = 0;
    int     i_bit;
    int     i;

    for (i = 9; i >= 0; --i)
        p_scratch[i] = p_challenge[pp_perm_challenge[i_key_type][i]];

    i_css_varient = ( i_key_type == 0 ) ? i_varient :
                    pp_perm_varient[i_key_type-1][i_varient];

    /*
     * This encryption engine implements one of 32 variations
     * one the same theme depending upon the choice in the
     * varient parameter (0 - 31).
     *
     * The algorithm itself manipulates a 40 bit input into
     * a 40 bit output.
     * The parameter 'input' is 80 bits.  It consists of
     * the 40 bit input value that is to be encrypted followed
     * by a 40 bit seed value for the pseudo random number
     * generators.
     */

    /* Feed the secret into the input values such that
     * we alter the seed to the LFSR's used above,  then
     * generate the bits to play with.
     */
    for( i = 5 ; --i >= 0 ; )
    {
        p_tmp1[i] = p_scratch[5 + i] ^ p_secret[i] ^ p_crypt_tab2[i];
    }

    /*
     * We use two LFSR's (seeded from some of the input data bytes) to
     * generate two streams of pseudo-random bits.  These two bit streams
     * are then combined by simply adding with carry to generate a final
     * sequence of pseudo-random bits which is stored in the buffer that
     * 'output' points to the end of - len is the size of this buffer.
     *
     * The first LFSR is of degree 25,  and has a polynomial of:
     * x^13 + x^5 + x^4 + x^1 + 1
     *
     * The second LSFR is of degree 17,  and has a (primitive) polynomial of:
     * x^15 + x^1 + 1
     *
     * I don't know if these polynomials are primitive modulo 2,  and thus
     * represent maximal-period LFSR's.
     *
     *
     * Note that we take the output of each LFSR from the new shifted in
     * bit,  not the old shifted out bit.  Thus for ease of use the LFSR's
     * are implemented in bit reversed order.
     *
     */
    
    /* In order to ensure that the LFSR works we need to ensure that the
     * initial values are non-zero.  Thus when we initialise them from
     * the seed,  we ensure that a bit is set.
     */
    i_lfsr0 = ( p_tmp1[0] << 17 ) | ( p_tmp1[1] << 9 ) |
              (( p_tmp1[2] & ~7 ) << 1 ) | 8 | ( p_tmp1[2] & 7 );
    i_lfsr1 = ( p_tmp1[3] << 9 ) | 0x100 | p_tmp1[4];

    i_index = sizeof(p_bits);
    i_carry = 0;

    do
    {
        for( i_bit = 0, i_val = 0 ; i_bit < 8 ; ++i_bit )
        {

            i_lfsr0_o = ( ( i_lfsr0 >> 24 ) ^ ( i_lfsr0 >> 21 ) ^
                        ( i_lfsr0 >> 20 ) ^ ( i_lfsr0 >> 12 ) ) & 1;
            i_lfsr0 = ( i_lfsr0 << 1 ) | i_lfsr0_o;

            i_lfsr1_o = ( ( i_lfsr1 >> 16 ) ^ ( i_lfsr1 >> 2 ) ) & 1;
            i_lfsr1 = ( i_lfsr1 << 1 ) | i_lfsr1_o;

            i_combined = !i_lfsr1_o + i_carry + !i_lfsr0_o;
            /* taking bit 1 */
            i_carry = ( i_combined >> 1 ) & 1;
            i_val |= ( i_combined & 1 ) << i_bit;
        }
    
        p_bits[--i_index] = i_val;
    } while( i_index > 0 );

    /* This term is used throughout the following to
     * select one of 32 different variations on the
     * algorithm.
     */
    i_cse = p_varients[i_css_varient] ^ p_crypt_tab2[i_css_varient];

    /* Now the actual blocks doing the encryption.  Each
     * of these works on 40 bits at a time and are quite
     * similar.
     */
    i_index = 0;
    for( i = 5, i_term = 0 ; --i >= 0 ; i_term = p_scratch[i] )
    {
        i_index = p_bits[25 + i] ^ p_scratch[i];
        i_index = p_crypt_tab1[i_index] ^ ~p_crypt_tab2[i_index] ^ i_cse;

        p_tmp1[i] = p_crypt_tab2[i_index] ^ p_crypt_tab3[i_index] ^ i_term;
    }
    p_tmp1[4] ^= p_tmp1[0];

    for( i = 5, i_term = 0 ; --i >= 0 ; i_term = p_tmp1[i] )
    {
        i_index = p_bits[20 + i] ^ p_tmp1[i];
        i_index = p_crypt_tab1[i_index] ^ ~p_crypt_tab2[i_index] ^ i_cse;

        p_tmp2[i] = p_crypt_tab2[i_index] ^ p_crypt_tab3[i_index] ^ i_term;
    }
    p_tmp2[4] ^= p_tmp2[0];

    for( i = 5, i_term = 0 ; --i >= 0 ; i_term = p_tmp2[i] )
    {
        i_index = p_bits[15 + i] ^ p_tmp2[i];
        i_index = p_crypt_tab1[i_index] ^ ~p_crypt_tab2[i_index] ^ i_cse;
        i_index = p_crypt_tab2[i_index] ^ p_crypt_tab3[i_index] ^ i_term;

        p_tmp1[i] = p_crypt_tab0[i_index] ^ p_crypt_tab2[i_index];
    }
    p_tmp1[4] ^= p_tmp1[0];

    for( i = 5, i_term = 0 ; --i >= 0 ; i_term = p_tmp1[i] )
    {
        i_index = p_bits[10 + i] ^ p_tmp1[i];
        i_index = p_crypt_tab1[i_index] ^ ~p_crypt_tab2[i_index] ^ i_cse;

        i_index = p_crypt_tab2[i_index] ^ p_crypt_tab3[i_index] ^ i_term;

        p_tmp2[i] = p_crypt_tab0[i_index] ^ p_crypt_tab2[i_index];
    }
    p_tmp2[4] ^= p_tmp2[0];

    for( i = 5, i_term = 0 ; --i >= 0 ; i_term = p_tmp2[i] )
    {
        i_index = p_bits[5 + i] ^ p_tmp2[i];
        i_index = p_crypt_tab1[i_index] ^ ~p_crypt_tab2[i_index] ^ i_cse;

        p_tmp1[i] = p_crypt_tab2[i_index] ^ p_crypt_tab3[i_index] ^ i_term;
    }
    p_tmp1[4] ^= p_tmp1[0];

    for(i = 5, i_term = 0 ; --i >= 0 ; i_term = p_tmp1[i] )
    {
        i_index = p_bits[i] ^ p_tmp1[i];
        i_index = p_crypt_tab1[i_index] ^ ~p_crypt_tab2[i_index] ^ i_cse;

        p_key[i] = p_crypt_tab2[i_index] ^ p_crypt_tab3[i_index] ^ i_term;
    }

    return;
}

/*****************************************************************************
 * CSSCracker : title key decryption by cracking
 *****************************************************************************
 * This function is called by CSSGetKeys to find a key
 *****************************************************************************/
static int CSSCracker( int i_start,
                       unsigned char * p_crypted,
                       unsigned char * p_decrypted,
                       u8* p_sector_seed,
                       u8* p_key )
{
    unsigned char p_buffer[10];
    unsigned int i_t1, i_t2, i_t3, i_t4, i_t5, i_t6;
    unsigned int i_try;
    unsigned int i_candidate;
    unsigned int i, j;
    int i_exit = -1;


    for( i = 0 ; i < 10 ; i++ )
    {
        p_buffer[i] = p_css_tab1[p_crypted[i]] ^ p_decrypted[i];
    }

    for( i_try = i_start ; i_try < 0x10000 ; i_try++ )
    {
        i_t1 = i_try >> 8 | 0x100;
        i_t2 = i_try & 0xff;
        i_t3 = 0;               /* not needed */
        i_t5 = 0;

        /* iterate cipher 4 times to reconstruct LFSR2 */
        for( i = 0 ; i < 4 ; i++ )
        {
            /* advance LFSR1 normaly */
            i_t4 = p_css_tab2[i_t2] ^ p_css_tab3[i_t1];
            i_t2 = i_t1 >> 1;
            i_t1 = ( ( i_t1 & 1 ) << 8 ) ^ i_t4;
            i_t4 = p_css_tab5[i_t4];
            /* deduce i_t6 & i_t5 */
            i_t6 = p_buffer[i];
            if( i_t5 )
            {
                i_t6 = ( i_t6 + 0xff ) & 0x0ff;
            }
            if( i_t6 < i_t4 )
            {
                i_t6 += 0x100;
            }
            i_t6 -= i_t4;
            i_t5 += i_t6 + i_t4;
            i_t6 = p_css_tab4[ i_t6 ];
            /* feed / advance i_t3 / i_t5 */
            i_t3 = ( i_t3 << 8 ) | i_t6;
            i_t5 >>= 8;
        }

        i_candidate = i_t3;

        /* iterate 6 more times to validate candidate key */
        for( ; i < 10 ; i++ )
        {
            i_t4 = p_css_tab2[i_t2] ^ p_css_tab3[i_t1];
            i_t2 = i_t1 >> 1;
            i_t1 = ( ( i_t1 & 1 ) << 8 ) ^ i_t4;
            i_t4 = p_css_tab5[i_t4];
            i_t6 = ((((((( i_t3 >> 3 ) ^ i_t3 ) >> 1 ) ^
                                         i_t3 ) >> 8 ) ^ i_t3 ) >> 5 ) & 0xff;
            i_t3 = ( i_t3 << 8 ) | i_t6;
            i_t6 = p_css_tab4[i_t6];
            i_t5 += i_t6 + i_t4;
            if( ( i_t5 & 0xff ) != p_buffer[i] )
            {
                break;
            }

            i_t5 >>= 8;
        }

        if( i == 10 )
        {
            /* Do 4 backwards steps of iterating t3 to deduce initial state */
            i_t3 = i_candidate;
            for( i = 0 ; i < 4 ; i++ )
            {
                i_t1 = i_t3 & 0xff;
                i_t3 = ( i_t3 >> 8 );
                /* easy to code, and fast enough bruteforce
                 * search for byte shifted in */
                for( j = 0 ; j < 256 ; j++ )
                {
                    i_t3 = ( i_t3 & 0x1ffff ) | ( j << 17 );
                    i_t6 = ((((((( i_t3 >> 3 ) ^ i_t3 ) >> 1 ) ^
                                   i_t3 ) >> 8 ) ^ i_t3 ) >> 5 ) & 0xff;
                    if( i_t6 == i_t1 )
                    {
                        break;
                    }
                }
            }

            i_t4 = ( i_t3 >> 1 ) - 4;
            for( i_t5 = 0 ; i_t5 < 8; i_t5++ )
            {
                if( ( ( i_t4 + i_t5 ) * 2 + 8 - ( (i_t4 + i_t5 ) & 7 ) )
                                                                      == i_t3 )
                {
                    p_key[0] = i_try>>8;
                    p_key[1] = i_try & 0xFF;
                    p_key[2] = ( ( i_t4 + i_t5 ) >> 0 ) & 0xFF;
                    p_key[3] = ( ( i_t4 + i_t5 ) >> 8 ) & 0xFF;
                    p_key[4] = ( ( i_t4 + i_t5 ) >> 16 ) & 0xFF;
                    i_exit = i_try + 1;
                }
            }
        }
    }

    if( i_exit >= 0 )
    {
        p_key[0] ^= p_sector_seed[0];
        p_key[1] ^= p_sector_seed[1];
        p_key[2] ^= p_sector_seed[2];
        p_key[3] ^= p_sector_seed[3];
        p_key[4] ^= p_sector_seed[4];
    }

    return i_exit;
}

/*****************************************************************************
 * CSSCryptKey : shuffles bits and deencrypt disc or title keys.
 *****************************************************************************
 * Used to deencrypt the disc key with a player key after requesting it 
 * in CSSGetDiscKey and to decrypt the title keys requested by CSSGetTitleKey
 * with the disc key.  The player key and the reslulting disc key are only
 * used as KEKs (key encryption keys).
 * i_key_type : 0->disc key, 1->title key.
 *****************************************************************************/
static void CSSDecryptKey( int i_key_type, u8* p_key, u8* p_in, u8* p_out)
{
    unsigned int t1, t2, t3, t4, t5, t6;
    unsigned char p_cipher[ 5 ];
    int i;

    t1 = p_key[0] | 0x100;
    t2 = p_key[1];
    t3 = p_key[4] << 16 | p_key[3] <<  8 | p_key[2];
    t4 = t3 & 7;
    t3 = ( t3 << 1 ) + 8 - t4;
    t5 = 0;
    for( i = 0; i < 5; i++ )
    {
	t4 = p_css_tab2[t2] ^ p_css_tab3[t1];
	t2 = t1 >> 1;
	t1 = ( ( t1 & 1 ) << 8 ) ^ t4;
	t4 = p_css_tab4[t4];
	t6 = ((((((( t3 >> 3 ) ^ t3) >> 1 ) ^ t3 ) >> 8) ^ t3 ) >> 5 ) & 0xff;
	t3 = ( t3 << 8 ) | t6;
	if( !i_key_type )
	  t6 = p_css_tab4[t6];
	else
	  t6 = p_css_tab5[t6];
	t5 += t6 + t4;
	p_cipher[i] = t5 & 0xff;
	t5 >>= 8;
    }
    
    /* Now we have the chiper stream, make to passes over the data */
    memcpy(p_out, p_in, 5);
    for( i = 9; i > 0; i-- )
    {
	p_out[i%5] = p_cipher[i%5] ^ p_css_tab1[p_out[i%5]] ^ p_out[(i - 1)%5];
    }
    p_out[0] = p_cipher[0] ^ p_css_tab1[p_out[0]];
}


