/*****************************************************************************
 * acss.c: Functions for DVD authentication and descrambling
 *****************************************************************************
 * Copyright (C) 1999-2001 VideoLAN
 * $Id: acss.c,v 1.5 2005/04/17 16:47:10 hshoexer Exp $
 *
 * Author: Stphane Borel <stef@via.ecp.fr>
 *         Hkan Hjort <d95hjort@dtek.chalmers.se>
 *
 * based on:
 *  - css-auth by Derek Fawcus <derek@spider.com>
 *  - DVD CSS ioctls example program by Andrew T. Veliath <andrewtv@usa.net>
 *  - The Divide and conquer attack by Frank A. Stevenson <frank@funcom.com>
 *  - DeCSSPlus by Ethan Hawke
 *  - DecVOB
 *  see http://www.lemuria.org/DeCSS/ by Tom Vogt for more information.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************/

/*****************************************************************************
 * Preamble
 *****************************************************************************/
#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <ssl/acss.h>
#include <rmd160.h>
#ifdef HAVE_SYS_PARAM_H
#   include <sys/param.h>
#endif
#ifdef HAVE_UNISTD_H
#   include <unistd.h>
#endif
#include <fcntl.h>

#ifdef HAVE_LIMITS_H
#   include <limits.h>
#endif

#include "libdvd/libdvd.h"

#include "common.h"
#include "acss.h"
#include "libdvd.h"
#include "scrambletables.h"
#include "ioctl.h"
#include "device.h"

/*****************************************************************************
 * Local prototypes
 *****************************************************************************/
static int  GetBusKey       ( libdvd_t );
static int  GetASF          ( libdvd_t );

static void CryptKey        ( libdvd_t, int, int, uint8_t const *, uint8_t * );
static void DecryptKey      ( uint8_t,
                              uint8_t const *, uint8_t const *, uint8_t * );

static int  DecryptDiscKey  ( libdvd_t, uint8_t const *, dvd_key_t );

static void DecryptTitleKey ( dvd_key_t, dvd_key_t );
static int  RecoverTitleKey ( int, uint8_t const *,
                              uint8_t const *, uint8_t const *, uint8_t * );
static int  CrackTitleKey   ( libdvd_t, int, int, dvd_key_t );

static int  AttackPattern   ( uint8_t const[], int, uint8_t * );
#if 0
static int  AttackPadding   ( uint8_t const[], int, uint8_t * );
#endif

/*****************************************************************************
 * _libdvd_test: check if the disc is encrypted or not
 *****************************************************************************/
int _libdvd_test( libdvd_t libdvd )
{
    int i_ret, i_copyright;

    i_ret = ioctl_ReadCopyright( libdvd->i_fd, 0 /* i_layer */, &i_copyright );

#ifdef WIN32
    if( i_ret < 0 )
    {
        /* Maybe we didn't have enough priviledges to read the copyright
         * (see ioctl_ReadCopyright comments).
         * Apparently, on unencrypted DVDs _libdvd_disckey() always fails, so
         * we can check this as a work-around. */
        i_ret = 0;
        if( _libdvd_disckey( libdvd ) < 0 )
            i_copyright = 0;
        else
            i_copyright = 1;
    }
#endif

    if( i_ret < 0 )
    {
        /* Since it's the first ioctl we try to issue, we add a notice */
        _libdvd_error( libdvd, "acss error: ioctl_ReadCopyright failed, "
                       "make sure there is a DVD in the drive, and that "
                       "you have used the correct device node." );

        return i_ret;
    }

    return i_copyright;
}

/*****************************************************************************
 * GetBusKey : Go through the CSS Authentication process
 *****************************************************************************
 * It simulates the mutual authentication between logical unit and host,
 * and stops when a session key (called bus key) has been established.
 * Always do the full auth sequence. Some drives seem to lie and always
 * respond with ASF=1.  For instance the old DVD roms on Compaq Armada says
 * that ASF=1 from the start and then later fail with a 'read of scrambled
 * block without authentication' error.
 *****************************************************************************/
static int GetBusKey( libdvd_t libdvd )
{
    uint8_t   p_buffer[10];
    uint8_t   p_challenge[2*KEY_SIZE];
    dvd_key_t p_key1;
    dvd_key_t p_key2;
    dvd_key_t p_key_check;
    uint8_t   i_variant = 0;
    char      psz_warning[80];
    int       i_ret = -1;
    int       i;

    _libdvd_debug( libdvd, "requesting AGID" );
    i_ret = ioctl_ReportAgid( libdvd->i_fd, &libdvd->scramble.i_agid );

    /* We might have to reset hung authentication processes in the drive
       by invalidating the corresponding AGID'.  As long as we haven't got
       an AGID, invalidate one (in sequence) and try again. */
    for( i = 0; i_ret == -1 && i < 4 ; ++i )
    {
        sprintf( psz_warning,
                 "ioctl ReportAgid failed, invalidating AGID %d", i );
        _libdvd_debug( libdvd, psz_warning );

        /* This is really _not good_, should be handled by the OS.
           Invalidating an AGID could make another process fail some
           where in it's authentication process. */
        libdvd->scramble.i_agid = i;
        ioctl_InvalidateAgid( libdvd->i_fd, &libdvd->scramble.i_agid );

        _libdvd_debug( libdvd, "requesting AGID" );
        i_ret = ioctl_ReportAgid( libdvd->i_fd, &libdvd->scramble.i_agid );
    }

    /* Unable to authenticate without AGID */
    if( i_ret == -1 )
    {
        _libdvd_error( libdvd, "ioctl ReportAgid failed, fatal" );
        return -1;
    }

    /* Setup a challenge, any values should work */
    for( i = 0 ; i < 10; ++i )
    {
        p_challenge[i] = i;
    }

    /* Get challenge from host */
    for( i = 0 ; i < 10 ; ++i )
    {
        p_buffer[9-i] = p_challenge[i];
    }

    /* Send challenge to LU */
    if( ioctl_SendChallenge( libdvd->i_fd,
                             &libdvd->scramble.i_agid, p_buffer ) < 0 )
    {
        _libdvd_error( libdvd, "ioctl SendChallenge failed" );
        ioctl_InvalidateAgid( libdvd->i_fd, &libdvd->scramble.i_agid );
        return -1;
    }

    /* Get key1 from LU */
    if( ioctl_ReportKey1( libdvd->i_fd, &libdvd->scramble.i_agid, p_buffer ) < 0)
    {
        _libdvd_error( libdvd, "ioctl ReportKey1 failed" );
        ioctl_InvalidateAgid( libdvd->i_fd, &libdvd->scramble.i_agid );
        return -1;
    }

    /* Send key1 to host */
    for( i = 0 ; i < KEY_SIZE ; i++ )
    {
        p_key1[i] = p_buffer[4-i];
    }

    for( i = 0 ; i < 32 ; ++i )
    {
        CryptKey( libdvd, 0, i, p_challenge, p_key_check );

        if( memcmp( p_key_check, p_key1, KEY_SIZE ) == 0 )
        {
            snprintf( psz_warning, sizeof(psz_warning),
                      "drive authenticated, using variant %d", i );
            _libdvd_debug( libdvd, psz_warning );
            i_variant = i;
            break;
        }
    }

    if( i == 32 )
    {
        _libdvd_error( libdvd, "drive would not authenticate" );
        ioctl_InvalidateAgid( libdvd->i_fd, &libdvd->scramble.i_agid );
        return -1;
    }

    /* Get challenge from LU */
    if( ioctl_ReportChallenge( libdvd->i_fd,
                               &libdvd->scramble.i_agid, p_buffer ) < 0 )
    {
        _libdvd_error( libdvd, "ioctl ReportKeyChallenge failed" );
        ioctl_InvalidateAgid( libdvd->i_fd, &libdvd->scramble.i_agid );
        return -1;
    }

    /* Send challenge to host */
    for( i = 0 ; i < 10 ; ++i )
    {
        p_challenge[i] = p_buffer[9-i];
    }

    CryptKey( libdvd, 1, i_variant, p_challenge, p_key2 );

    /* Get key2 from host */
    for( i = 0 ; i < KEY_SIZE ; ++i )
    {
        p_buffer[4-i] = p_key2[i];
    }

    /* Send key2 to LU */
    if( ioctl_SendKey2( libdvd->i_fd, &libdvd->scramble.i_agid, p_buffer ) < 0 )
    {
        _libdvd_error( libdvd, "ioctl SendKey2 failed" );
        ioctl_InvalidateAgid( libdvd->i_fd, &libdvd->scramble.i_agid );
        return -1;
    }

    /* The drive has accepted us as authentic. */
    _libdvd_debug( libdvd, "authentication established" );

    memcpy( p_challenge, p_key1, KEY_SIZE );
    memcpy( p_challenge + KEY_SIZE, p_key2, KEY_SIZE );

    CryptKey( libdvd, 2, i_variant, p_challenge, libdvd->scramble.p_bus_key );

    return 0;
}

/*****************************************************************************
 * PrintKey : debug function that dumps a key value
 *****************************************************************************/
static void PrintKey( libdvd_t libdvd, char *prefix, uint8_t const *data )
{
    char psz_output[80];

    sprintf( psz_output, "%s%02x:%02x:%02x:%02x:%02x", prefix,
             data[0], data[1], data[2], data[3], data[4] );
    _libdvd_debug( libdvd, psz_output );
}

/*****************************************************************************
 * _libdvd_title: crack or decrypt the current title key if needed
 *****************************************************************************
 * This function should only be called by libdvd->pf_seek and should eventually
 * not be external if possible.
 *****************************************************************************/
int _libdvd_title ( libdvd_t libdvd, int i_block )
{
    dvd_title_t *p_title;
    dvd_title_t *p_newtitle;
    dvd_key_t    p_title_key;
    int          i_fd, i_ret = -1, b_cache = 0;

    if( ! libdvd->b_scrambled )
    {
        return 0;
    }

    /* Check if we've already cracked this key */
    p_title = libdvd->p_titles;
    while( p_title != NULL
            && p_title->p_next != NULL
            && p_title->p_next->i_startlb <= i_block )
    {
        p_title = p_title->p_next;
    }

    if( p_title != NULL
         && p_title->i_startlb == i_block )
    {
        /* We've already cracked this key, nothing to do */
        memcpy( libdvd->scramble.p_title_key, p_title->p_key, sizeof(dvd_key_t) );
        return 0;
    }

    /* Check whether the key is in our disk cache */
    if( libdvd->psz_cachefile[0] )
    {
        /* XXX: be careful, we use sprintf and not snprintf */
        sprintf( libdvd->psz_block, "%.10x", i_block );
        i_fd = open( libdvd->psz_cachefile, O_RDONLY );
        b_cache = 1;

        if( i_fd >= 0 )
        {
            if( read( i_fd, p_title_key, 5 ) == 5 )
            {
                _libdvd_debug( libdvd, "key found in cache" );
                /* Don't try to save it again */
                b_cache = 0;
                i_ret = 1;
            }
            close( i_fd );
        }
    }

    /* Crack or decrypt CSS title key for current VTS */
    if( i_ret < 0 )
    {
        i_ret = _libdvd_titlekey( libdvd, i_block, p_title_key );

        if( i_ret < 0 )
        {
            _libdvd_error( libdvd, "fatal error in vts scramble key" );
            return i_ret;
        }

        if( i_ret == 0 )
        {
            _libdvd_debug( libdvd, "unencrypted title" );
            /* We cache this anyway, so we don't need to check again. */
        }
    }

    /* Key is valid, we store it on disk. */
    if( b_cache )
    {
        i_fd = open( libdvd->psz_cachefile, O_RDWR|O_CREAT|O_EXCL, 0644 );
        if( i_fd >= 0 )
        {
            write( i_fd, p_title_key, 5 );
            close( i_fd );
        }
    }

    /* Find our spot in the list */
    p_newtitle = NULL;
    p_title = libdvd->p_titles;
    while( ( p_title != NULL ) && ( p_title->i_startlb < i_block ) )
    {
        p_newtitle = p_title;
        p_title = p_title->p_next;
    }

    /* Save the found title */
    p_title = p_newtitle;

    /* Write in the new title and its key */
    p_newtitle = malloc( sizeof( dvd_title_t ) );
    p_newtitle->i_startlb = i_block;
    memcpy( p_newtitle->p_key, p_title_key, KEY_SIZE );

    /* Link it at the head of the (possibly empty) list */
    if( p_title == NULL )
    {
        p_newtitle->p_next = libdvd->p_titles;
        libdvd->p_titles = p_newtitle;
    }
    /* Link the new title inside the list */
    else
    {
        p_newtitle->p_next = p_title->p_next;
        p_title->p_next = p_newtitle;
    }

    memcpy( libdvd->scramble.p_title_key, p_title_key, KEY_SIZE );
    return 0;
}

/*****************************************************************************
 * _libdvd_disckey: get disc key.
 *****************************************************************************
 * This function should only be called if DVD ioctls are present.
 * It will set libdvd->i_method = LIBDVD_METHOD_TITLE if it fails to find
 * a valid disc key.
 * Two decryption methods are offered:
 *  -disc key hash crack,
 *  -decryption with player keys if they are available.
 *****************************************************************************/
int _libdvd_disckey( libdvd_t libdvd )
{
    unsigned char p_buffer[ DVD_DISCKEY_SIZE ];
    dvd_key_t p_disc_key;
    int i;

    if( GetBusKey( libdvd ) < 0 )
    {
        return -1;
    }

    /* Get encrypted disc key */
    if( ioctl_ReadDiscKey( libdvd->i_fd, &libdvd->scramble.i_agid, p_buffer ) < 0 )
    {
        _libdvd_error( libdvd, "ioctl ReadDiscKey failed" );
        return -1;
    }

    /* This should have invaidated the AGID and got us ASF=1. */
    if( GetASF( libdvd ) != 1 )
    {
        /* Region mismatch (or region not set) is the most likely source. */
        _libdvd_error( libdvd,
                       "ASF not 1 after reading disc key (region mismatch?)" );
        ioctl_InvalidateAgid( libdvd->i_fd, &libdvd->scramble.i_agid );
        return -1;
    }

    /* Decrypt disc key using bus key */
    for( i = 0 ; i < DVD_DISCKEY_SIZE ; i++ )
    {
        p_buffer[ i ] ^= libdvd->scramble.p_bus_key[ 4 - (i % KEY_SIZE) ];
    }

    switch( libdvd->i_method )
    {
        case LIBDVD_METHOD_KEY:

            /* Decrypt disc key with player key. */
            _libdvd_debug( libdvd, "decrypting disc key with player keys" );
            if( ! DecryptDiscKey( libdvd, p_buffer, p_disc_key ) )
            {
                PrintKey( libdvd, "decrypted disc key is ", p_disc_key );
                break;
            }
            _libdvd_debug( libdvd, "failed to decrypt the disc key, "
                                   "faulty drive/kernel? "
                                   "cracking title keys instead" );

            /* Fallback, but not to DISC as the disc key might be faulty */
            libdvd->i_method = LIBDVD_METHOD_TITLE;
            break;

#if 0
        case LIBDVD_METHOD_DISC:

            /* Crack Disc key to be able to use it */
            _libdvd_debug( libdvd, "cracking disc key from key hash ..."
                                   " this will take some time" );
            memcpy( p_disc_key, p_buffer, KEY_SIZE );
            if( ! CrackDiscKey( libdvd, p_disc_key ) )
            {
                PrintKey( libdvd, "cracked disc key is ", p_disc_key );
                break;
            }
            _libdvd_debug( libdvd, "failed to crack the disc key" );
            memset( p_disc_key, 0, KEY_SIZE );
            libdvd->i_method = LIBDVD_METHOD_TITLE;
            break;
#endif

        default:

            _libdvd_debug( libdvd, "disc key needs not be decrypted" );
            memset( p_disc_key, 0, KEY_SIZE );
            break;
    }

    memcpy( libdvd->scramble.p_disc_key, p_disc_key, KEY_SIZE );

    return 0;
}


/*****************************************************************************
 * _libdvd_titlekey: get title key.
 *****************************************************************************/
int _libdvd_titlekey( libdvd_t libdvd, int i_pos, dvd_key_t p_title_key )
{
    static uint8_t p_garbage[ LIBDVD_BLOCK_SIZE ];  /* we never read it back */
    uint8_t p_key[ KEY_SIZE ];
    int i, i_ret = 0;

    if( libdvd->b_ioctls && ( libdvd->i_method == LIBDVD_METHOD_KEY /*||
                              libdvd->i_method == LIBDVD_METHOD_DISC*/ ) )
    {
        /* We have a decrypted Disc key and the ioctls are available,
         * read the title key and decrypt it.
         */

        _libdvd_debug( libdvd, "getting title key the classic way" );

        /* We need to authenticate again every time to get a new session key */
        if( GetBusKey( libdvd ) < 0 )
        {
            return -1;
        }

        /* Get encrypted title key */
        if( ioctl_ReadTitleKey( libdvd->i_fd, &libdvd->scramble.i_agid,
                                i_pos, p_key ) < 0 )
        {
            _libdvd_debug( libdvd,
                           "ioctl ReadTitleKey failed (region mismatch?)" );
            i_ret = -1;
        }

        /* Test ASF, it will be reset to 0 if we got a Region error */
        switch( GetASF( libdvd ) )
        {
            case -1:
                /* An error getting the ASF status, something must be wrong. */
                _libdvd_debug( libdvd, "lost ASF requesting title key" );
                ioctl_InvalidateAgid( libdvd->i_fd, &libdvd->scramble.i_agid );
                i_ret = -1;
                break;

            case 0:
                /* This might either be a title that has no key,
                 * or we encountered a region error. */
                _libdvd_debug( libdvd, "lost ASF requesting title key" );
                break;

            case 1:
                /* Drive status is ok. */
                /* If the title key request failed, but we did not loose ASF,
                 * we might stil have the AGID.  Other code assume that we
                 * will not after this so invalidate it(?). */
                if( i_ret < 0 )
                {
                    ioctl_InvalidateAgid( libdvd->i_fd, &libdvd->scramble.i_agid );
                }
                break;
        }

        if( !( i_ret < 0 ) )
        {
            /* Decrypt title key using the bus key */
            for( i = 0 ; i < KEY_SIZE ; i++ )
            {
                p_key[ i ] ^= libdvd->scramble.p_bus_key[ 4 - (i % KEY_SIZE) ];
            }

            /* If p_key is all zero then there really wasn't any key present
             * even though we got to read it without an error. */
            if( !( p_key[0] | p_key[1] | p_key[2] | p_key[3] | p_key[4] ) )
            {
                i_ret = 0;
            }
            else
            {
                DecryptTitleKey( libdvd->scramble.p_disc_key, p_key );
                i_ret = 1;
            }

            /* All went well either there wasn't a key or we have it now. */
            memcpy( p_title_key, p_key, KEY_SIZE );
            PrintKey( libdvd, "title key is ", p_title_key );

            return i_ret;
        }

        /* The title key request failed */
        _libdvd_debug( libdvd, "resetting drive and cracking title key" );

        /* Read an unscrambled sector and reset the drive */
        libdvd->pf_seek( libdvd, 0 );
        libdvd->pf_read( libdvd, p_garbage, 1 );
        libdvd->pf_seek( libdvd, 0 );
        _libdvd_disckey( libdvd );

        /* Fallback */
    }

    /* METHOD is TITLE, we can't use the ioctls or requesting the title key
     * failed above.  For these cases we try to crack the key instead. */

    /* For now, the read limit is 9Gb / 2048 =  4718592 sectors. */
    i_ret = CrackTitleKey( libdvd, i_pos, 4718592, p_key );

    memcpy( p_title_key, p_key, KEY_SIZE );
    PrintKey( libdvd, "title key is ", p_title_key );

    return i_ret;
}

/*****************************************************************************
 * _libdvd_unscramble: does the actual descrambling of data
 *****************************************************************************
 * sec : sector to unscramble
 * key : title key for this sector
 *****************************************************************************/
int _libdvd_unscramble( dvd_key_t p_key, uint8_t *p_sec )
{
    unsigned char keydata[5];
    int i;
    ACSS_KEY akey;

    /* PES_scrambling_control */
    if( p_sec[0x14] & 0x30)
    {
        for (i = 0; i < ACSS_KEYSIZE; i++)
		keydata[i] = p_key[i] ^ p_sec[0x54 + i];
	
	p_sec += 0x80;
	acss_setkey(&akey, keydata, 0, ACSS_MODE1);
	acss(&akey, 1920, p_sec, p_sec);
    }

    return 0;
}

/* Following functions are local */

/*****************************************************************************
 * GetASF : Get Authentication success flag
 *****************************************************************************
 * Returns :
 *  -1 on ioctl error,
 *  0 if the device needs to be authenticated,
 *  1 either.
 *****************************************************************************/
static int GetASF( libdvd_t libdvd )
{
    int i_asf = 0;

    if( ioctl_ReportASF( libdvd->i_fd, NULL, &i_asf ) != 0 )
    {
        /* The ioctl process has failed */
        _libdvd_error( libdvd, "GetASF fatal error" );
        return -1;
    }

    if( i_asf )
    {
        _libdvd_debug( libdvd, "GetASF authenticated, ASF=1" );
    }
    else
    {
        _libdvd_debug( libdvd, "GetASF not authenticated, ASF=0" );
    }

    return i_asf;
}

/* Produce a 5 byte value from some salt and user provided input. */
static void genseed(libdvd_t libdvd, uint8_t *value, int offset)
{
    int i;
    unsigned char digest[20];
    RMD160_CTX ctx;

    if (sizeof(salts) < (unsigned int)((offset & 0x3f) * 2 + 1))
	return;

    RMD160Init(&ctx);
    RMD160Update(&ctx, libdvd->psz_challenge, strlen(libdvd->psz_challenge));
    RMD160Update(&ctx, (unsigned char*)&salts[offset * 2], sizeof(u_int32_t));
    RMD160Final(digest, &ctx);

    for (i = 0; i < 3; i++)
	value[4 - i] = digest[i];

    RMD160Init(&ctx);
    RMD160Update(&ctx, libdvd->psz_challenge, strlen(libdvd->psz_challenge));
    RMD160Update(&ctx, (unsigned char*)&salts[offset * 2 + 1], sizeof(u_int32_t));
    RMD160Final(digest, &ctx);
	
    for (i = 0; i < 2; i++)
	value[1 - i] = digest[i];

    bzero(digest, sizeof digest);
    
    return;

}

/*****************************************************************************
 * CryptKey : shuffles bits and unencrypt keys.
 *****************************************************************************
 * Used during authentication and disc key negociation in GetBusKey.
 * i_key_type : 0->key1, 1->key2, 2->buskey.
 * i_variant : between 0 and 31.
 *****************************************************************************/
static void CryptKey( libdvd_t libdvd, int i_key_type, int i_variant,
                      uint8_t const *p_challenge, uint8_t *p_key )
{
    /* Permutation table for challenge */
    uint8_t pp_perm_challenge[3][10] =
            { { 1, 3, 0, 7, 5, 2, 9, 6, 4, 8 },
              { 6, 1, 9, 3, 8, 5, 7, 4, 0, 2 },
              { 4, 0, 3, 5, 7, 2, 8, 6, 1, 9 } };

    /* Permutation table for variant table for key2 and buskey */
    uint8_t pp_perm_variant[2][32] =
            { { 0x0a, 0x08, 0x0e, 0x0c, 0x0b, 0x09, 0x0f, 0x0d,
                0x1a, 0x18, 0x1e, 0x1c, 0x1b, 0x19, 0x1f, 0x1d,
                0x02, 0x00, 0x06, 0x04, 0x03, 0x01, 0x07, 0x05,
                0x12, 0x10, 0x16, 0x14, 0x13, 0x11, 0x17, 0x15 },
              { 0x12, 0x1a, 0x16, 0x1e, 0x02, 0x0a, 0x06, 0x0e,
                0x10, 0x18, 0x14, 0x1c, 0x00, 0x08, 0x04, 0x0c,
                0x13, 0x1b, 0x17, 0x1f, 0x03, 0x0b, 0x07, 0x0f,
                0x11, 0x19, 0x15, 0x1d, 0x01, 0x09, 0x05, 0x0d } };

    uint8_t p_variants[32] =
            {   0xB7, 0x74, 0x85, 0xD0, 0xCC, 0xDB, 0xCA, 0x73,
                0x03, 0xFE, 0x31, 0x03, 0x52, 0xE0, 0xB7, 0x42,
                0x63, 0x16, 0xF2, 0x2A, 0x79, 0x52, 0xFF, 0x1B,
                0x7A, 0x11, 0xCA, 0x1A, 0x9B, 0x40, 0xAD, 0x01 };

    uint8_t p_sseed[KEY_SIZE];

    uint8_t p_bits[30], p_scratch[10], p_tmp1[5], p_tmp2[5];
    uint8_t i_scramble_variant, i_cse, i_index;
    int i_term = 0;
    int i;

    unsigned char keydata[5];
    unsigned char bits[30];
    ACSS_KEY akey;

    for (i = 9; i >= 0; --i)
        p_scratch[i] = p_challenge[pp_perm_challenge[i_key_type][i]];

    i_scramble_variant = ( i_key_type == 0 ) ? i_variant :
                    pp_perm_variant[i_key_type-1][i_variant];

    /*
     * This encryption engine implements one of 32 variations
     * one the same theme depending upon the choice in the
     * variant parameter (0 - 31).
     *
     * The algorithm itself manipulates a 40 bit input into
     * a 40 bit output.
     * The parameter 'input' is 80 bits.  It consists of
     * the 40 bit input value that is to be encrypted followed
     * by a 40 bit seed value for the pseudo random number
     * generators.
     */

    genseed(libdvd, p_sseed, 31);

    /* Feed an additional seed into the input values such that
     * we alter the seed to the LFSR's used above,  then
     * generate the bits to play with.
     */
    for( i = 5 ; --i >= 0 ; )
    {
        p_tmp1[i] = p_scratch[5 + i] ^ p_sseed[i] ^ p_crypt_tab2[i];
    }

    for (i = 0; i < 5; i++) {
	    keydata[4 - i] = p_tmp1[i];
    }

    acss_setkey(&akey, keydata, 0, ACSS_MODE3);
    acss(&akey, 30, (unsigned char *)0, bits);

    for (i = 0; i < 30; i++)
	    p_bits[29 - i] = bits[i];

    /* This term is used throughout the following to
     * select one of 32 different variations on the
     * algorithm.
     */
    i_cse = p_variants[i_scramble_variant] ^ p_crypt_tab2[i_scramble_variant];

    /* Now the actual blocks doing the encryption.  Each
     * of these works on 40 bits at a time and are quite
     * similar.
     */
    i_index = 0;
    for( i = 5, i_term = 0 ; --i >= 0 ; i_term = p_scratch[i] )
    {
        i_index = p_bits[25 + i] ^ p_scratch[i];
        i_index = p_crypt_tab1[i_index] ^ ~p_crypt_tab2[i_index] ^ i_cse;

        p_tmp1[i] = p_crypt_tab2[i_index] ^ p_crypt_tab3[i_index] ^ i_term;
    }
    p_tmp1[4] ^= p_tmp1[0];

    for( i = 5, i_term = 0 ; --i >= 0 ; i_term = p_tmp1[i] )
    {
        i_index = p_bits[20 + i] ^ p_tmp1[i];
        i_index = p_crypt_tab1[i_index] ^ ~p_crypt_tab2[i_index] ^ i_cse;

        p_tmp2[i] = p_crypt_tab2[i_index] ^ p_crypt_tab3[i_index] ^ i_term;
    }
    p_tmp2[4] ^= p_tmp2[0];

    for( i = 5, i_term = 0 ; --i >= 0 ; i_term = p_tmp2[i] )
    {
        i_index = p_bits[15 + i] ^ p_tmp2[i];
        i_index = p_crypt_tab1[i_index] ^ ~p_crypt_tab2[i_index] ^ i_cse;
        i_index = p_crypt_tab2[i_index] ^ p_crypt_tab3[i_index] ^ i_term;

        p_tmp1[i] = p_crypt_tab0[i_index] ^ p_crypt_tab2[i_index];
    }
    p_tmp1[4] ^= p_tmp1[0];

    for( i = 5, i_term = 0 ; --i >= 0 ; i_term = p_tmp1[i] )
    {
        i_index = p_bits[10 + i] ^ p_tmp1[i];
        i_index = p_crypt_tab1[i_index] ^ ~p_crypt_tab2[i_index] ^ i_cse;

        i_index = p_crypt_tab2[i_index] ^ p_crypt_tab3[i_index] ^ i_term;

        p_tmp2[i] = p_crypt_tab0[i_index] ^ p_crypt_tab2[i_index];
    }
    p_tmp2[4] ^= p_tmp2[0];

    for( i = 5, i_term = 0 ; --i >= 0 ; i_term = p_tmp2[i] )
    {
        i_index = p_bits[5 + i] ^ p_tmp2[i];
        i_index = p_crypt_tab1[i_index] ^ ~p_crypt_tab2[i_index] ^ i_cse;

        p_tmp1[i] = p_crypt_tab2[i_index] ^ p_crypt_tab3[i_index] ^ i_term;
    }
    p_tmp1[4] ^= p_tmp1[0];

    for(i = 5, i_term = 0 ; --i >= 0 ; i_term = p_tmp1[i] )
    {
        i_index = p_bits[i] ^ p_tmp1[i];
        i_index = p_crypt_tab1[i_index] ^ ~p_crypt_tab2[i_index] ^ i_cse;

        p_key[i] = p_crypt_tab2[i_index] ^ p_crypt_tab3[i_index] ^ i_term;
    }

    return;
}

/*****************************************************************************
 * DecryptKey: decrypt p_crypted with p_key.
 *****************************************************************************
 * Used to decrypt the disc key, with a player key, after requesting it
 * in _libdvd_disckey and to decrypt title keys, with a disc key, requested
 * in _libdvd_titlekey.
 * The player keys and the resulting disc key are only used as KEKs
 * (key encryption keys).
 * Decryption is slightly dependant on the type of key:
 *  -for disc key, invert is 0x00,
 *  -for title key, invert if 0xff.
 *****************************************************************************/
static void DecryptKey( uint8_t invert, uint8_t const *p_key,
                        uint8_t const *p_crypted, uint8_t *p_result )
{
    uint8_t         k[5];

    ACSS_KEY akey;

    if (invert)
        acss_setkey(&akey, p_key, 0, ACSS_MODE2);
    else
        acss_setkey(&akey, p_key, 0, ACSS_MODE0);

    acss(&akey, KEY_SIZE, 0, k);

    p_result[4] = k[4] ^ p_scramble_tab1[p_crypted[4]] ^ p_crypted[3];
    p_result[3] = k[3] ^ p_scramble_tab1[p_crypted[3]] ^ p_crypted[2];
    p_result[2] = k[2] ^ p_scramble_tab1[p_crypted[2]] ^ p_crypted[1];
    p_result[1] = k[1] ^ p_scramble_tab1[p_crypted[1]] ^ p_crypted[0];
    p_result[0] = k[0] ^ p_scramble_tab1[p_crypted[0]] ^ p_result[4];

    p_result[4] = k[4] ^ p_scramble_tab1[p_result[4]] ^ p_result[3];
    p_result[3] = k[3] ^ p_scramble_tab1[p_result[3]] ^ p_result[2];
    p_result[2] = k[2] ^ p_scramble_tab1[p_result[2]] ^ p_result[1];
    p_result[1] = k[1] ^ p_scramble_tab1[p_result[1]] ^ p_result[0];
    p_result[0] = k[0] ^ p_scramble_tab1[p_result[0]];

    return;
}

/*****************************************************************************
 * DecryptDiscKey
 *****************************************************************************
 * Decryption of the disc key with player keys if they are available.
 * Try to decrypt the disc key from every position with every player key.
 * p_struct_disckey: the 2048 byte DVD_STRUCT_DISCKEY data
 * p_disc_key: result, the 5 byte disc key
 *****************************************************************************/
static int DecryptDiscKey( libdvd_t libdvd, uint8_t const *p_struct_disckey,
                           dvd_key_t p_disc_key )
{
    uint8_t p_verify[KEY_SIZE];
    unsigned int i, n = 0;
    uint8_t p_sseed[KEY_SIZE];

    /* Decrypt disc key with the above player keys */
    while( n < 31)
    {
        genseed(libdvd, p_sseed, n);

        for( i = 1; i < 409; i++ )
        {
            /* Check if player key n is the right key for position i. */
            DecryptKey( 0, p_sseed, p_struct_disckey + 5 * i,
                        p_disc_key );

            /* The first part in the struct_disckey block is the
             * 'disc key' encrypted with itself.  Using this we
             * can check if we decrypted the correct key. */
            DecryptKey( 0, p_disc_key, p_struct_disckey, p_verify );

            /* If the position / player key pair worked then return. */
            if( memcmp( p_disc_key, p_verify, KEY_SIZE ) == 0 )
            {
                return 0;
            }
        }
        n++;
    }

    /* Have tried all combinations of positions and keys,
     * and we still didn't succeed. */
    memset( p_disc_key, 0, KEY_SIZE );
    return -1;
}

/*****************************************************************************
 * DecryptTitleKey
 *****************************************************************************
 * Decrypt the title key using the disc key.
 * p_disc_key: result, the 5 byte disc key
 * p_titlekey: the encrypted title key, gets overwritten by the decrypted key
 *****************************************************************************/
static void DecryptTitleKey( dvd_key_t p_disc_key, dvd_key_t p_titlekey )
{
    DecryptKey( 0xff, p_disc_key, p_titlekey, p_titlekey );
}

/*****************************************************************************
 * RecoverTitleKey: (title) key recovery from cipher and plain text
 * Function designed by Frank Stevenson
 *****************************************************************************
 * Called from Attack* which are in turn called by CrackTitleKey.  Given
 * a guessed(?) plain text and the cipher text.  Returns -1 on failure.
 *****************************************************************************/
static int RecoverTitleKey( int i_start, uint8_t const *p_crypted,
                            uint8_t const *p_decrypted,
                            uint8_t const *p_sector_seed, uint8_t *p_key )
{
    uint8_t p_buffer[10];
    unsigned int i_t1, i_t2, i_t3, i_t4, i_t5, i_t6;
    unsigned int i_try;
    unsigned int i_candidate;
    unsigned int i, j;
    ACSS_KEY akey;
    unsigned char stream;

    int i_exit = -1;

    /* Derive sum of LFSRs from plaintext and ciphertext bytes. */
    for( i = 0 ; i < 10 ; i++ )
    {
        p_buffer[i] = p_scramble_tab1[p_crypted[i]] ^ p_decrypted[i];
    }

    /*
     * Plaintext is at the start of the decrypted section, therefore
     * searchspace is only 16 bit, as carry bit is zero and bit 8 is 1.
     */
    for( i_try = i_start ; i_try < 0x10000 ; i_try++ )
    {
        i_t1 = i_try >> 8 | 0x100;
        i_t2 = i_try & 0xff;
        i_t3 = 0;               /* not needed */
        i_t5 = 0;

        /*
	 * Clock LFSR1 4 times and reconstruct LFSR2 from state of LFSR1 and
	 * the LFSR sum.
	 */
        for( i = 0 ; i < 4 ; i++ )
        {
            /* advance LFSR1 normaly */
            i_t4 = p_scramble_tab2[i_t2] ^ p_scramble_tab3[i_t1];
            i_t2 = i_t1 >> 1;
            i_t1 = ( ( i_t1 & 1 ) << 8 ) ^ i_t4;
            i_t4 = p_scramble_tab5[i_t4];
            /* deduce i_t6 & i_t5 */
            i_t6 = p_buffer[i];
            if( i_t5 )
            {
                i_t6 = ( i_t6 + 0xff ) & 0x0ff;
            }
            if( i_t6 < i_t4 )
            {
                i_t6 += 0x100;
            }
            i_t6 -= i_t4;
            i_t5 += i_t6 + i_t4;
            i_t6 = p_scramble_tab4[ i_t6 ];
            /* feed / advance i_t3 / i_t5 */
            i_t3 = ( i_t3 << 8 ) | i_t6;
            i_t5 >>= 8;
        }

        i_candidate = i_t3;

        /* iterate 6 more times to validate candidate key */
	bzero(&akey, sizeof akey);
	akey.lfsr17 = (p_scramble_tab4[i_t1 & 0xff] << 9)
		| (i_t1 & 0x100)
		| p_scramble_tab4[i_t2 & 0xff];
	akey.lfsr25 = (p_scramble_tab4[i_t3 & 0xff] << 17)
		| (p_scramble_tab4[(i_t3 >> 8) & 0xff] << 9)
		| (p_scramble_tab4[(i_t3 >> 16) & 0xff] << 1)
		| ((i_t3 & 0x1000000) >> 24);
	akey.lfsrsum = i_t5 << 8;	/* carry */
	akey.mode = ACSS_MODE1;

	for (; i < 10; i++) {
		acss(&akey, 1, NULL, &stream);
		if (stream != p_buffer[i])
			break;
	}

        if( i == 10 )
        {
            /* Do 4 backwards steps of iterating t3 to deduce initial state */
            i_t3 = i_candidate;
            for( i = 0 ; i < 4 ; i++ )
            {
                i_t1 = i_t3 & 0xff;
                i_t3 = ( i_t3 >> 8 );
                /* easy to code, and fast enough bruteforce
                 * search for byte shifted in */
                for( j = 0 ; j < 256 ; j++ )
                {
                    i_t3 = ( i_t3 & 0x1ffff ) | ( j << 17 );
                    i_t6 = ((((((( i_t3 >> 3 ) ^ i_t3 ) >> 1 ) ^
                                   i_t3 ) >> 8 ) ^ i_t3 ) >> 5 ) & 0xff;
                    if( i_t6 == i_t1 )
                    {
                        break;
                    }
                }
            }

            i_t4 = ( i_t3 >> 1 ) - 4;
            for( i_t5 = 0 ; i_t5 < 8; i_t5++ )
            {
                if( ( ( i_t4 + i_t5 ) * 2 + 8 - ( (i_t4 + i_t5 ) & 7 ) )
                                                                      == i_t3 )
                {
                    p_key[0] = i_try>>8;
                    p_key[1] = i_try & 0xFF;
                    p_key[2] = ( ( i_t4 + i_t5 ) >> 0 ) & 0xFF;
                    p_key[3] = ( ( i_t4 + i_t5 ) >> 8 ) & 0xFF;
                    p_key[4] = ( ( i_t4 + i_t5 ) >> 16 ) & 0xFF;
                    i_exit = i_try + 1;
                }
            }
        }
    }

    if( i_exit >= 0 )
    {
        p_key[0] ^= p_sector_seed[0];
        p_key[1] ^= p_sector_seed[1];
        p_key[2] ^= p_sector_seed[2];
        p_key[3] ^= p_sector_seed[3];
        p_key[4] ^= p_sector_seed[4];
    }

    return i_exit;
}

/******************************************************************************
 * Various pices for the title crack engine.
 ******************************************************************************
 * The length of the PES packet is located at 0x12-0x13.
 * The the copyrigth protection bits are located at 0x14 (bits 0x20 and 0x10).
 * The data of the PES packet begins at 0x15 (if there isn't any PTS/DTS)
 * or at 0x?? if there are both PTS and DTS's.
 * The seed value used with the unscrambling key is the 5 bytes at 0x54-0x58.
 * The scrabled part of a sector begins at 0x80.
 *****************************************************************************/

/* Statistics */
static int i_tries = 0, i_success = 0;

/*****************************************************************************
 * CrackTitleKey: try to crack title key from the contents of a VOB.
 *****************************************************************************
 * This function is called by _libdvd_titlekey to find a title key, if we've
 * chosen to crack title key instead of decrypting it with the disc key.
 * The DVD should have been opened and be in an authenticated state.
 * i_pos is the starting sector, i_len is the maximum number of sectors to read
 *****************************************************************************/
static int CrackTitleKey( libdvd_t libdvd, int i_pos, int i_len,
                          dvd_key_t p_titlekey )
{
    uint8_t       p_buf[ LIBDVD_BLOCK_SIZE ];
    const uint8_t p_packstart[4] = { 0x00, 0x00, 0x01, 0xba };
    int i_reads = 0;
    int i_encrypted = 0;
    int b_stop_scanning = 0;
    int b_read_error = 0;
    int i_ret;

    _libdvd_debug( libdvd, "cracking title key" );

    i_tries = 0;
    i_success = 0;

    do
    {
        i_ret = libdvd->pf_seek( libdvd, i_pos );

        if( i_ret != i_pos )
        {
            _libdvd_error( libdvd, "seek failed" );
        }

        i_ret = libdvd_read( libdvd, p_buf, 1, LIBDVD_NOFLAGS );

        /* Either we are at the end of the physical device or the auth
         * have failed / were not done and we got a read error. */
        if( i_ret <= 0 )
        {
            if( i_ret == 0 )
            {
                _libdvd_debug( libdvd, "read returned 0 (end of device?)" );
            }
            else if( !b_read_error )
            {
                _libdvd_debug( libdvd, "read error, resorting to secret "
                                       "arcanes to recover" );

                /* Reset the drive before trying to continue */
                _libdvd_close( libdvd );
                _libdvd_open( libdvd );

                b_read_error = 1;
                continue;
            }
            break;
        }

        /* Stop when we find a non MPEG stream block.
         * (We must have reached the end of the stream).
         * For now, allow all blocks that begin with a start code. */
        if( memcmp( p_buf, p_packstart, 3 ) )
        {
            _libdvd_debug( libdvd, "non MPEG block found (end of title)" );
            break;
        }

        if( p_buf[0x0d] & 0x07 )
            _libdvd_debug( libdvd, "stuffing in pack header" );

        /* PES_scrambling_control does not exist in a system_header,
         * a padding_stream or a private_stream2 (and others?). */
        if( p_buf[0x14] & 0x30  && ! ( p_buf[0x11] == 0xbb
                                       || p_buf[0x11] == 0xbe
                                       || p_buf[0x11] == 0xbf ) )
        {
            i_encrypted++;

            if( AttackPattern(p_buf, i_reads, p_titlekey) > 0 )
            {
                b_stop_scanning = 1;
            }
#if 0
            if( AttackPadding(p_buf, i_reads, p_titlekey) > 0 )
            {
                b_stop_scanning = 1;
            }
#endif
        }

        i_pos++;
        i_len--;
        i_reads++;

        /* Emit a progress indication now and then. */
        if( !( i_reads & 0xfff ) )
        {
            _libdvd_debug( libdvd, "still cracking..." );
        }

        /* Stop after 2000 blocks if we haven't seen any encrypted blocks. */
        if( i_reads >= 2000 && i_encrypted == 0 ) break;

    } while( !b_stop_scanning && i_len > 0);

    if( !b_stop_scanning )
    {
        _libdvd_debug( libdvd, "end of title reached" );
    }

    { /* Print some statistics. */
        char psz_info[128];
        snprintf( psz_info, sizeof(psz_info),
                  "%d of %d attempts successful, %d of %d blocks scrambled",
                  i_success, i_tries, i_encrypted, i_reads );
        _libdvd_debug( libdvd, psz_info );
    }

    if( i_success > 0 /* b_stop_scanning */ )
    {
        _libdvd_debug( libdvd, "vts key initialized" );
        return 1;
    }

    if( i_encrypted == 0 && i_reads > 0 )
    {
        memset( p_titlekey, 0, KEY_SIZE );
        _libdvd_debug( libdvd, "file was unscrambled" );
        return 0;
    }

    memset( p_titlekey, 0, KEY_SIZE );
    return -1;
}

/******************************************************************************
 * The original Ethan Hawke (DeCSSPlus) attack (modified).
 ******************************************************************************
 * Tries to find a repeating pattern just before the encrypted part starts.
 * Then it guesses that the plain text for first encrypted bytes are
 * a contiuation of that pattern.
 *****************************************************************************/
static int AttackPattern( uint8_t const p_sec[ LIBDVD_BLOCK_SIZE ],
                          int i_pos, uint8_t *p_key )
{
    unsigned int i_best_plen = 0;
    unsigned int i_best_p = 0;
    unsigned int i, j;

    /* For all cycle length from 2 to 48 */
    for( i = 2 ; i < 0x30 ; i++ )
    {
        /* Find the number of bytes that repeats in cycles. */
        for( j = i + 1;
             j < 0x80 && ( p_sec[0x7F - (j%i)] == p_sec[0x7F - j] );
             j++ )
        {
            /* We have found j repeating bytes with a cycle length i. */
            if( j > i_best_plen )
            {
                i_best_plen = j;
                i_best_p = i;
            }
        }
    }

    /* We need at most 10 plain text bytes?, so a make sure that we
     * have at least 20 repeated bytes and that they have cycled at
     * least one time.  */
    if( ( i_best_plen > 3 ) && ( i_best_plen / i_best_p >= 2) )
    {
        int res;

        i_tries++;
        memset( p_key, 0, KEY_SIZE );
        res = RecoverTitleKey( 0,  &p_sec[0x80],
                      &p_sec[ 0x80 - (i_best_plen / i_best_p) * i_best_p ],
                      &p_sec[0x54] /* key_seed */, p_key );
        i_success += ( res >= 0 );
#if 0
        if( res >= 0 )
        {
            fprintf( stderr, "key is %02x:%02x:%02x:%02x:%02x ",
                     p_key[0], p_key[1], p_key[2], p_key[3], p_key[4] );
            fprintf( stderr, "at block %5d pattern len %3d period %3d %s\n",
                     i_pos, i_best_plen, i_best_p, (res>=0?"y":"n") );
        }
#endif
        return ( res >= 0 );
    }

    return 0;
}


#if 0
/******************************************************************************
 * Encrypted Padding_stream attack.
 ******************************************************************************
 * DVD specifies that there must only be one type of data in every sector.
 * Every sector is one pack and so must obviously be 2048 bytes long.
 * For the last pice of video data before a VOBU boundary there might not
 * be exactly the right amount of data to fill a sector. Then one has to
 * pad the pack to 2048 bytes. For just a few bytes this is done in the
 * header but for any large amount you insert a PES packet from the
 * Padding stream. This looks like 0x00 00 01 be xx xx ff ff ...
 * where xx xx is the length of the padding stream.
 *****************************************************************************/
static int AttackPadding( uint8_t const p_sec[ LIBDVD_BLOCK_SIZE ],
                          int i_pos, uint8_t *p_key )
{
    unsigned int i_pes_length;
    /*static int i_tries = 0, i_success = 0;*/

    i_pes_length = (p_sec[0x12]<<8) | p_sec[0x13];

    /* Coverd by the test below but usfull for debuging. */
    if( i_pes_length == LIBDVD_BLOCK_SIZE - 0x14 ) return 0;

    /* There must be room for at least 4? bytes of padding stream,
     * and it must be encrypted.
     * sector size - pack/pes header - padding startcode - padding length */
    if( ( LIBDVD_BLOCK_SIZE - 0x14 - 4 - 2 - i_pes_length < 4 ) ||
        ( p_sec[0x14 + i_pes_length + 0] == 0x00 &&
          p_sec[0x14 + i_pes_length + 1] == 0x00 &&
          p_sec[0x14 + i_pes_length + 2] == 0x01 ) )
    {
      fprintf( stderr, "plain %d %02x:%02x:%02x:%02x (type %02x sub %02x)\n",
               LIBDVD_BLOCK_SIZE - 0x14 - 4 - 2 - i_pes_length,
               p_sec[0x14 + i_pes_length + 0],
               p_sec[0x14 + i_pes_length + 1],
               p_sec[0x14 + i_pes_length + 2],
               p_sec[0x14 + i_pes_length + 3],
               p_sec[0x11], p_sec[0x17 + p_sec[0x16]]);
      return 0;
    }

    /* If we are here we know that there is a where in the pack a
       encrypted PES header is (startcode + length). It's never more
       than  two packets in the pack, so we 'know' the length. The
       plaintext at offset (0x14 + i_pes_length) will then be
       00 00 01 e0/bd/be xx xx, in the case of be the following bytes
       are also known. */

    /* An encrypted SPU PES packet with another encrypted PES packet following.
       Normaly if the following was a padding stream that would be in plain
       text. So it will be another SPU PES packet. */
    if( p_sec[0x11] == 0xbd &&
        p_sec[0x17 + p_sec[0x16]] >= 0x20 &&
        p_sec[0x17 + p_sec[0x16]] <= 0x3f )
    {
        i_tries++;
    }

    /* A Video PES packet with another encrypted PES packet following.
     * No reason execpt for time stamps to break the data into two packets.
     * So it's likely that the following PES packet is a padding stream. */
    if( p_sec[0x11] == 0xe0 )
    {
        i_tries++;
    }

    if( 1 )
    {
        /*fprintf( stderr, "key is %02x:%02x:%02x:%02x:%02x ",
                   p_key[0], p_key[1], p_key[2], p_key[3], p_key[4] );*/
        fprintf( stderr, "at block %5d padding len %4d "
                 "type %02x sub %02x\n",  i_pos, i_pes_length,
                 p_sec[0x11], p_sec[0x17 + p_sec[0x16]]);
    }

    return 0;
}
#endif
