/* libdvd.c: DVD reading library.
 *
 * Authors: Stphane Borel <stef@via.ecp.fr>
 *          Samuel Hocevar <sam@zoy.org>
 *          Hkan Hjort <d95hjort@dtek.chalmers.se>
 *
 * Copyright (C) 1998-2002 VideoLAN
 * $Id: libdvd.c,v 1.3 2004/03/01 21:36:17 hshoexer Exp $
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 */

/**
 * \mainpage libdvd developer documentation
 *
 * \section intro Introduction
 *
 * \e libdvd is a simple library designed for accessing DVDs like a block
 * device without having to bother about the decryption. The important features
 * are:
 * \li portability: currently supported platforms are GNU/Linux, FreeBSD,
 *     NetBSD, OpenBSD, BSD/OS, BeOS, Windows 95/98, Windows NT/2000, MacOS X,
 *     Solaris, HP-UX and OS/2.
 * \li adaptability: unlike most similar projects, libdvd doesn't require
 *     the region of your drive to be set and will try its best to read from
 *     the disc even in the case of a region mismatch.
 * \li simplicity: a DVD player can be built around the \e libdvd API using
 *     no more than 4 or 5 library calls.
 *
 * \e libdvd is free software, released under the General Public License.
 * This ensures that \e libdvd remains free and used only with free
 * software.
 *
 * \section api The libdvd API
 *
 * The complete \e libdvd programming interface is documented in the
 * libdvd.h file.
 *
 * \section env Environment variables
 *
 * Some environment variables can be used to change the behaviour of
 * \e libdvd without having to modify the program which uses it. These
 * variables are:
 *
 * \li \b LIBDVD_VERBOSE: sets the verbosity level.
 *     - \c 0 outputs no messages at all.
 *     - \c 1 outputs error messages to stderr.
 *     - \c 2 outputs error messages and debug messages to stderr.
 *
 * \li \b LIBDVD_METHOD: sets the authentication and decryption method
 *     that \e libdvd will use to read scrambled discs. Can be one
 *     of \c title, \c key or \c disc.
 *     - \c key is the default method. \e libdvd will use a set of
 *       calculated player keys to try and get the disc key. This can fail
 *       if the drive does not recognize any of the player keys.
 *     - \c disc is a fallback method when \c key has failed. Instead of
 *       using player keys, \e libdvd will crack the disc key using
 *       a brute force algorithm. This process is CPU intensive and requires
 *       64 MB of memory to store temporary data.
 *     - \c title is the fallback when all other methods have failed. It does
 *       not rely on a key exchange with the DVD drive, but rather uses a
 *       crypto attack to guess the title key. On rare cases this may fail
 *       because there is not enough encrypted data on the disc to perform
 *       a statistical attack, but in the other hand it is the only way to
 *       decrypt a DVD stored on a hard disc, or a DVD with the wrong region
 *       on an RPC2 drive.
 *
 * \li \b LIBDVD_RAW_DEVICE: specify the raw device to use. Exact usage will
 *     depend on your operating system, the Linux utility to set up raw devices
 *     is \c raw(8) for instance. Please note that on most operating systems,
 *     using a raw device requires highly aligned buffers: Linux requires a
 *     2048 bytes alignment (which is the size of a DVD sector).
 *
 * \li \b LIBDVD_CACHE: specify a directory in which to store title key
 *     values. This will speed up descrambling of DVDs which are in the
 *     cache. The LIBDVD_CACHE directory is created if it does not exist,
 *     and a subdirectory is created named after the DVD's title or
 *     manufacturing date. If LIBDVD_CACHE is not set or is empty, \e libdvd
 *     will use the default value which is "${HOME}/.libdvd/" under Unix and
 *     "C:\Documents and Settings\$USER\Application Data\libdvd\" under Win32.
 *     The special value "off" disables caching.
 */

/*
 * Preamble
 */
#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifdef HAVE_SYS_PARAM_H
#   include <sys/param.h>
#endif
#ifdef HAVE_PWD_H
#   include <pwd.h>
#endif
#include <fcntl.h>
#include <errno.h>

#ifdef HAVE_UNISTD_H
#   include <unistd.h>
#endif

#ifdef HAVE_LIMITS_H
#   include <limits.h>
#endif

#ifdef HAVE_DIRECT_H
#   include <direct.h>
#endif

#include "libdvd/libdvd.h"

#include "common.h"
#include "acss.h"
#include "libdvd.h"
#include "ioctl.h"
#include "device.h"

/**
 * \brief Symbol for version checks.
 *
 * The name of this symbol contains the library major number, which makes it
 * easy to check which \e libdvd development headers are installed on the
 * system with tools such as autoconf.
 *
 * The variable itself contains the exact version number of the library,
 * which can be useful for specific feature needs.
 */
char * libdvd_interface_2 = VERSION;

/**
 * \brief Open a DVD device or directory and return a libdvd instance.
 *
 * \param psz_target a string containing the target name, for instance
 *        "/dev/hdc" or "E:".
 * \return a handle to a libdvd instance or NULL on error.
 *
 * Initialize the \e libdvd library and open the requested DVD device or
 * directory. \e libdvd checks whether ioctls can be performed on the disc,
 * and when possible, the disc key is retrieved.
 *
 * libdvd_open() returns a handle to be used for all subsequent \e libdvd
 * calls. If an error occured, NULL is returned.
 */
extern libdvd_t libdvd_open ( char *psz_target )
{
    char psz_buffer[PATH_MAX];
    int i_ret, i_fd, i_savechallenge;

    char *psz_method = getenv( "LIBDVD_METHOD" );
    char *psz_verbose = getenv( "LIBDVD_VERBOSE" );
    char *psz_cache = getenv( "LIBDVD_CACHE" );
#ifndef WIN32
    char *psz_raw_device = getenv( "LIBDVD_RAW_DEVICE" );
#endif

    libdvd_t libdvd;

    /*
     *  Allocate the library structure
     */
    libdvd = malloc( sizeof( struct libdvd_s ) );
    if( libdvd == NULL )
    {
        return NULL;
    }

    /*
     *  Initialize structure with default values
     */
#ifndef WIN32
    libdvd->i_raw_fd = -1;
#endif
    libdvd->p_titles = NULL;
    libdvd->psz_device = (char *)strdup( psz_target );
    libdvd->psz_error = "no error";
    libdvd->i_method = LIBDVD_METHOD_KEY;
    libdvd->psz_cachefile[0] = '\0';
    libdvd->b_debug = 0;
    libdvd->b_errors = 0;

    /*
     *  Find verbosity from LIBDVD_VERBOSE environment variable
     */
    if( psz_verbose != NULL )
    {
        int i = atoi( psz_verbose );

        if( i >= 2 ) libdvd->b_debug = 1;
        if( i >= 1 ) libdvd->b_errors = 1;
    }

    /*
     *  Find method from LIBDVD_METHOD environment variable
     */
    if( psz_method != NULL )
    {
        if( !strncmp( psz_method, "key", 4 ) )
        {
            libdvd->i_method = LIBDVD_METHOD_KEY;
        }
        else if( !strncmp( psz_method, "title", 5 ) )
        {
            libdvd->i_method = LIBDVD_METHOD_TITLE;
        }
        else
        {
            _libdvd_error( libdvd, "unknown decrypt method, please choose "
                                   "from 'title'or 'key'" );
            free( libdvd->psz_device );
            free( libdvd );
            return NULL;
        }
    }

    /*
     *  If LIBDVD_CACHE was not set, try to guess a default value
     */
    if( psz_cache == NULL || psz_cache[0] == '\0' )
    {
#ifdef HAVE_DIRECT_H
        typedef HRESULT( WINAPI *SHGETFOLDERPATH )
                       ( HWND, int, HANDLE, DWORD, LPTSTR );

#   define CSIDL_FLAG_CREATE 0x8000
#   define CSIDL_APPDATA 0x1A
#   define SHGFP_TYPE_CURRENT 0

        char psz_home[MAX_PATH];
        HINSTANCE p_dll;
        SHGETFOLDERPATH p_getpath;

        *psz_home = '\0';

        /* Load the shfolder dll to retrieve SHGetFolderPath */
        p_dll = LoadLibrary( "shfolder.dll" );
        if( p_dll )
        {
            p_getpath = (void*)GetProcAddress( p_dll, "SHGetFolderPathA" );
            if( p_getpath )
            {
                /* Get the "Application Data" folder for the current user */
                if( p_getpath( NULL, CSIDL_APPDATA | CSIDL_FLAG_CREATE,
                               NULL, SHGFP_TYPE_CURRENT, psz_home ) == S_OK )
                {
                    FreeLibrary( p_dll );
                }
                else
                {
                    *psz_home = '\0';
                }
            }
            FreeLibrary( p_dll );
        }

        /* Cache our keys in
         * C:\Documents and Settings\$USER\Application Data\libdvd\ */
        if( *psz_home )
        {
            snprintf( psz_buffer, PATH_MAX, "%s/libdvd", psz_home );
            psz_buffer[PATH_MAX-1] = '\0';
            psz_cache = psz_buffer;
        }
#else
        char *psz_home = NULL;
#   ifdef HAVE_PWD_H
        struct passwd *p_pwd;

        /* Try looking in password file for home dir. */
        p_pwd = getpwuid(getuid());
        if( p_pwd )
        {
            psz_home = p_pwd->pw_dir;
        }
#   endif

        if( psz_home == NULL )
        {
            psz_home = getenv( "HOME" );
        }

        /* Cache our keys in ${HOME}/.libdvd/ */
        if( psz_home )
        {
            snprintf( psz_buffer, PATH_MAX, "%s/.libdvd", psz_home );
            psz_buffer[PATH_MAX-1] = '\0';
            psz_cache = psz_buffer;
        }
#endif
    }

    /*
     *  Find cache dir from the LIBDVD_CACHE environment variable
     */
    if( psz_cache != NULL )
    {
        if( psz_cache[0] == '\0' || !strcmp( psz_cache, "off" ) )
        {
            psz_cache = NULL;
        }
        /* Check that we can add the ID directory and the block filename */
        else if( strlen( psz_cache ) + 1 + 32 + 1 + 10 + 1 > PATH_MAX )
        {
            _libdvd_error( libdvd, "cache directory name is too long" );
            psz_cache = NULL;
        }
    }

    /*
     *  Open device
     */
    i_ret = _libdvd_open( libdvd );
    if( i_ret < 0 )
    {
        free( libdvd->psz_device );
        free( libdvd );
        return NULL;
    }

    libdvd->b_scrambled = 1; /* Assume the worst */
    libdvd->b_ioctls = _libdvd_use_ioctls( libdvd );

    if( libdvd->b_ioctls )
    {
        i_ret = _libdvd_test( libdvd );
        if( i_ret < 0 )
        {
            /* Disable the CSS ioctls and hope that it works? */
            _libdvd_debug( libdvd,
                           "could not check whether the disc was scrambled" );
            libdvd->b_ioctls = 0;
        }
        else
        {
            _libdvd_debug( libdvd, i_ret ? "disc is scrambled"
                                         : "disc is unscrambled" );
            libdvd->b_scrambled = i_ret;
        }
    }

    /*
     * Get Challenge.  If available, use cached challenge.  Otherwise request
     * on from user.
     */
    snprintf(libdvd->psz_challengefile, PATH_MAX, "%s/challenge", psz_cache);

    i_savechallenge = 0;

    i_fd = open(libdvd->psz_challengefile, O_CREAT | O_EXCL | O_RDONLY, 0644);
    if (i_fd < 0 && errno == EEXIST) {
	i_fd = open(libdvd->psz_challengefile, O_RDONLY);
	if (i_fd < 0)
	    _libdvd_error(libdvd, "failed opening cache file");
	else {
	    i_ret = read(i_fd, libdvd->psz_challenge,
			    sizeof(libdvd->psz_challenge) - 1);
	    if (i_ret < 0)
		_libdvd_error(libdvd, "failed to read challenge from cache");
	}
    }

    if (0 < i_fd)
	close(i_fd);

    if (libdvd->psz_challenge[0] == '\0') {
	char *tmp;

	tmp = getpass("Enter challenge, e.g. the name of your OS: ");

	if (tmp == NULL) {
	    _libdvd_error(libdvd, "getpass failed");
	    return NULL;
	}

	strlcpy(libdvd->psz_challenge, tmp,
		      sizeof(libdvd->psz_challenge));

	/* We do this later, as $HOME/.libdvd/ might not yet exist. */
	i_savechallenge = 1;
    }

    /* If disc is CSS protected and the ioctls work, authenticate the drive */
    if( libdvd->b_scrambled && libdvd->b_ioctls )
    {
        i_ret = _libdvd_disckey( libdvd );

        if( i_ret < 0 )
        {
            _libdvd_close( libdvd );
            free( libdvd->psz_device );
            free( libdvd );
            return NULL;
        }
    }

    /* If the cache is enabled, extract a unique disc ID */
    if( psz_cache )
    {
        uint8_t p_sector[LIBDVD_BLOCK_SIZE];
        unsigned char   psz_debug[PATH_MAX+30];
        unsigned char * psz_title, * psz_serial;
        int i;

        /* We read sector 0. If it starts with 0x000001ba (BE), we are
         * reading a VOB file, and we should not cache anything. */

        i_ret = libdvd->pf_seek( libdvd, 0 );
        if( i_ret != 0 )
        {
            goto nocache;
        }

        i_ret = libdvd->pf_read( libdvd, p_sector, 1 );
        if( i_ret != 1 )
        {
            goto nocache;
        }

        if( p_sector[0] == 0x00 && p_sector[1] == 0x00
             && p_sector[2] == 0x01 && p_sector[3] == 0xba )
        {
            goto nocache;
        }

        /* The data we are looking for is at sector 16 (32768 bytes):
         *  - offset 40: disc title (32 uppercase chars)
         *  - offset 813: manufacturing date + serial no (16 digits) */

        i_ret = libdvd->pf_seek( libdvd, 16 );
        if( i_ret != 16 )
        {
            goto nocache;
        }

        i_ret = libdvd->pf_read( libdvd, p_sector, 1 );
        if( i_ret != 1 )
        {
            goto nocache;
        }

        /* Get the disc title */
        psz_title = p_sector + 40;
        psz_title[32] = '\0';

        for( i = 0 ; i < 32 ; i++ )
        {
            if( psz_title[i] <= ' ' )
            {
                psz_title[i] = '\0';
                break;
            }
            else if( psz_title[i] == '/' || psz_title[i] == '\\' )
            {
                psz_title[i] = '-';
            }
        }

        /* Get the date + serial */
        psz_serial = p_sector + 813;
        psz_serial[16] = '\0';

        /* Check that all characters are digits, otherwise convert. */
        for( i = 0 ; i < 16 ; i++ )
        {
            if( psz_serial[i] < '0' || psz_serial[i] > '9' )
            {
                sprintf( psz_serial,
                         "%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X",
                         psz_serial[0], psz_serial[1], psz_serial[2],
                         psz_serial[3], psz_serial[4], psz_serial[5],
                         psz_serial[6], psz_serial[7] );
                break;
            }
        }

        /* We have a disc name or ID, we can create the cache dir */
        i = sprintf( libdvd->psz_cachefile, "%s", psz_cache );
#if !defined( WIN32 ) || defined( SYS_CYGWIN )
        i_ret = mkdir( libdvd->psz_cachefile, 0755 );
#else
        i_ret = mkdir( libdvd->psz_cachefile );
#endif
        if( i_ret < 0 && errno != EEXIST )
        {
            _libdvd_error( libdvd, "failed creating cache directory" );
            libdvd->psz_cachefile[0] = '\0';
            goto nocache;
        }

	/* Save challenge */
	if (i_savechallenge) {
	    i_fd = open(libdvd->psz_challengefile, O_CREAT | O_RDWR, 0644);
	    if (i_fd < 0)
		_libdvd_error(libdvd, "failed creating challenge cache file");
	    
	    i_ret = write(i_fd, libdvd->psz_challenge,
		    strlen(libdvd->psz_challenge));

	    if (i_ret < (int)strlen(libdvd->psz_challenge)) {
		_libdvd_error(libdvd, "failed to save challenge");

		/* Just try to get rid of the cache file */
		unlink(libdvd->psz_challengefile);
	    }

	    close(i_fd);
	}

        i += sprintf( libdvd->psz_cachefile + i, "/%s#%s", psz_title,
                      psz_serial );
#if !defined( WIN32 ) || defined( SYS_CYGWIN )
        i_ret = mkdir( libdvd->psz_cachefile, 0755 );
#else
        i_ret = mkdir( libdvd->psz_cachefile );
#endif
        if( i_ret < 0 && errno != EEXIST )
        {
            _libdvd_error( libdvd, "failed creating cache subdirectory" );
            libdvd->psz_cachefile[0] = '\0';
            goto nocache;
        }
        i += sprintf( libdvd->psz_cachefile + i, "/");

        /* Pointer to the filename we will use. */
        libdvd->psz_block = libdvd->psz_cachefile + i;

        sprintf( psz_debug, "using key cache dir: %s",
                            libdvd->psz_cachefile );
        _libdvd_debug( libdvd, psz_debug );
    }
    nocache:

#ifndef WIN32
    if( psz_raw_device != NULL )
    {
        _libdvd_raw_open( libdvd, psz_raw_device );
    }
#endif

    /* Seek at the beginning, just for safety. */
    libdvd->pf_seek( libdvd, 0 );

    return libdvd;
}

/**
 * \brief Return a string containing the latest error that occured in the
 *        given \e libdvd instance.
 *
 * \param libdvd a \e libdvd instance.
 * \return a null-terminated string containing the latest error message.
 *
 * This function returns a constant string containing the latest error that
 * occured in \e libdvd. It can be used to format error messages at your
 * convenience in your application.
 */
extern char * libdvd_error ( libdvd_t libdvd )
{
    return libdvd->psz_error;
}

/**
 * \brief Seek in the disc and change the current key if requested.
 *
 * \param libdvd a \e libdvd instance.
 * \param i_blocks an absolute block offset to seek to.
 * \param i_flags #LIBDVD_NOFLAGS, optionally ored with one of
 * #LIBDVD_SEEK_KEY or #LIBDVD_SEEK_MPEG.
 * \return the new position in blocks, or a negative value in case an error
 *         happened.
 *
 * This function seeks to the requested position, in logical blocks.
 *
 * You typically set \p i_flags to #LIBDVD_NOFLAGS when seeking in a .IFO.
 *
 * If #LIBDVD_SEEK_MPEG is specified in \p i_flags and if \e libdvd finds it
 * reasonable to do so (ie, if the libdvd method is not "title"), the current
 * title key will be checked and a new one will be calculated if necessary.
 * This flag is typically used when reading data from a VOB.
 *
 * If #LIBDVD_SEEK_KEY is specified, the title key will be always checked,
 * even with the "title" method. This is equivalent to using the now
 * deprecated libdvd_title() call. This flag is typically used when seeking
 * in a new title.
 */
extern int libdvd_seek ( libdvd_t libdvd, int i_blocks, int i_flags )
{
    /* title cracking method is too slow to be used at each seek */
    if( ( ( i_flags & LIBDVD_SEEK_MPEG )
             && ( libdvd->i_method != LIBDVD_METHOD_TITLE ) )
       || ( i_flags & LIBDVD_SEEK_KEY ) )
    {
        /* check the title key */
        if( _libdvd_title( libdvd, i_blocks ) )
        {
            return -1;
        }
    }

    return libdvd->pf_seek( libdvd, i_blocks );
}

/**
 * \brief Read from the disc and decrypt data if requested.
 *
 * \param libdvd a \e libdvd instance.
 * \param p_buffer a buffer that will contain the data read from the disc.
 * \param i_blocks the amount of blocks to read.
 * \param i_flags #LIBDVD_NOFLAGS, optionally ored with #LIBDVD_READ_DECRYPT.
 * \return the amount of blocks read, or a negative value in case an
 *         error happened.
 *
 * This function reads \p i_blocks logical blocks from the DVD.
 *
 * You typically set \p i_flags to #LIBDVD_NOFLAGS when reading data from a
 * .IFO file on the DVD.
 *
 * If #LIBDVD_READ_DECRYPT is specified in \p i_flags, libdvd_read() will
 * automatically decrypt scrambled sectors. This flag is typically used when
 * reading data from a .VOB file on the DVD. It has no effect on unscrambled
 * discs or unscrambled sectors, and can be safely used on those.
 *
 * \warning libdvd_read() expects to be able to write \p i_blocks *
 *          #LIBDVD_BLOCK_SIZE bytes in \p p_buffer.
 */
extern int libdvd_read ( libdvd_t libdvd, void *p_buffer,
                                          int i_blocks,
                                          int i_flags )
{
    int i_ret, i_index;

    i_ret = libdvd->pf_read( libdvd, p_buffer, i_blocks );

    if( i_ret <= 0
         || !libdvd->b_scrambled
         || !(i_flags & LIBDVD_READ_DECRYPT) )
    {
        return i_ret;
    }

    if( ! memcmp( libdvd->scramble.p_title_key, "\0\0\0\0\0", 5 ) )
    {
        /* For what we believe is an unencrypted title,
         * check that there are no encrypted blocks */
        for( i_index = i_ret; i_index; i_index-- )
        {
            if( ((uint8_t*)p_buffer)[0x14] & 0x30 )
            {
                _libdvd_error( libdvd, "no key but found encrypted block" );
                /* Only return the initial range of unscrambled blocks? */
                /* or fail completely? return 0; */
                break;
            }
            p_buffer = (void *) ((uint8_t *)p_buffer + LIBDVD_BLOCK_SIZE);
        }
    }
    else
    {
        /* Decrypt the blocks we managed to read */
        for( i_index = i_ret; i_index; i_index-- )
        {
            _libdvd_unscramble( libdvd->scramble.p_title_key, p_buffer );
            ((uint8_t*)p_buffer)[0x14] &= 0x8f;
            p_buffer = (void *) ((uint8_t *)p_buffer + LIBDVD_BLOCK_SIZE);
        }
    }

    return i_ret;
}

/**
 * \brief Read from the disc into multiple buffers and decrypt data if
 *        requested.
 *
 * \param libdvd a \e libdvd instance.
 * \param p_iovec a pointer to an array of iovec structures that will contain
 *        the data read from the disc.
 * \param i_blocks the amount of blocks to read.
 * \param i_flags #LIBDVD_NOFLAGS, optionally ored with #LIBDVD_READ_DECRYPT.
 * \return the amount of blocks read, or a negative value in case an
 *         error happened.
 *
 * This function reads \p i_blocks logical blocks from the DVD and writes them
 * to an array of iovec structures.
 *
 * You typically set \p i_flags to #LIBDVD_NOFLAGS when reading data from a
 * .IFO file on the DVD.
 *
 * If #LIBDVD_READ_DECRYPT is specified in \p i_flags, libdvd_readv() will
 * automatically decrypt scrambled sectors. This flag is typically used when
 * reading data from a .VOB file on the DVD. It has no effect on unscrambled
 * discs or unscrambled sectors, and can be safely used on those.
 *
 * \warning libdvd_readv() expects to be able to write \p i_blocks *
 *          #LIBDVD_BLOCK_SIZE bytes in the buffers pointed by \p p_iovec.
 *          Moreover, all iov_len members of the iovec structures should be
 *          multiples of #LIBDVD_BLOCK_SIZE.
 */
extern int libdvd_readv ( libdvd_t libdvd, void *p_iovec,
                                           int i_blocks,
                                           int i_flags )
{
    struct iovec *_p_iovec = (struct iovec *)p_iovec;
    int i_ret, i_index;
    void *iov_base;
    size_t iov_len;

    i_ret = libdvd->pf_readv( libdvd, _p_iovec, i_blocks );

    if( i_ret <= 0
         || !libdvd->b_scrambled
         || !(i_flags & LIBDVD_READ_DECRYPT) )
    {
        return i_ret;
    }

    /* Initialize loop for decryption */
    iov_base = _p_iovec->iov_base;
    iov_len = _p_iovec->iov_len;

    /* Decrypt the blocks we managed to read */
    for( i_index = i_ret; i_index; i_index-- )
    {
        /* Check that iov_len is a multiple of 2048 */
        if( iov_len & 0x7ff )
        {
            return -1;
        }

        while( iov_len == 0 )
        {
            _p_iovec++;
            iov_base = _p_iovec->iov_base;
            iov_len = _p_iovec->iov_len;
        }

        _libdvd_unscramble( libdvd->scramble.p_title_key, iov_base );
        ((uint8_t*)iov_base)[0x14] &= 0x8f;

        iov_base = (void *) ((uint8_t*)iov_base + LIBDVD_BLOCK_SIZE);
        iov_len -= LIBDVD_BLOCK_SIZE;
    }

    return i_ret;
}

/**
 * \brief Close the DVD and clean up the library.
 *
 * \param libdvd a \e libdvd instance.
 * \return zero in case of success, a negative value otherwise.
 *
 * This function closes the DVD device and frees all the memory allocated
 * by \e libdvd. On return, the #libdvd_t is invalidated and may not be
 * used again.
 */
extern int libdvd_close ( libdvd_t libdvd )
{
    dvd_title_t *p_title;
    int i_ret;

    /* Free our list of keys */
    p_title = libdvd->p_titles;
    while( p_title )
    {
        dvd_title_t *p_tmptitle = p_title->p_next;
        free( p_title );
        p_title = p_tmptitle;
    }

    i_ret = _libdvd_close( libdvd );

    if( i_ret < 0 )
    {
        return i_ret;
    }

    free( libdvd->psz_device );
    free( libdvd );

    return 0;
}

/*
 *  Deprecated. See libdvd_seek().
 */
#undef libdvd_title
extern int libdvd_title ( libdvd_t libdvd, int i_block )
{
    return _libdvd_title( libdvd, i_block );
}

