/***************************************************************************
                          main_menu.c  -  description
                             -------------------
    begin                : Thu Jun 28 2001
    copyright            : (C) 2001 by Michael Speck
    email                : kulkanie@gmx.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <SDL.h>

#ifdef WITH_SOUND
#include <SDL_mixer.h>
#include "audio.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "sdl.h"
#include "config.h"
#include "tools.h"
#include "dynlist.h"
#include "file.h"
#include "nation.h"
#include "unit.h"
#include "player.h"
#include "date.h"
#include "map.h"
#include "scenario.h"
#include "campaign.h"
#include "gui.h"
#include "theme.h"
#include "ai_action.h"
#include "engine_tools.h"
#include "window.h"
#include "engine.h"
#include "game_loop.h"
#include "pack_info.h"
#include "main_menu.h"

extern Config config;
extern Scen scen;
extern Camp camp;
extern Sdl sdl;

/* the type as which selection window is run */
enum { AS_SCEN, AS_CAMP, AS_PACK };
/* if selection window is run there is filename list with the real file names
   (but another list created in update_file_list() is displayed instead) */
Text *fnames = 0;
/* current path of selection window */
char path[512];

/*
====================================================================
Create window that is used to set player controls and other
scenario stuff that is set for before starting the scenario and
doesn't change when playing.
====================================================================
*/
Window* create_setup_window( Theme *theme )
{
    int width, height;
    Window *window = 0;
    int i;
    Window *button;

    width = 300;
    height = 40 + scen.player_count * 40 + 50;

    window = create_wallpaper_window( theme->parent_back ,theme->parent_frame, width, height );

    /* description */
    create_label( window, NONE, 10, 10, window->width - 20, 20, theme->stan_font,
                  ALIGN_X_CENTER | ALIGN_Y_CENTER, "Set Player Controls:" );
    /* accept, cancel buttons */
    create_text_button( window, ID_ACCEPT, theme->menu_text_button,
                        window->width / 2 - 80, window->height - 40, 80, 20,
                        theme->stan_font, "Ok" );
    create_text_button( window, ID_CANCEL, theme->menu_text_button,
                        window->width / 2, window->height - 40, 80, 20,
                        theme->stan_font, "Cancel" );

    /* add player stuff add 30 offset */
    for ( i = 0; i < scen.player_count; i++ ) {
        create_label( window, NONE, 10, 40 + i * 40,
                      180, 40,
                      theme->stan_font,
                      ALIGN_X_LEFT | ALIGN_Y_CENTER, scen.players[i]->name );
        button = create_text_button( window, 101 + i, theme->menu_text_button,
                            200, 50 + i * 40, 70, 20,
                            theme->stan_font, "" );
        /* set button text */
        if ( scen.players[i]->cont == HUMAN )
            set_simple_window_text( button, "Human", NO_REFRESH );
        else
            set_simple_window_text( button, "CPU", NO_REFRESH );
    }

    return window;
}
/*
====================================================================
Get current player controls of loaded scenario.
====================================================================
*/
int* get_cur_cont() {
    int *cont;
    int i;
    cont = calloc( scen.player_count, sizeof( int ) );
    for ( i = 0; i < scen.player_count; i++ )
        cont[i] = scen.players[i]->cont;
    return cont;
}
/*
====================================================================
Run the scenario setup window and save result in passed pointers.
player_cont is the player control either HUMAN or CPU
====================================================================
*/
void run_setup_window( Window *setup_window )
{
    int i;
    int leave = 0;
    SDL_Event event;
    Window *child = 0;
    int *player_cont = 0;

    player_cont = get_cur_cont();

    open_window( setup_window, REFRESH  );
    while ( !leave ) {
        if ( SDL_WaitEvent( &event ) ) {
            handle_window_event( setup_window, &event, &child );
            update_window( setup_window, IF_NEEDED );
            if ( child )
                switch ( child->id ) {
                    case ID_ACCEPT: leave = 1; break;
                    case ID_CANCEL: leave = 1; break;
                    default:
                        i = child->id - 101;
                        if ( player_cont[i] == HUMAN )
                            player_cont[i] = CPU;
                        else
                            player_cont[i] = HUMAN;
                        if ( player_cont[i] == HUMAN )
                            set_simple_window_text( child, "Human", REFRESH );
                        else
                            set_simple_window_text( child, "CPU", REFRESH );
                        break;
                }
        }
    }
    close_window( setup_window );
    /* if ACCEPT apply changes */
    if ( child->id == ID_ACCEPT )
        for ( i = 0; i < scen.player_count; i++ )
            scen.players[i]->cont = player_cont[i];
    free( player_cont );
}

/*
====================================================================
Create the window that is used for menu 'Play Scen/Camp' and
'Packages'.
====================================================================
*/
Window *create_sel_window( Theme *theme )
{
    Window *window = 0;

    /* size as follows */
    /* border 20 + list 180 + gap 20 + list 320 + 20 */
    /* border 20 + label 20 + gap 20 + height 260 + 20 */
    /* the right listbox is a bit (about 40) smaller than the other one giving space */
    /* for three buttons: start, settings(scenario), cancel */
    window = create_wallpaper_window( theme->parent_back ,theme->parent_frame, 560, 340 );

    /* description */
    create_label( window, ID_LABEL, 20, 10, window->width - 40, 20, theme->stan_font,
                  ALIGN_X_CENTER | ALIGN_Y_CENTER, "" );
    /* file list */
    create_list( window, ID_FILE_LIST, theme->list_frame,
                        20, 40,
                        180, window->height - 60,
                        6,
                        theme->stan_font, 0, SINGLE_SELECT, 0x0000ff,
                        0, theme->down_arrow, theme->up_arrow, theme->track_ball,
                        20 );
    /* description list */
    create_list( window, ID_DESC_LIST, theme->list_frame,
                        220, 40,
                        320, window->height - 100,
                        6,
                        theme->stan_font, 0, NO_SELECT, 0x0,
                        0, theme->down_arrow, theme->up_arrow, theme->track_ball,
                        20 );

    /* buttons */
    create_text_button( window, ID_ACCEPT, theme->menu_text_button,
                        window->width -  340, window->height - 40, 80, 20,
                        theme->stan_font, "Start" );
    create_text_button( window, ID_SCEN_SET, theme->menu_text_button,
                        window->width -  240, window->height - 40, 120, 20,
                        theme->stan_font, "Settings..." );
    create_text_button( window, ID_CANCEL, theme->menu_text_button,
                        window->width -  100, window->height - 40, 80, 20,
                        theme->stan_font, "Cancel" );


    return window;
}

/*
====================================================================
Add the directory name to the current path or remove last name if
it's '..'.
====================================================================
*/
void add_dir_to_path( char *dir_name )
{
    char *str = 0;
    char aux_path[512];

    /* one dir up? */
    if ( dir_name[0] == '.' && dir_name[1] == '.') {
        str = strrchr( path, '/' );
        if ( str ) {
            if ( str != path ) {
                memset( str, 0, strlen( str ) );
                str[0] = 0;
            }
            else
                strcpy( path, "/" );
        }
    }
    else {
        /* add dir */
        sprintf( aux_path, "%s/%s", path, dir_name );
        strcpy( path, aux_path );
    }
}

/*
====================================================================
Set filename list (and window list) from path and type.
====================================================================
*/
void update_file_list( Window *list_window, char *ext, char *root, int type )
{
    int i;
    char str[256];
    Text *wnames = 0;

    /* remove old file list */
    if ( fnames ) delete_text( fnames );
    /* get new file list */
    fnames = get_file_list( path, ext, root );

    /* create new window list from file name list */
    wnames = calloc( 1, sizeof( Text ) );
    wnames->count = fnames->count;
    wnames->lines = calloc( wnames->count, sizeof( char* ) );
    for ( i = 0; i < wnames->count; i++ ) {
        /* directories always look the same */
        if ( fnames->lines[i][0] == '*' ) {
            sprintf( str, "<%s>", fnames->lines[i] + 1 );
            wnames->lines[i] = strdup( str );
        }
        else
            if ( strlen( fnames->lines[i] ) > 4 ) { /* less than 4 signs means there can't be an extension */
                /* display name of normal file */
                /* uppercase first letter and leave away extension */
                strcpy( str, fnames->lines[i] );
                str[strlen( str ) - 4] = 0; /* extensions are always four signs long */
                str[0] = toupper( str[0] ); /* upper case */
                wnames->lines[i] = strdup( str );
            }
    }

    /* set this list */
    set_window_text( list_window, wnames, NO_REFRESH );
}

/*
====================================================================
Setup selection window depending on type:
AS_SCEN / AS_CAMP scenarion, campaign selection
AS_PACK package credits
====================================================================
*/
void prep_sel_window( Window *sel_window, int type )
{
    char ext[4];
    char root[64];

    /* disable start and settings button */
    /* if campaign or scenario was selected the start button will go active */
    /* as will the settings button if its a scenario */
    get_child( sel_window, ID_SCEN_SET )->active = 0;
    get_child( sel_window, ID_ACCEPT )->active = 0;

    /* set label */
    switch ( type ) {
        case AS_SCEN:
            set_simple_window_text( get_child( sel_window, ID_LABEL ), "Play Single Scenario", NO_REFRESH );
            break;
        case AS_CAMP:
            set_simple_window_text( get_child( sel_window, ID_LABEL ), "Play Campaign", NO_REFRESH );
            break;
        case AS_PACK:
            set_simple_window_text( get_child( sel_window, ID_LABEL ), "Installed Packages", NO_REFRESH );
            break;
    }

    /* description window will be empty in the beginning but the file list must be set */
    set_window_text( get_child( sel_window, ID_DESC_LIST ), 0, REFRESH );
    sprintf( path, "%s", SRC_DIR );
    switch ( type ) {
        case AS_SCEN:
            strcpy( root, "/scenarios" );
            strcpy( ext, "scn" );
            break;
        case AS_CAMP:
            strcpy( root, "/campaigns" );
            strcpy( ext, "cmp" );
            break;
        case AS_PACK:
            strcpy( root, "/packages" );
            strcpy( ext, "inf" );
            break;
    }
    strcat( path, root );
    update_file_list( get_child( sel_window, ID_FILE_LIST ), ext, root, type );
}

/*
====================================================================
Get the file_name without SRC_DIR/{scenarios, campaigns} as this
path is added when game's initiated. List_name is the very file name
belonging to the list label displayed on screen.
====================================================================
*/
char* get_file_name( char *list_name, int type )
{
    char *short_path;
    char *file_name;
    /* as we don't allow leaving the very source directory it's either this or a subdirectory */
    /* +1 because of the slash */
    short_path = path + strlen( SRC_DIR ) + 1;
    /* now either 'campaigns or 'scenarios' follows */
    switch ( type ) {
        case AS_SCEN: short_path += strlen( "scenarios" ); break;
        case AS_CAMP: short_path += strlen( "campaigns" ); break;
    }
    /* do only add +1 if a slash follows */
    if ( short_path[0] == '/' ) short_path += 1;
    /* now short_path either contains the subdirectory within or just nothing */
    /* if it's empty don't add it */
    if ( short_path[0] == 0 )
        file_name = strdup( list_name );
    else {
        file_name = calloc( strlen( short_path ) + 1 + strlen( list_name ) + 1, sizeof( char ) );
        sprintf( file_name, "%s/%s", short_path, list_name );
    }
    return file_name;
}

/*
====================================================================
Handle selection window in case its used to select a scenario.
child is the subwindow that caused the event.
Set leave true if selection window shall be left.
====================================================================
*/
void handle_scen_menu( Engine *engine, Window *sel_window, Window *child, int *engine_used )
{
    int id; /* index in fname_list */
    char text_str[4096];
    char *file_name; /* file name in source directory without SRC_DIR */
    Window *setup_window = 0;
    int *player_cont = 0;

    /* settings button hit? */
    if ( child->id == ID_SCEN_SET ) {
        setup_window = create_setup_window( engine->theme );
        run_setup_window( setup_window );
    }
    /* file list used? */
    if ( child->id == ID_FILE_LIST ) {
        /* nothing was selected */
        if ( child->sel_lines_count == 0 ) {
            set_active( get_child( sel_window, ID_ACCEPT ), 0, REFRESH );
            set_active( get_child( sel_window, ID_SCEN_SET ), 0, REFRESH );
            set_window_text( get_child( sel_window, ID_DESC_LIST ), 0, REFRESH );
            return;
        }
        /* get id */
        id = child->cur_line_sel;
        /* directory was selected */
        if ( fnames->lines[id][0] == '*' ) {
            /* deactivate buttons */
            set_active( get_child( sel_window, ID_ACCEPT ), 0, REFRESH );
            set_active( get_child( sel_window, ID_SCEN_SET ), 0, REFRESH );
            set_window_text( get_child( sel_window, ID_DESC_LIST ), 0, REFRESH );
            /* load directory */
            add_dir_to_path( fnames->lines[id] + 1 );
            update_file_list( child, "scn", "/scenarios", AS_SCEN );
            return;
        }
        /* scenario was selected */
        /* load scenario description */
        file_name = get_file_name( fnames->lines[id], AS_SCEN );
        if ( load_scen( file_name, SHALLOW ) ) {
            /* scenario works and found so activate buttons */
            set_active( get_child( sel_window, ID_ACCEPT ), 1, REFRESH );
            set_active( get_child( sel_window, ID_SCEN_SET ), 1, REFRESH );
            /* create description and pass to description window */
            sprintf( text_str, "%s##%s##%i Players#%i Turns##Authors: %s",
                     scen.title, scen.desc, scen.player_count, scen.minor_turn_limit, scen.authors );
            set_window_text( get_child( sel_window, ID_DESC_LIST ),
                             create_text( text_str, get_child( sel_window, ID_DESC_LIST )->char_limit ),
                             REFRESH );
        }
        free( file_name );
        /* scenario is not cleared as some information are needed for settings (player controls) */
        /* this doesn't matter as a scenario is cleared first everytime load_scen() is called */
    }

    /* if ACCEPT button is on a valid scenario has been chosen */
    if ( child->id == ID_ACCEPT ) {
        /* get id */
        id = get_child( sel_window, ID_FILE_LIST )->cur_line_sel;
        /* check settings */
        file_name = get_file_name( fnames->lines[id], AS_SCEN );
        /* use the control settings of the currently loaded shallow scenario */
        /* controls are freed by init_engine */
        player_cont = get_cur_cont();
        prep_game( engine, file_name, 1, player_cont );
        run_game( engine );
        free( file_name );
        /* leave this window when done */
        *engine_used = 1;
    }
}

/*
====================================================================
Handle selection window in case its used to select a campaign.
child is the subwindow that caused the event.
Set leave true if selection window shall be left.
====================================================================
*/
void handle_camp_menu( Engine *engine, Window *sel_window, Window *child, int *engine_used )
{
    int id; /* index in fname_list */
    char text_str[4096];
    char *file_name;

    /* file list used? */
    if ( child->id == ID_FILE_LIST ) {
        /* nothing was selected */
        if ( child->sel_lines_count == 0 ) {
            set_active( get_child( sel_window, ID_ACCEPT ), 0, REFRESH );
            set_window_text( get_child( sel_window, ID_DESC_LIST ), 0, REFRESH );
            return;
        }
        /* get id */
        id = child->cur_line_sel;
        /* directory was selected */
        if ( fnames->lines[id][0] == '*' ) {
            /* deactivate buttons */
            set_active( get_child( sel_window, ID_ACCEPT ), 0, REFRESH );
            /* load directory */
            add_dir_to_path( fnames->lines[id] + 1 );
            update_file_list( child, "cmp", "/campaigns", AS_CAMP );
            set_window_text( get_child( sel_window, ID_DESC_LIST ), 0, REFRESH );
            return;
        }
        /* campaign was selected */
        /* load campaign description */
        file_name = get_file_name( fnames->lines[id], AS_CAMP );
        if ( load_camp( file_name ) ) {
            /* scenario works and found so activate buttons */
            set_active( get_child( sel_window, ID_ACCEPT ), 1, REFRESH );
            /* create description and pass to description window */
            sprintf( text_str, "%s##%s##Authors: %s",
                     camp.title, camp.desc, camp.authors );
            set_window_text( get_child( sel_window, ID_DESC_LIST ),
                             create_text( text_str, get_child( sel_window, ID_DESC_LIST )->char_limit ),
                             REFRESH );
        }
        /* clear scenario */
        free( file_name );
        clear_camp();
    }

    /* if ACCEPT button is on a valid campagin has been chosen */
    if ( child->id == ID_ACCEPT ) {
        /* get id */
        id = get_child( sel_window, ID_FILE_LIST )->cur_line_sel;
        /* run game */
        file_name = get_file_name( fnames->lines[id], AS_CAMP );
        prep_game( engine, file_name, 0, 0 );
        run_game( engine );
        free( file_name );
        /* leave this window when done */
        *engine_used = 1;
    }
}

/*
====================================================================
Handle packages window in case its used to show information on
installed packages.
child is the subwindow that caused the event.
Set leave true if selection window shall be left.
====================================================================
*/
void handle_pack_menu( Window *sel_window, Window *child )
{
    int id; /* index in fname_list */
    char text_str[4096];
    Pack_Info *pack_info = 0; /* information about an installed package */
    int i;
    char str[256];

    /* file list used? */
    if ( child->id == ID_FILE_LIST ) {
        /* nothing was selected */
        if ( child->sel_lines_count == 0 ) {
            set_active( get_child( sel_window, ID_ACCEPT ), 0, REFRESH );
            set_window_text( get_child( sel_window, ID_DESC_LIST ), 0, REFRESH );
            return;
        }
        /* get id */
        id = child->cur_line_sel;
        /* directory was selected */
        if ( fnames->lines[id][0] == '*' ) {
            /* deactivate buttons */
            set_active( get_child( sel_window, ID_ACCEPT ), 0, REFRESH );
            /* load directory */
            add_dir_to_path( fnames->lines[id] + 1 );
            update_file_list( child, "inf", "/packages", AS_PACK );
            set_window_text( get_child( sel_window, ID_DESC_LIST ), 0, REFRESH );
            return;
        }
        /* packages was selected */
        /* load package info */
        if ( ( pack_info = load_pack_info( fnames->lines[id] ) ) != 0 ) {
            /* create description and pass to description window */
            sprintf( text_str, "%s##%s##Authors: %s##Files:#",
                     pack_info->title, pack_info->desc, pack_info->authors );
            /* add files */
            for ( i = 0; i < pack_info->file_count; i++ ) {
                sprintf( str, "# %s", pack_info->file_list[i] );
                strcat( text_str, str );
            }
            set_window_text( get_child( sel_window, ID_DESC_LIST ),
                             create_text( text_str, get_child( sel_window, ID_DESC_LIST )->char_limit),
                             REFRESH );
            delete_pack_info( pack_info );
        }
    }

}

/*
====================================================================
Prepare and run selection window either as SCEN, CAMP or PACK. Use
functions handle_scen_menu(), handle_camp_menu(), handle_pack_menu()
to handle selection window.
====================================================================
*/
void run_sel_window( Engine *engine, Window *sel_window, int type )
{
    SDL_Event event;
    int leave = 0;
    Window *child = 0;
    int engine_used = 0; /* if engine was used proceed to title screen */

    /* prepare window; creates fname_list */
    prep_sel_window( sel_window, type );

    /* run window */
    open_window( sel_window, REFRESH  );
    while ( !leave ) {
        if ( SDL_WaitEvent( &event ) )
            handle_window_event( sel_window, &event, &child );
        if ( child ) {
            if ( child->id == ID_CANCEL )
                leave = 1;
            else
                switch ( type ) {
                    case AS_SCEN:
                        handle_scen_menu( engine, sel_window, child, &engine_used );
                        break;
                    case AS_CAMP:
                        handle_camp_menu( engine, sel_window, child, &engine_used );
                        break;
                    case AS_PACK:
                        handle_pack_menu( sel_window, child );
                        break;
            }
        }
        if ( !engine_used ) {
            update_window( sel_window, IF_NEEDED );
            refresh_rects();
        }
        else break;
    }
    if ( !engine_used )
        close_window( sel_window );

    /* free filename list */
    delete_text( fnames ); fnames = 0;
}

/*
====================================================================
Fill screen with wallpaper.
====================================================================
*/
void draw_title_wallpaper( Theme *theme )
{
    int x, y;	
    /* if title window is smaller fill whole screen with parent_back wallpaper */
    for ( x = 0; x < sdl.screen->w; x += theme->parent_back->w )
        for ( y = 0; y < sdl.screen->h; y += theme->parent_back->h ) {
            DEST( sdl.screen, x, y, theme->parent_back->w, theme->parent_back->h );
            SOURCE( theme->parent_back, 0, 0 );
            blit_surf();
        }
}
/*
====================================================================
Create the title menu window with a title picture (if any) a label
and all nescessary buttons. Use the standard font and maybe other
grapihcs from the passed theme.
====================================================================
*/
Window* create_title_window( Theme *theme )
{
    Window *window = 0;
    Window *label = 0;
    int x_offset, y_offset, y_add;
    int width = 640, height = 480;

    if ( theme == 0 ) return 0;

    /* Get the title picture. As this is  optional use create_wallpaper_window() */
    /* to create a black pic if not found. */
    theme->stan_font->align = ALIGN_X_LEFT | ALIGN_Y_BOTTOM;
    /* urls */
    write_text( theme->stan_font, theme->title_pic, 2, theme->title_pic->h - theme->stan_font->height - 4, "http://lgeneral.sourceforge.net", OPAQUE );
    write_text( theme->stan_font, theme->title_pic, 2, theme->title_pic->h - 2, "http://lgames.sourceforge.net", OPAQUE );
    /* copyright */
    theme->stan_font->align = ALIGN_X_RIGHT | ALIGN_Y_BOTTOM;
    write_text( theme->stan_font, theme->title_pic, theme->title_pic->w - 1, theme->title_pic->h - 2, "Copyright 2001 Michael Speck", OPAQUE );

    window = create_wallpaper_window( theme->title_pic, 0, width, height );
    		
    /* add a label which will show hints on the buttons */
    label = create_label( window, 101,
    					  window->width / 4, height - 60,
    					  window->width / 2, 20,
    					  theme->stan_font,
    					  ALIGN_X_CENTER | ALIGN_Y_CENTER, "" );
    /* add these buttons in a row at the right side */
    x_offset = 400;
    y_offset = 200;
    y_add = 20;
    /* campaign button */
    create_text_button( window, ID_CAMP_MENU, theme->menu_text_button,
    	x_offset, y_offset + y_add * 0, 160, 20, theme->stan_font,
    	"Play Campaign" );
    /* scenario button */
    create_text_button( window, ID_SCEN_MENU, theme->menu_text_button,
    	x_offset, y_offset + y_add * 1, 160, 20, theme->stan_font,
    	"Play Scenario" );
    /* load button */
    create_text_button( window, ID_LOAD_MENU, theme->menu_text_button,
    	x_offset, y_offset + y_add * 2, 160, 20, theme->stan_font,
    	"Load Game" );
    /* editor button */
    create_text_button( window, ID_EDIT_MENU, theme->menu_text_button,
    	x_offset, y_offset + y_add * 3, 160, 20, theme->stan_font,
    	"Lauch Editor" )->active = 0;
    /* package credits */
    create_text_button( window, ID_PACK_MENU, theme->menu_text_button,
    	x_offset, y_offset + y_add * 4, 160, 20, theme->stan_font,
    	"Packages" );
    /* options */
    create_text_button( window, ID_OPTIONS, theme->menu_text_button,
    	x_offset, y_offset + y_add * 5, 160, 20, theme->stan_font,
    	"Options" );
    /* quit */
    create_text_button( window, ID_CANCEL, theme->menu_text_button,
    	x_offset, y_offset + y_add * 6, 160, 20, theme->stan_font,
    	"Quit LGeneral" );
    /* gimmick :-) */
    create_simple_button( window, 1000, 0, 60, 320, 60, 75 );
    create_simple_button( window, 1001, 0, 130, 300, 60, 70 );
    create_simple_button( window, 1002, 0, 200, 300, 45, 65 );
    create_simple_button( window, 1003, 0, 255, 285, 40, 60 );
    	
    return window;
}

/*
====================================================================
Open title window: Draw wallpaper, reset window position and open.
====================================================================
*/
void open_title_window( Engine *engine, Window *title_window )
{
    draw_title_wallpaper( engine->theme );
    open_window( title_window, NO_REFRESH  );
    add_refresh_rect( 0, 0, sdl.screen->w, sdl.screen->h );
}
/*
====================================================================
Run the title menu and execute submenus if wanted.
====================================================================
*/
void run_title_window( Engine *engine )
{
	Window *title_window = create_title_window( engine->theme );
	Window *sel_window = create_sel_window( engine->theme );
	SDL_Event event;
	int leave = 0;
	Window *child = 0;
	int wheel_count = 0;
	Video_Mode old_mode = config.video_mode;
	Video_Mode new_mode = old_mode;
	
	open_title_window( engine, title_window );
	while ( !leave ) {
		if ( SDL_WaitEvent( &event ) )
			handle_window_event( title_window, &event, &child );
		/* check events */
		if ( child )
			switch ( child->id ) {
			    case ID_LOAD_MENU:
			        run_save_menu( engine, ID_LOAD_MENU );
			        if ( engine->scen_loaded ) {
			            run_game( engine );
    			        /* redraw title */
    			        if ( old_mode.id != config.video_mode.id ) {
                            old_mode = config.video_mode;
                            delete_window( title_window );
                            delete_window( sel_window );
                            title_window = create_title_window( engine->theme );
                            sel_window = create_sel_window( engine->theme );
    			        }
    			        open_title_window( engine, title_window );
    			    }
			        break;
			    case ID_SCEN_MENU:
			        run_sel_window( engine, sel_window, AS_SCEN );
			        /* redraw title */
			        if ( old_mode.id != config.video_mode.id ) {
			            old_mode = config.video_mode;
			            delete_window( title_window );
			            delete_window( sel_window );
                        title_window = create_title_window( engine->theme );
                        sel_window = create_sel_window( engine->theme );
			        }
			        open_title_window( engine, title_window );
			        break;
			    case ID_CAMP_MENU:
			        run_sel_window( engine, sel_window, AS_CAMP );
			        /* redraw title */
			        if ( old_mode.id != config.video_mode.id ) {
			            old_mode = config.video_mode;
			            delete_window( title_window );
			            delete_window( sel_window );
                        title_window = create_title_window( engine->theme );
                        sel_window = create_sel_window( engine->theme );
			        }
			        open_title_window( engine, title_window );
			        break;
			    case ID_PACK_MENU:
			        run_sel_window( engine, sel_window, AS_PACK );
			        break;
			    case ID_OPTIONS:
			        new_mode = config.video_mode;
			        prep_options_window( engine );
			        if ( run_options_window( engine, &new_mode ) ) {
    			        apply_options( engine );
    			        /* redraw title */
    			        if ( config.video_mode.id != new_mode.id ) {
    			            /* set new video mode */
    			            config.video_mode = new_mode;
    			            set_video_mode( config.video_mode );
    			            old_mode = config.video_mode;
    			            /* recreate title and selection window */
    			            delete_window( title_window );
    			            delete_window( sel_window );
                            title_window = create_title_window( engine->theme );
                            sel_window = create_sel_window( engine->theme );
                            /* recreate engine windows */
                            delete_windows( engine );
                            create_windows( engine );
                            /* show menu again */
        			        open_title_window( engine, title_window );
    			        }
    			    }
			        break;
				case ID_CANCEL:
					/* leave game after confirmation */
					set_simple_window_text(
						get_child( engine->conf_window, ID_UPPER_LABEL ),
						"Do you really want to quit?", NO_REFRESH );
					set_simple_window_text(
						get_child( engine->conf_window, ID_LOWER_LABEL ),
						"", NO_REFRESH );
					if ( run_stan_window( engine->conf_window ) == ID_ACCEPT )
						leave = 1;
					break;
                case 1000:	
                case 1001:
                case 1002:
                case 1003:
                    child->active = 0;
                    wheel_count++;
                    if ( wheel_count < 4 )
                        printf( "%i. more to go...\n", 4 - wheel_count );
                    else
                        printf( "Congratulations, you did it!\n"
                                "You've won... nothing!\n"
                                "Remember those funny quests in Quake? ;-D\n" );
                    break;
			}
        update_window( title_window, IF_NEEDED );
        refresh_rects();
	}
	
	delete_window( title_window );
	delete_window( sel_window );
}

