/*
 * lftp and utils
 *
 * Copyright (c) 1998 by Alexander V. Lukyanov (lav@yars.free.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <config.h>
#include "buffer.h"
#include "xmalloc.h"
#include "FileAccess.h"

#define BUFFER_INC (8*1024) // should be power of 2

void Buffer::Get(const char **buf,int *size)
{
   if(in_buffer==0)
   {
      *size=0;
      if(eof)
	 *buf=0;
      else
	 *buf="";
      return;
   }
   *buf=buffer+buffer_ptr;
   *size=in_buffer;
}

void Buffer::GetSaved(const char **buf,int *size) const
{
   if(!save)
   {
      *size=0;
      *buf=0;
      return;
   }
   *buf=buffer;
   *size=buffer_ptr+in_buffer;
}

void Buffer::SaveRollback(long p)
{
   if(buffer_ptr<p)
      save=false;
   if(!save)
   {
      buffer_ptr=0;
      in_buffer=0;
   }
   else
   {
      buffer_ptr=p;
      in_buffer=0;
   }
}

void Buffer::Allocate(int size)
{
   if(in_buffer==0 && !save)
      buffer_ptr=0;

   int in_buffer_real=in_buffer;
   if(save)
      in_buffer_real+=buffer_ptr;

   if(buffer_allocated<in_buffer_real+size)
   {
      buffer_allocated=(in_buffer_real+size+(BUFFER_INC-1)) & ~(BUFFER_INC-1);
      buffer=(char*)xrealloc(buffer,buffer_allocated);
   }
   // could be round-robin, but this is easier
   if(!save && buffer_ptr+in_buffer+size>buffer_allocated)
   {
      memmove(buffer,buffer+buffer_ptr,in_buffer);
      buffer_ptr=0;
   }
}

void Buffer::SaveMaxCheck(int size)
{
   if(save && buffer_ptr+size>save_max)
      save=false;
}

void Buffer::Put(const char *buf,int size)
{
   SaveMaxCheck(size);

   if(in_buffer==0 && !save)
   {
      buffer_ptr=0;

      if(size>=1024)
      {
	 int res=Put_LL(buf,size);
	 if(res>=0)
	 {
	    buf+=res;
	    size-=res;
	    pos+=res;
	 }
      }
   }

   if(size==0)
      return;

   Allocate(size);

   memcpy(buffer+buffer_ptr+in_buffer,buf,size);
   in_buffer+=size;
   pos+=size;
}

void Buffer::Skip(int len)
{
   if(len>in_buffer)
      len=in_buffer;
   in_buffer-=len;
   buffer_ptr+=len;
   pos+=len;
}
void Buffer::UnSkip(int len)
{
   if(len>buffer_ptr)
      len=buffer_ptr;
   in_buffer+=len;
   buffer_ptr-=len;
   pos-=len;
}

void Buffer::Empty()
{
   in_buffer=0;
   buffer_ptr=0;
   if(save_max>0)
      save=true;
}

int Buffer::Do()
{
   return STALL;
}

Buffer::Buffer()
{
   error_text=0;
   saved_errno=0;
   buffer=0;
   buffer_allocated=0;
   in_buffer=0;
   buffer_ptr=0;
   eof=false;
   broken=false;
   save=false;
   save_max=0;
   pos=0;
}
Buffer::~Buffer()
{
   xfree(error_text);
   xfree(buffer);
}

void Buffer::SetError(const char *e)
{
   xfree(error_text);
   error_text=xstrdup(e);
}

// FileOutputBuffer implementation
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
FileOutputBuffer::FileOutputBuffer(FDStream *o)
{
   out=o;
   event_time=now;
}
FileOutputBuffer::~FileOutputBuffer()
{
   delete out;
}
int FileOutputBuffer::Do()
{
   if(Done() || Error())
      return STALL;
   if(in_buffer==0)
      return STALL;
   int res=Put_LL(buffer+buffer_ptr,in_buffer);
   if(res>0)
   {
      in_buffer-=res;
      buffer_ptr+=res;
      event_time=now;
      return MOVED;
   }
   if(res<0)
   {
      event_time=now;
      return MOVED;
   }
   int fd=out->getfd();
   if(fd>=0)
      Block(fd,POLLOUT);
   else
      TimeoutS(1);
   return STALL;
}
int FileOutputBuffer::Put_LL(const char *buf,int size)
{
   if(out->broken())
   {
      broken=true;
      return -1;
   }

   int res;
   int fd=out->getfd();
   if(fd==-1)
   {
      if(out->error())
	 goto out_err;
      event_time=now;
      return 0;
   }

   res=write(fd,buf,size);
   if(res==-1)
   {
      if(errno==EAGAIN || errno==EINTR)
	 return 0;
      if(errno==EPIPE)
      {
	 broken=true;
	 return -1;
      }
      saved_errno=errno;
      out->MakeErrorText();
      goto out_err;
   }
   return res;

out_err:
   SetError(out->error_text);
   return -1;
}

FgData *FileOutputBuffer::GetFgData(bool fg)
{
   if(out->getfd()!=-1)
      return new FgData(out->GetProcGroup(),fg);
   return 0;
}

bool FileOutputBuffer::Done()
{
   if(broken || Error()
   || (eof && Buffer::Done()))
      return out->Done(); // out->Done indicates if sub-process finished
   return false;
}

time_t FileOutputBuffer::EventTime()
{
   return event_time;
}

// FileInputBuffer implementation
#undef super
#define super Buffer
FileInputBuffer::FileInputBuffer(FDStream *i)
{
   in=i;
   in_FA=0;
   event_time=now;
}
FileInputBuffer::FileInputBuffer(FileAccess *i)
{
   in=0;
   in_FA=i;
   event_time=now;
}
FileInputBuffer::~FileInputBuffer()
{
   if(in)
      delete in;
   if(in_FA)
      in_FA->Close();
}
int FileInputBuffer::Do()
{
   if(Done() || Error())
      return STALL;

   int res=Get_LL(GET_BUFSIZE);
   if(res>0)
   {
      in_buffer+=res;
      SaveMaxCheck(0);
      event_time=now;
      return MOVED;
   }
   if(res<0)
   {
      event_time=now;
      return MOVED;
   }
   if(eof)
   {
      event_time=now;
      return MOVED;
   }
   if(in)
   {
      int fd=in->getfd();
      if(fd>=0)
	 Block(fd,POLLIN);
      else
	 Timeout(1000);
   }
   return STALL;
}
int FileInputBuffer::Get_LL(int size)
{
   int res=0;

   if(in_FA)
   {
      Allocate(size);

      res=in_FA->Read(buffer+buffer_ptr+in_buffer,size);
      if(res<0)
      {
	 if(res==FA::DO_AGAIN)
	    return 0;
	 SetError(in_FA->StrError(res));
	 return -1;
      }
   }
   else if(in)
   {
      int fd=in->getfd();
      if(fd==-1)
      {
	 if(in->error())
	    goto in_err;
	 return 0;
      }

      Allocate(size);

      res=read(fd,buffer+buffer_ptr+in_buffer,size);
      if(res==-1)
      {
	 if(errno==EAGAIN || errno==EINTR)
	    return 0;
	 saved_errno=errno;
	 in->MakeErrorText();
	 goto in_err;
      }
   }
   if(res==0)
      eof=true;
   return res;

in_err:
   SetError(in->error_text);
   return -1;
}

FgData *FileInputBuffer::GetFgData(bool fg)
{
   if(!in)
      return 0;
   if(in->getfd()!=-1)
      return new FgData(in->GetProcGroup(),fg);
   return 0;
}

bool FileInputBuffer::Done()
{
   if(broken || Error() || eof)
   {
      if(in_FA)
	 return true;
      return in->Done(); // in->Done indicates if sub-process finished
   }
   return false;
}

time_t FileInputBuffer::EventTime()
{
   if(suspended)
      return now;
   return event_time;
}

void FileInputBuffer::Suspend()
{
   if(in_FA)
      in_FA->Suspend();
   super::Suspend();
}
void FileInputBuffer::Resume()
{
   super::Resume();
   if(in_FA)
      in_FA->Resume();
}
