/*
fetchnews -- post articles to and get news from upstream server(s)

Written by Arnt Gulbrandsen <agulbra@troll.no> and copyright 1995
Troll Tech AS, Postboks 6133 Etterstad, 0602 Oslo, Norway, fax +47
22646949.
Modified by Cornelius Krasel <krasel@wpxx02.toxi.uni-wuerzburg.de>
and Randolf Skerka <Randolf.Skerka@gmx.de>.
Copyright of the modifications 1997.
Modified by Kent Robotti <robotti@erols.com>. Copyright of the
modifications 1998.
Modified by Markus Enzenberger <enz@cip.physik.uni-muenchen.de>.
Copyright of the modifications 1998.
Modified by Cornelius Krasel <krasel@wpxx02.toxi.uni-wuerzburg.de>.
Copyright of the modifications 1998, 1999.
Modified by Ralf Wildenhues <ralf.wildenhues@gmx.de>.
Copyright of the modifications 2002.
Modified by Jonathan Larmour <jifl@jifvik.org>.
Copyright of the modifications 2002.
Modified by Richard van der Hoff <richard@rvanderhoff.org.uk>
Copyright of the modifications 2002.
Enhanced and modified by Matthias Andree <matthias.andree@web.de>.
Copyright of the modifications 2000 - 2003.

See file COPYING for restrictions on the use of this software.
*/

#include "leafnode.h"
#include "fetchnews.h"
#include "mastring.h"
#include "ln_log.h"
#include "mysigact.h"

#include <sys/types.h>
#include <ctype.h>
#include "system.h"
#include <fcntl.h>
#include <netdb.h>
#include <netinet/in.h>
#include <setjmp.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <syslog.h>
#include <sys/resource.h>
#include <unistd.h>
#include <utime.h>
#include <sys/wait.h>

#include "groupselect.h"

int verbose = 0;
int debug = 0;

int artno;

time_t now;

/* Variables set by command-line options which are specific for fetch */
unsigned long extraarticles = 0;
int usesupplement = 0;
int postonly = 0;		/* if 1, don't read files from upstream server */
int noexpire = 0;		/* if 1, don't automatically unsubscribe newsgroups */
int forceactive = 0;		/* if 1, reread complete active file */

static sigjmp_buf jmpbuffer;
static volatile sig_atomic_t canjump;

static int isgrouponserver(const struct server *current_server, char *newsgroups /* will be modified */);
static int ismsgidonserver(const struct server *current_server, char *msgid);
static int age( /*@null@*/ const char *date);
#ifdef NOT_YET
static void supersede(const char *msgid);
#endif
static unsigned long getgroup(const struct server *current_server,
	/*@null@*/ struct newsgroup *g,
	unsigned long server);
static int postarticles(const struct server *current_server);

static void
ignore_answer(FILE * f)
{
    char *l;
    while (((l = getaline(f)) != NULL) && strcmp(l, "."));
}

static RETSIGTYPE
sig_int(int signo)
{
    if (canjump == 0)
	return;		/* ignore unexpected signals */
    if (signo == SIGINT || signo == SIGTERM) {
	canjump = 0;
	mysigact(SIGALRM, 0, SIG_IGN, 0);	/* do not let mgetaline timeout interfere! */
	alarm(0);
	siglongjmp(jmpbuffer, 1);
    }
}

static void
usage(void)
{
    fprintf(stderr, "Usage: fetchnews [-v] [-x #] [-l] [-n] [-f] [-P]\n"
	    "  -v: more verbose (may be repeated)\n"
	    "  -x: check for # extra articles in each group\n"
	    "  -l: do not use supplementary servers\n"
	    "  -n: do not automatically expire unread groups\n"
	    "  -f: force reload of groupinfo file\n"
	    "  -P: only post outgoing articles, don't fetch any\n");
}

/*
 * check whether any of the newsgroups is on server
 * return TRUE if yes, FALSE otherwise
 */
int
isgrouponserver(const struct server *current_server, char *newsgroups)
{
    char *p, *q;
    int retval;

    if (!newsgroups)
	return FALSE;

    retval = FALSE;
    p = newsgroups;
    do {
	q = strchr(p, ',');
	if (q)
	    *q++ = '\0';
	if (gs_match(current_server->group_pcre, p) == 1) {
	    xsnprintf(lineout, SIZE_lineout, "GROUP %s\r\n", p);
	    putaline();
	    if (nntpreply(current_server) == 211)
		retval = TRUE;
	    p = q;
	    while (p && *p && isspace((unsigned char)*p))
		p++;
	}
    } while (q && !retval);

    return retval;
}

/*
 * check whether message-id is on server
 * return TRUE if yes, FALSE otherwise
 */
int
ismsgidonserver(const struct server *current_server, char *msgid)
{
    int r;
    if (!msgid)
	return FALSE;
    xsnprintf(lineout, SIZE_lineout, "%s %s\r\n",
	      stat_is_evil ? "HEAD" : "STAT", msgid);
    putaline();
    r = nntpreply(current_server);
    if (r >= 220 && r <= 223) {
	if (stat_is_evil)
	    ignore_answer(nntpin);
	return TRUE;
    } else
	return FALSE;
}

int
age( /*@null@*/ const char *date)
{
    char monthname[4]; /* RATS: ignore */
    int month;
    int year;
    int day;
    const char *d;
    time_t tmp;
    struct tm time_struct;

    if (!date)
	return 1000;		/* large number: OLD */
    d = date;
    if (!(strncasecmp(d, "date:", 5)))
	d += 5;
    while (isspace((unsigned char)*d))
	d++;

    if (isalpha((unsigned char)*d)) {
	while (*d && !isspace((unsigned char)*d))	/* skip "Mon" or "Tuesday," */
	    d++;
    }

    /* RFC 822 says we have 1*LWSP-char between tokens */
    while (isspace((unsigned char)*d))
	d++;

    /* parsing with sscanf leads to crashes */
    day = atoi(d);
    while (isdigit((unsigned char)*d) || isspace((unsigned char)*d))
	d++;
    if (!isalpha((unsigned char)*d)) {
	syslog(LOG_INFO, "Unable to parse %s", date);
	return 1003;
    }
    monthname[0] = *d++;
    monthname[1] = *d++;
    monthname[2] = *d++;
    monthname[3] = '\0';
    if (strlen(monthname) != 3) {
	syslog(LOG_INFO, "Unable to parse month in %s", date);
	return 1004;
    }
    while (isalpha((unsigned char)*d))
	d++;
    while (isspace((unsigned char)*d))
	d++;
    year = atoi(d);

    if ((year < 1970) && (year > 99)) {
	syslog(LOG_INFO, "Unable to parse year in %s", date);
	return 1005;
    } else if (!(day > 0 && day < 32)) {
	syslog(LOG_INFO, "Unable to parse day in %s", date);
	return 1006;
    } else {
	if (!strcasecmp(monthname, "jan"))
	    month = 0;
	else if (!strcasecmp(monthname, "feb"))
	    month = 1;
	else if (!strcasecmp(monthname, "mar"))
	    month = 2;
	else if (!strcasecmp(monthname, "apr"))
	    month = 3;
	else if (!strcasecmp(monthname, "may"))
	    month = 4;
	else if (!strcasecmp(monthname, "jun"))
	    month = 5;
	else if (!strcasecmp(monthname, "jul"))
	    month = 6;
	else if (!strcasecmp(monthname, "aug"))
	    month = 7;
	else if (!strcasecmp(monthname, "sep"))
	    month = 8;
	else if (!strcasecmp(monthname, "oct"))
	    month = 9;
	else if (!strcasecmp(monthname, "nov"))
	    month = 10;
	else if (!strcasecmp(monthname, "dec"))
	    month = 11;
	else {
	    syslog(LOG_INFO, "Unable to parse %s", date);
	    return 1001;
	}
	if (year < 70)		/* years 2000-2069 in two-digit form */
	    year += 100;
	else if (year > 1970)	/* years > 1970 in four-digit form */
	    year -= 1900;

	memset(&time_struct, 0, sizeof(time_struct));
	time_struct.tm_sec = 0;
	time_struct.tm_min = 0;
	time_struct.tm_hour = 0;
	time_struct.tm_mday = day;
	time_struct.tm_mon = month;
	time_struct.tm_year = year;
	time_struct.tm_isdst = 0;

	tmp = mktime(&time_struct);

	if (tmp == -1)
	    return 1002;

	return ((now - tmp) / SECONDS_PER_DAY);
    }
}

#ifdef NOT_YET
/*
 * delete a message
 */
static void
supersede(const char *msgid)
{
    const char *filename;
    /*@dependent@*/ char *p, *q, *r;
    char *hdr;
    struct stat st;

    filename = lookup(msgid);
    if (!filename)
	return;

    hdr = getheader(filename, "Xref:");
    if (!hdr)
	return;
    p = hdr;

    /* jump hostname entry */
    while (*p && !isspace((unsigned char)*p))
	p++;
    SKIPLWS(p);
    /* now p points to the first newsgroup */

    /* unlink all the hardlinks in the various newsgroups directories */
    while (p && ((q = strchr(p, ':')) != NULL)) {
	*q++ = '\0';
	if (chdirgroup(p, FALSE)) {
	    r = p;
	    p = q;
	    while (*q && !isspace((unsigned char)*q))
		q++;
	    *q = '\0';
#ifdef NOTYET
	    if (unlink(p))
		syslog(LOG_ERR, "Failed to unlink %s: %s: %m", r, p);
	    else {
#endif
		if (verbose > 2)
		    printf("Superseded %s in %s\n", p, r);
		syslog(LOG_INFO, "Superseded %s in %s", p, r);
#ifdef NOTYET
	    }
#endif
	}
	while (*p && !isspace((unsigned char)*p))
	    p++;
	SKIPLWS(p);
    }

    /* unlink the message-id hardlink */
    if (stat(filename, &st))
	syslog(LOG_ERR, "%s does not exist: %m", filename);
    else if (st.st_nlink > 1)
	syslog(LOG_ERR, "%s: link count is %d", filename, (int)st.st_nlink);
#ifdef NOTYET
    else if (unlink(filename))
	syslog(LOG_ERR, "Failed to unlink %s: %m", filename);
#endif
    else {
	if (verbose > 2)
	    printf("Superseded %s\n", filename);
	syslog(LOG_INFO, "Superseded %s", filename);
    }
    free(hdr);
}
#endif

/*
 * Get body of a single message of which the header has already been
 * downloaded and append it to the file with the header.
 * Returns 0 if file could not be retrieved, 1 otherwise.
 */
static int
getbody_insitu(const struct server *current_server, struct newsgroup *group, unsigned long id)
{
    const char *c;
    int rc = 0;
    char *l;
    char *messageid;
    FILE *f;
    char s[SIZE_s+1];
    off_t pos;

    if (!chdirgroup(group->name, FALSE))
	return 0;

    /* extract message-id: header */
    xsnprintf(s, SIZE_s, "%lu", id);
    messageid = getheader(s, "Message-ID:");
    if (!messageid)
	return 0;

    /* check whether we can retrieve the body */
    if (verbose > 2)
	printf("%s: BODY %s\n", group->name, messageid);
    xsnprintf(lineout, SIZE_lineout, "BODY %s\r\n", messageid);
    putaline();

    if (nntpreply(current_server) != 222) {
	syslog(LOG_ERR, "%s: Retrieving body %s failed. No response",
	       group->name, messageid);
	rc = 0;
	goto getbody_bail;
    }

    xsnprintf(s, SIZE_s, "%lu", id);
    c = lookup(messageid);
    if (!(f = fopen(c, "a"))) {
	syslog(LOG_ERR, "%s: cannot open %s for appending", group->name, c);
	rc = 0;
	goto getbody_bail;
    }
    pos = ftell(f);
    fputc('\n', f); /* blank line -- separate header and body */

    debug--;
    while (((l = getaline(nntpin)) != NULL) && strcmp(l, ".") && !ferror(f)) {
	if (*l == '.')
	    ++l;
	fputs(l, f);
	fputc('\n', f);
    }
    debug = debugmode;

    if (!l) {
	syslog(LOG_ERR, "%s: Retrieving body %s failed. "
	       "Transmission interrupted.", group->name, messageid);
	ftruncate(fileno(f), pos);
	fclose(f);
	rc = 0;
	goto getbody_bail;
    }

    /* abort when disk is full */
    if (fclose(f) && errno == ENOSPC) {
	truncate(s, pos);
	raise(SIGINT);
	return 0;
    }

    rc = 1;

  getbody_bail:
    if (messageid)
	free(messageid);
    return rc;
}

static int
getbody_newno(const struct server *current_server, struct newsgroup *group, unsigned long id)
{
    const char *c;
    int rc = 0;
    char *l;
    char *p, *q;
    char *messageid;
    char *newsgroups;		/* I hope this is enough */
    char *xref;
    FILE *f, *g;
    char s[SIZE_s+1];

    if (!chdirgroup(group->name, FALSE))
	return 0;

    /* extract message-id: and xref: headers */
    xsnprintf(s, SIZE_s, "%lu", id);
    if (!(f = fopen(s, "r"))) {
	syslog(LOG_INFO, "%s: cannot open %s for reading -- possibly expired",
	       group->name, s);
	return 1;		/* pretend to have read file successfully so that
				   it is purged from the list */
    }
    messageid = NULL;
    newsgroups = NULL;
    xref = NULL;
    debug--;
    while ((l = getaline(f)) != NULL) {
	if (!newsgroups && !strncmp(l, "Newsgroups: ", 12))
	    newsgroups = critstrdup(l + 12, "getbody");
	if (!messageid && !strncmp(l, "Message-ID: ", 12))
	    messageid = critstrdup(l + 12, "getbody");
	if (!xref && !strncmp(l, "Xref: ", 6))
	    xref = critstrdup(l + 6, "getbody");
    }
    debug = debugmode;
    fclose(f);

    /* check whether we can retrieve the body */
    if (verbose > 2)
	printf("%s: BODY %s\n", group->name, messageid);
    xsnprintf(lineout, SIZE_lineout, "BODY %s\r\n", messageid);
    putaline();

    if (nntpreply(current_server) != 222) {
	syslog(LOG_ERR, "%s: Retrieving body %s failed. No response",
	       group->name, messageid);
	rc = 0;
	goto getbody_bail;
    }

    xsnprintf(s, SIZE_s, "%lu", id);
    c = lookup(messageid);
    unlink(c);			/* make space for new file */

    if (!(f = fopen(c, "w"))) {
	syslog(LOG_ERR, "%s: cannot open %s for writing", group->name, c);
	link(s, c);		/* if we can't open new file restore old one */
	rc = 0;
	goto getbody_bail;
    }

    /* copy all headers except Xref: into new file */
    g = fopen(s, "r");
    if (!g) {
	syslog(LOG_ERR, "%s: open %s failed", group->name, s);
	rc = 0;
	goto getbody_bail;
    }
    debug--;
    while ((l = getaline(g)) != NULL) {
	/* skip xref: headers */
	if (strncasecmp(l, "Xref: ", 6) != 0)
	    fprintf(f, "%s\n", l);
    }
    debug = debugmode;
    fclose(g);

    /* create a whole bunch of new hardlinks */
    store(c, f, newsgroups, messageid);

    /* retrieve body */
    fprintf(f, "\n");		/* Empty line between header and body. */
    debug--;
    while (((l = getaline(nntpin)) != NULL) && strcmp(l, ".") && !ferror(f)) {
	if (*l == '.')
	    ++l;
	fputs(l, f);
	fputc('\n', f);
    }
    debug = debugmode;
    if (!l) {
	syslog(LOG_ERR, "%s: Retrieving body %s failed. "
	       "Transmission interrupted.", group->name, messageid);
	fprintf(f, "\n\t[ Leafnode: ]\n"
		"\t[ An error occured while " "retrieving this message. ]\n");
	fclose(f);
	rc = 0;
	goto getbody_bail;
    }
    /* abort when disk is full */
    if (fclose(f) && errno == ENOSPC)
	raise(SIGINT);

    /* Remove old article files since we don't need them anymore.
       This is done by evaluating the old Xref: header.
     */
    if (!xref) {
	/* no Xref: header --> make a fake one */
	xref = critmalloc(50 + strlen(fqdn) + strlen(group->name), "getbody");
	sprintf(xref, "%s %s:%lu", fqdn, group->name, id);	/* RATS: ignore */
    }

    if (debugmode)
	syslog(LOG_DEBUG, "xref: %s", xref);
    c = strchr(xref, ' ');
#ifdef __LCLINT__
    assert(c != NULL);		/* we know c != NULL */
#endif				/* __LCLINT__ */
    while ((c++) && (*c) && (q = strchr(c, ':')) != NULL) {
	*q++ = '\0';
	if ((p = strchr(q, ' ')) != NULL)
	    *p = '\0';

	/* c points to the newsgroup, q to the article number */
	if (!chdirgroup(c, FALSE)) {
	    return 0;
	}
	if (unlink(q))
	    syslog(LOG_NOTICE,
		   "retrieved body, but unlinking headers-only file %s/%s failed",
		   c, q);
	else if (debugmode)
	    syslog(LOG_DEBUG,
		   "retrieved body, now unlinking headers-only file %s/%s", c,
		   q);

	c = strchr(q, ' ');
    }
    rc = 1;
  getbody_bail:
    if (xref)
	free(xref);
    if (messageid)
	free(messageid);
    if (newsgroups)
	free(newsgroups);
    return rc;
}

static int
getbody(const struct server *cs, struct newsgroup *group, unsigned long id) {
    static int (*func)(const struct server *, struct newsgroup *, unsigned long);

    if (!func) {
	func = db_situ ? getbody_insitu : getbody_newno;
    }

    return func(cs, group, id);
}

/*
 * Get bodies of messages that have marked for download.
 * The group must already be selected at the remote server and
 * the current directory must be the one of the group.
 */
static void
getmarked(const struct server *current_server, struct newsgroup *group)
{
    int n, i;
    int had_bodies = 0;
    FILE *f;
    mastr *filename = mastr_new(PATH_MAX);
    unsigned long id[BODY_DOWNLOAD_LIMIT]; /* RATS: ignore */
    char *t;

    /* #1 read interesting.groups file */
    n = 0;
    mastr_vcat(filename, spooldir, "/interesting.groups/", group->name, NULL);
    if (!(f = fopen(mastr_str(filename), "r")))
	syslog(LOG_ERR, "Cannot open %s for reading", mastr_str(filename));
    else {
	struct stat st;
	if (fstat(fileno(f), &st) == 0 && st.st_size > 0) {
	    had_bodies = 1;
	    if (verbose)
		printf("%s: getting bodies of marked messages...\n",
			group->name);
	    while ((t = getaline(f)) && n < BODY_DOWNLOAD_LIMIT) {
		if (sscanf(t, "%lu", &id[n]) == 1)
		    ++n;
	    }
	}
	fclose(f);
    }
    /* #2 get bodies */
    if (delaybody || had_bodies) {
	syslog(LOG_INFO, "%s: marked bodies %d", group->name, n);
	if (verbose > 1)
	    printf("%s: marked bodies %d\n", group->name, n);
    }
    for (i = 0; i < n; ++i)
	if (getbody(current_server, group, id[i]))
	    id[i] = 0;

    /* #3 write back ids of all articles which could not be retrieved */
    if (had_bodies) {
	if (!(f = fopen(mastr_str(filename), "w")))
	    syslog(LOG_ERR, "Cannot open %s for writing", mastr_str(filename));
	else {
	    for (i = 0; i < n; ++i)
		if (id[i] != 0)
		    fprintf(f, "%lu\n", id[i]);
	    fclose(f);
	}
    }
    if (delaybody || had_bodies) {
	if (verbose)
	    printf("%s: Done getting article bodies.\n", group->name);
    }
    mastr_delete(filename);
}

/*
 * get newsgroup from a server. "server" is the last article that
 * was previously read from this group on that server
 */
static unsigned long
getgroup(const struct server *current_server, struct newsgroup *g, unsigned long server)
{
#define HD_MAX 10
    static char *hd[HD_MAX];
    const char *hnames[HD_MAX] = { "Path: ", "Message-ID: ", "From: ",
	"Newsgroups: ", "Subject: ", "Date: ",
	"References: ", "Lines: ", "Xref: ", ""
    };

    /* order of headers in XOVER */
    enum enames { h_sub = 1, h_fro, h_dat, h_mid, h_ref, h_byt, h_lin };

    unsigned long fetched, killed;
    unsigned long h;
    long n;
    unsigned long last;
    unsigned long window;	/* last ARTICLE n command sent */
    char *l;
    FILE *f;
    const char *c;
    struct stat st;
    long outstanding = 0;
    unsigned long i;
    static /*@only@*/ unsigned long *stufftoget;
    int localmaxage = maxage;
    int maxagelimit = -1;
    const char *limitfrom = "";
    int exp;

    if (!g)
	return server;

    if (g->first > g->last && g->first - g->last > 1)
	g->last = g->first - 1;

    if ((exp = lookup_expiredays(g->name)) > 0) {
	if (localmaxage > exp) {
	    maxagelimit = exp;
	    limitfrom = "groupexpire";
	}
    } else {
	if (localmaxage > expiredays) {
	    maxagelimit = expiredays;
	    limitfrom = "global expire";
	}
    }

    if (*limitfrom && localmaxage > maxagelimit) {
	if (clamp_maxage) {
	    syslog(LOG_NOTICE, "clamping maxage for %s to %s %d",
		   g->name, limitfrom, maxagelimit);
	    localmaxage = maxagelimit;
	} else {
	    fprintf(stderr,
		   "warning: group %s: maxage of %d is inappropriate for your "
		   "applicable %s of %d. This can cause excessive downloads of "
		   "articles that were previously downloaded and expired. "
		   "Fix your configuration.\n", g->name, localmaxage, limitfrom,
		   maxagelimit);
	    syslog(LOG_WARNING,
		   "warning: group %s: maxage of %d is inappropriate for your "
		   "applicable %s of %d. This can cause excessive downloads of "
		   "articles that were previously downloaded and expired. "
		   "Fix your configuration.", g->name, localmaxage, limitfrom,
		   maxagelimit);
	}
    }

    /* skip */
    if (gs_match(current_server->group_pcre, g->name) != 1) {
	if (verbose) {
	    printf("skipped %s, not in only_groups_pcre\n", g->name);
	}
	return server;
    }

    xsnprintf(lineout, SIZE_lineout, "GROUP %s\r\n", g->name);
    putaline();

    l = getaline(nntpin);
    if (!l)
	return server;

    if (sscanf(l, "%3ld", &n) && n == 480) {
	if (authenticate(current_server)) {
	    xsnprintf(lineout, SIZE_lineout, "GROUP %s\r\n", g->name);
	    putaline();
	    l = getaline(nntpin);
	    if (!l)
		return server;
	} else
	    return server;
    }

    if (debug == 1)
	syslog(LOG_DEBUG, "<%s", l);

    if (n == 411) {		/* group not available on server */
	if (verbose > 1)
	    printf("%s: no such group\n", g->name);
	return 0;
    }

    if (sscanf(l, "%3ld %lu %lu %lu ", &n, &h, &window, &last) < 4 || n != 211)
    {
	printf("%s: cannot parse server reply \"%s\"\n", g->name, l);
	syslog(LOG_WARNING, "%s: cannot parse server reply \"%s\"", g->name, l);
	return server;
    }

    if (h == 0) {
	if (verbose > 1)
	    printf("%s: upstream group is empty\n", g->name);
	return server;
    }

    if (extraarticles) {
	if (server > extraarticles)
	    i = server - extraarticles;
	else
	    i = 0;
	if (i < window)
	    i = window;
	if (verbose > 1)
	    printf("%s: backing up from %lu to %lu\n", g->name, server, i);
	server = i;
    }

    if (server > last + 1) {
	syslog(LOG_INFO,
	       "%s: last seen article was %lu, server now has %lu-%lu",
	       g->name, server, window, last);
	if (server > last + 5) {
	    if (verbose)
		printf("switched upstream servers? %lu > %lu\n",
		       server - 1, last);
	    server = window;	/* insane - recover thoroughly */
	} else {
	    if (verbose)
		printf("rampant spam cancel? %lu > %lu\n", server - 1, last);
	    server = last - 5;	/* a little bit too much */
	}
    }

    if (initiallimit && server == 1 && last > server
	&& last - server > initiallimit) {
	if (verbose > 1)
	    printf("skipping articles %lu-%lu inclusive (initial limit)\n",
		   server, last - initiallimit);
	syslog(LOG_INFO, "skipping articles %lu-%lu inclusive (initial limit)",
	       server, last - initiallimit);
	server = last - initiallimit + 1;
    }

    if (artlimit && last > server && last - server > artlimit) {
	if (verbose > 1)
	    printf("skipping articles %lu-%lu inclusive (article limit)\n",
		   server, last - artlimit - 1);
	syslog(LOG_INFO, "skipping articles %lu-%lu inclusive (article limit)",
	       server, last - artlimit - 1);
	server = last - artlimit;
    }

    getmarked(current_server, g);

    if (window < server)
	window = server;
    if (window < 1)
	window = 1;
    server = window;

    if (server > last) {
	if (verbose > 1)
	    printf("%s: no new articles\n", g->name);
	syslog(LOG_INFO, "%s: no new articles\n", g->name);
	return server;
    }

    if (verbose > 1)
	printf("%s: considering articles %lu - %lu\n", g->name, server, last);
    syslog(LOG_INFO, "%s: considering articles %lu - %lu\n", g->name, server,
	   last);

    fetched = 0;
    killed = 0;

    stufftoget =
	(unsigned long *)realloc(stufftoget, sizeof(stufftoget[0]) * (last + 1 - server));
    if (!stufftoget) {
	printf("Not enough memory for XHDRs.\r\n");
	syslog(LOG_ERR, "not enough memory for XHDRs");
	return server;
    }
    memset(stufftoget, 0, sizeof(stufftoget[0]) * (last + 1 - server));

    xsnprintf(lineout, SIZE_lineout, "XOVER %lu-%lu\r\n", server, last);
    putaline();
    if (nntpreply(current_server) == 224) {
	debug--;
	while ((l = getaline(nntpin)) && strcmp(l, ".")) {
	    /*@dependent@*/ char *fields[HD_MAX];
	    unsigned long art;
	    const char *t;
	    char *q;
	    static mastr *ol;
	    if (!ol) ol = mastr_new(1024);

	    mastr_cpy(ol, l);
	    /* split xover */
	    /*@+loopexec@*/
	    for (i = 0; l && l[0] && i < HD_MAX; i++) {
		char *y;
		fields[i] = l;
		if ((y = strchr(l, '\t'))) {
		    y[0] = '\0';
		    l = y + 1;
		} else {
		    l = NULL;
		};
	    };
	    /*@=loopexec@*/
	    /* short line -- log and skip */
	    if (i < 8) {
		ln_log(LNLOG_SWARNING, LNLOG_CTOP,
			"%s: %s: Warning: got unparsable XOVER line from server, "
			"too few fields (%ld): \"%s\"",
			current_server->name, g->name, i, mastr_str(ol));
		continue;
	    }
	    for (; i < HD_MAX; i++)
		fields[i] = NULL;
	    art = strtoul(fields[0], &q, 10);
	    if (q && art >= server && art <= last) {
		if (maxbytes && fields[h_byt]
		    && (strtoul(fields[h_byt], NULL, 10) > maxbytes)) {
		    killed++;
		    if (verbose > 2)
			printf("%s: killed %lu (%s), "
			       "too many bytes (%lu > %lu)\n",
			       g->name, art, fields[h_mid],
			       strtoul(fields[h_byt], NULL, 10), maxbytes);
		} else if (localmaxage && fields[h_dat]
			   && (age(fields[h_dat]) > localmaxage)) {
		    killed++;
		    if (verbose > 2)
			printf("%s: killed %lu (%s), "
			       "too old (%d > %d) days\n",
			       g->name, art, fields[h_mid],
			       age(fields[h_dat]), localmaxage);
		} else if (lstat(t = lookup(fields[h_mid]), &st) == 0) {
		    killed++;
		    if (verbose > 2)
			printf("%s: killed %lu (%s), already fetched before\n",
				g->name, art, fields[h_mid]);
		} else {
		    stufftoget[outstanding] = art;
		    outstanding++;
		    if (verbose > 2)
			printf("%s: will fetch %lu (%s)\n", g->name, art, t);
		}
	    }
	}
	debug = debugmode;
	goto have_outstanding;
    }
    if (verbose)
	printf("XOVER failed, trying XHDR\n");

    xsnprintf(lineout, SIZE_lineout, "XHDR Message-ID %lu-%lu\r\n", server,
	      last);
    putaline();
    if (nntpreply(current_server) == 221) {
	debug--;
	while ((l = getaline(nntpin)) && strcmp(l, ".")) {
	    unsigned long art;
	    char *t;
	    art = strtoul(l, &t, 10);
	    if (t && isspace((unsigned char)*t)) {
		while (isspace((unsigned char)*t))
		    t++;
		if (art >= server && art <= last && stat(lookup(t), &st) != 0) {
		    stufftoget[outstanding] = art;
		    outstanding++;
		    if (verbose > 2)
			printf("%s: will fetch %lu (%s)\n", g->name, art, t);
		}
	    }
	}
	debug = debugmode;
	if (!l)
	    return server;
    } else {
	return server;
    }

    if (outstanding == 0) {
	return last + 1;
    }

    syslog(LOG_INFO, "%s: will fetch %ld articles", g->name, outstanding);

    i = 0;
    if (maxbytes) {
	xsnprintf(lineout, SIZE_lineout, "XHDR Bytes %lu-%lu\r\n", server,
		  last);
	putaline();
	if (nntpreply(current_server) == 221) {
	    while ((l = getaline(nntpin)) && strcmp(l, ".")) {
		unsigned long art, bytes = 0;
		char *t;
		art = strtoul(l, &t, 10);
		if (t)
		    bytes = strtoul(t, NULL, 10);
		if ((art == stufftoget[i]) && (bytes > maxbytes)) {
		    stufftoget[i] = 0;
		    syslog(LOG_INFO, "Killed article %lu (%lu > %lu) bytes",
			   art, bytes, maxbytes);
		    if (verbose > 2)
			printf("Killed article %lu (%lu > %lu) bytes.\n",
			       art, bytes, maxbytes);
		    killed++;
		}
		i++;
	    }
	    if (!l)		/* timeout */
		return server;
	}
    }
    i = 0;
    if (localmaxage) {
	xsnprintf(lineout, SIZE_lineout, "XHDR Date %lu-%lu\r\n", server, last);
	putaline();
	if (nntpreply(current_server) == 221) {
	    debug--;
	    while ((l = getaline(nntpin)) && strcmp(l, ".")) {
		unsigned long art;
		long aage = 0;
		char *t;

		art = strtoul(l, &t, 10);
		if (t)
		    aage = age(t);
		if ((art == stufftoget[i]) && (aage > localmaxage)) {
		    stufftoget[i] = 0;
		    syslog(LOG_INFO,
			   "Killed article %lu (%ld > %d = localmaxage)", art,
			   aage, localmaxage);
		    if (verbose > 2)
			printf("Killed article %lu (%ld > %d = localmaxage).\n",
			       art, aage, localmaxage);
		    killed++;
		    i++;
		}
	    }
	    debug = debugmode;
	}
    }

    /* now we have a list of articles in stufftoget[] */
    /* let's get the header and possibly bodies of these */
  have_outstanding:
    for (i = 0; outstanding > 0; i++) {
	int takethis = 1;
	int requested_body;
	const char *cmd;

	if (!stufftoget[i])
	    continue;
	outstanding--;

	if (stufftoget[i] < server) {
	    if (verbose > 2)
		printf("%s: skipping %lu - not available or too old\n",
		       g->name, stufftoget[i]);
	    syslog(LOG_INFO, "%s: skipping %lu - not available or too old",
		   g->name, stufftoget[i]);
	    continue;
	}

	debug = debugmode;
	requested_body = ((!filterfile || article_despite_filter) && !delaybody);
	cmd = requested_body ? "ARTICLE" : "HEAD";
	xsnprintf(lineout, SIZE_lineout, "%s %lu\r\n", cmd, stufftoget[i]);
	putaline();
	l = getaline(nntpin);
	/* timeout */
	if (!l)
	{
	    if (verbose > 2)
		printf("Server disconnection or timeout before article could be retrieved\n");
	    if (debugmode)
		syslog(LOG_DEBUG, "Server disconnection or timeout before article could be retrieved");
	    return server;
	}
	/* check proper reply code */
	if (sscanf(l, "%3ld %lu", &n, &h) < 2 || ((n / 10) != 22)) {
	    if (verbose > 2)
		printf("%s %s %lu: reply %s (%ld more up in the air)\n",
		       g->name, cmd, stufftoget[i], l, outstanding);
	    syslog(LOG_INFO, "%s %s %lu: reply %s (%ld more up in the air)",
		   g->name, cmd, stufftoget[i], l, outstanding);
	    continue;
	}

	/* anything below this line will have to make sure that data is
	 * drained properly in case */

	debug--;
	if (verbose > 2)
	    printf("%s: receiving article %lu (%ld more up in the air)\n",
		   g->name, stufftoget[i], outstanding);

	for (h = 0; h < 10; h++) {
	    if (hd[h])
		free(hd[h]);
	    hd[h] = critstrdup("", "getgroup");
	}
	c = NULL;
	n = 9;			/* "other" header */
	while ((l = getfoldedline(nntpin)) && *l && strcmp(l, ".")) {
	    /* regexp pattern matching */
	    if (filterfile && dofilter(l)) {
		killed++;
		if (verbose > 2)
		    printf(".filtered: match on \"%s\"\n", l);
		syslog(LOG_INFO, "filtered article %lu: match on \"%s\"",
		       stufftoget[i], l);
		takethis = 0;
		free(l);
		l = NULL;
		continue;
	    }

	    n = 0;
	    while (strncasecmp(l, hnames[n], strlen(hnames[n])))
		n++;
	    if (n < 9 && hd[n] && *(hd[n]))
		/* second occurance of the same recognized header
		 * is treated as if it was not listed
		 * in hnames (as "other header") */
		n = 9;
	    hd[n] = critrealloc(hd[n], strlen(hd[n]) + strlen(l) + 2,
				"Fetching article header");
	    if (strlen(hd[n]))
		strcat(hd[n], "\n");	/* RATS: ignore */
	    strcat(hd[n], l);	/* RATS: ignore */
	    if (verbose > 3 && hnames[n] && *hnames[n])
		printf("...saw header %s\n", hnames[n]);
	    free(l);
	    l = NULL;
	} /* end while */
	if (l)
	    free(l);

	if (!takethis) {
	    if (requested_body) ignore_answer(nntpin);
	    continue;		/* filtered article */
	}

	debug = debugmode;
	if (!l)			/* timeout */
	    return server;

	/* check headers */
	for (h = 0; h < 6; h++) {
	    if (!hd[h] || !*(hd[h])) {
		if (verbose)
		    printf("Discarding article %lu - no %s found\n",
			   stufftoget[i], hnames[h]);
		syslog(LOG_INFO,
		       "Discarding article %lu - no %s found",
		       stufftoget[i], hnames[h]);
		killed++;
		if (requested_body) ignore_answer(nntpin);
		takethis = 0;
		break;
	    }
	}

	/* mandatory header missing */
	if (!takethis)
	    continue;

	if (localmaxage && age(hd[5]) > localmaxage) {
	    if (verbose > 2)
		printf("Discarding article %lu - older than %d days\n",
		       stufftoget[i], localmaxage);
	    syslog(LOG_INFO, "Discarding article %lu %s - older than %d days",
		   stufftoget[i], hd[4], localmaxage);
	    killed++;
	    if (requested_body) ignore_answer(nntpin);
	    continue;
	}

	if (minlines || maxlines) {
	    char *t;
	    t = strchr(hd[7], ' ');
	    if (t) {
		n = strtol(t, NULL, 10);
		if (minlines && n < minlines) {
		    if (verbose > 2)
			printf("Discarding article %lu - %ld < minlines\n",
			       stufftoget[i], n);
		    syslog(LOG_INFO,
			   "Discarding article %lu %s -- %ld < minlines",
			   stufftoget[i], hd[4], n);
		    killed++;
		    if (requested_body) ignore_answer(nntpin);
		    continue;
		}
		if (maxlines && n > maxlines) {
		    if (verbose > 2)
			printf("Discarding article %lu - %ld > maxlines\n",
			       stufftoget[i], n);
		    syslog(LOG_INFO,
			   "Discarding article %lu %s -- %ld > maxlines",
			   stufftoget[i], hd[4], n);
		    killed++;
		    if (requested_body) ignore_answer(nntpin);
		    continue;
		}
	    }
	}

	if (crosspostlimit) {
	    char *t;
	    t = hd[3];
	    n = 1;		/* number of groups the article is posted to */
	    while ((t = strchr(t, ',')) != NULL) {
		t++;
		n++;
	    }
	    if (crosspostlimit < n) {
		if (verbose > 2)
		    printf("Discarding article %lu - posted to %ld groups "
			   "(max. %ld)\n", stufftoget[i], n, crosspostlimit);
		syslog(LOG_INFO,
		       "Discarding article %lu %s - posted to %ld groups "
		       "(max. %ld)", stufftoget[i], hd[4], n, crosspostlimit);
		killed++;
		if (requested_body) ignore_answer(nntpin);
		continue;
	    }
	}

	/* store articles */
	f = NULL;
	c = lookup(strchr(hd[1], '<'));	/* lookup also replaces '/' with '@' */

	if (!c) {
	    syslog(LOG_ERR, "lookup of %s failed", hd[1]);
	    if (requested_body) ignore_answer(nntpin);
	    continue;
	}

	if (!stat(c, &st)) {
	    syslog(LOG_INFO, "article %s already stored", c);
	    if (requested_body) ignore_answer(nntpin);
	    continue;		/* for some reasons, article is already there */
	} else if (errno == ENOENT) {
	    if ((f = fopen(c, "w")) == NULL) {
		/* check whether directory exists and possibly create new one */
		char *newc, *slp;
		newc = critstrdup(c, "getgroup");
		slp = strrchr(newc, '/');
		if (slp)
		    *slp = '\0';
		if (stat(newc, &st)) {
		    if (mkdir(newc, 0775) < 0) {
			syslog(LOG_ERR, "Cannot create directory %s: %m", newc);
			if (verbose)
			    printf("Cannot create directory %s\n", newc);
		    }
		}
		free(newc);
		f = fopen(c, "w");	/* Try opening file again */
	    }
	    if (!f) {
		syslog(LOG_ERR, "unable to create article %s: %m", c);
		if (verbose)
		    printf("unable to create article %s\n", c);
		if (requested_body) ignore_answer(nntpin);
		continue;
	    }
	} else {
	    syslog(LOG_ERR, "unable to store article %s: %m", c);
	    if (verbose)
		printf("unable to store article %s\n", c);
	    if (requested_body) ignore_answer(nntpin);
	    continue;
	}

	for (h = 0; h < 10; h++)
	    if (h != 8 && hd[h] && *(hd[h]))
		fprintf(f, "%s\n", hd[h]);

	h = 0;
	/* replace tabs and other odd signs with spaces */
	while (h < 8) {
	    char *p1;
	    char *p2;
	    p1 = p2 = hd[h];
	    while (p1 && *p1) {
		if (isspace((unsigned char)*p1)) {
		    *p2 = ' ';
		    do {
			p1++;
		    } while (isspace((unsigned char)*p1));
		} else {
		    *p2 = *p1++;
		}
		p2++;
	    }
	    *p2 = '\0';
	    h++;
	}

	if (fflush(f)) {
	    (void)fclose(f);
	    (void)unlink(c);
	    if (requested_body) ignore_answer(nntpin);
	    continue;
	}

	/* generate hardlinks; this procedure also increments g->last */
	store(c, f, *hd[3] ? hd[3] + strlen(hnames[3]) : "",
	      *hd[1] ? hd[1] + strlen(hnames[1]) : "");

	if (delaybody) {
	    if (fclose(f)) {
		int e = errno;
		(void)truncate(c, 0);
		(void)unlink(c);
		if (e == ENOSPC)
		    raise(SIGINT);
	    } else {
		fetched++;
	    }
	    continue;
	}

	if (!requested_body) {
	    xsnprintf(lineout, SIZE_lineout, "BODY %ld\r\n", stufftoget[i]);
	    putaline();
	    l = getaline(nntpin);
	    if (!l) {		/* timeout */
		(void)fflush(f);
		(void)ftruncate(fileno(f), 0);
		(void)fclose(f);
		unlink(c);
		return server;
	    }
	    if (sscanf(l, "%3ld", &n) != 1 || (n / 10 != 22)) {
		syslog(LOG_INFO, "%lu: reply %s", stufftoget[i], l);
		(void)fflush(f);
		(void)ftruncate(fileno(f), 0);
		(void)fclose(f);
		unlink(c);
		continue;
	    }
	}
	debug--;
	fputs("\n", f);		/* empty line between header and body */
	while (((l = getaline(nntpin)) != NULL) && strcmp(l, ".")) {
	    if (*l == '.')
		l++;
	    clearerr(f);
	    fputs(l, f);
	    fputc('\n', f);
	    if (feof(f)) {
		l = NULL;
		break;
	    }
	}
	debug = debugmode;
	fetched++;
	if (fflush(f)) {
	    l = NULL;
	}
	if (fclose(f)) {
	    l = NULL;
	}
	if (l == NULL) {	/* article didn't terminate with a .: error */
	    (void)truncate(c, 0);
	    (void)unlink(c);
	    return server;
	} else if (hd[9]) {
#ifdef NOTYET
	    /* Handle Supersedes: header */
	    c = hd[9];
	    while (c && *c && strncasecmp(c, "Supersedes:", 11)) {
		c = strchr(hd[9], '\n');
		if (c)
		    c++;
	    }
	    /* Now c either points to the start of the Supersedes: line
	       or is NULL
	     */
	    if (c) {
		while ((c = strchr(c, '<')) != NULL)
		    supersede(c);
	    }
#endif
	}
    }

    syslog(LOG_INFO, "%s: %lu articles fetched (to %lu), %lu killed",
	   g->name, fetched, g->last, killed);
    if (verbose > 1)
	printf("%s: %lu articles fetched, %lu killed\n",
	       g->name, fetched, killed);
    return last + 1;
}

/* return 1 == success, 0 == failure */
static int expire_interesting(void) {
    DIR *d;
    struct dirent *de;
    char s[SIZE_s+1];

    xsnprintf(s, SIZE_s, "%s/interesting.groups/", spooldir);

    d = opendir(s);
    if (d == NULL) {
	ln_log(LNLOG_SERR, LNLOG_CTOP, "cannot open %s for reading: %m", s);
	return 0;
    }

    while ((de = readdir(d))) {
	struct stat st;

	xsnprintf(s, SIZE_s, "%s/interesting.groups/%s", spooldir, de->d_name);
	if (stat(s, &st) < 0)
	    continue;
	/* reading a newsgroup changes the ctime; if the newsgroup is
	   newly created, the mtime is changed as well */
	if (((st.st_mtime == st.st_ctime) &&
	     (now - st.st_ctime > (timeout_short * SECONDS_PER_DAY))) ||
	    (now - st.st_ctime > (timeout_long * SECONDS_PER_DAY))) {
	    if (verbose > 1)
		printf("unsubscribing from %s\n", de->d_name);
	    syslog(LOG_INFO, "unsubscribing from %s (current time: %ld): "
		   "ctime age %ld, mtime age %ld", de->d_name, (long)now,
		   (long)now - st.st_ctime, (long)now - st.st_mtime);
	    unlink(s);
	}
    }

    (void)closedir(d);
    return 1;
}


/** get active file from current_server.
 * \returns 0 for success, non-zero for error.
 */
static int
nntpactive(struct server *current_server, time_t update)
{
    struct stat st;
    char *l, *p;
    struct stringlist *groups = NULL;
    struct stringlist *helpptr = NULL;
    char timestr[64]; /* RATS: ignore */
    long reply = 0l;
    int error, merge = 0;
    char s[SIZE_s+1];
    int try_xgtitle = 1;
    static time_t cur_date;
    static int cur_date_init = 0;

    if (!cur_date_init) {
	cur_date = time(NULL);
	cur_date_init = 1;
    }
    
    xsnprintf(s, SIZE_s, "%s/leaf.node/%s", spooldir, current_server->name);
    if (active && !forceactive && (stat(s, &st) == 0)) {
	if (verbose)
	    printf("Getting new newsgroups from %s\n", current_server->name);
	/* to avoid a compiler warning we print out a four-digit year;
	 * but since we need only the last two digits, we skip them
	 * in the next line
	 */
	(void)strftime(timestr, sizeof(timestr),
		       "%Y%m%d %H%M00", gmtime(&update));
	xsnprintf(lineout, SIZE_lineout, "NEWGROUPS %s GMT\r\n", timestr + 2);
	putaline();
	/* we used to expect 231 here, but some broken servers (MC-link
	 * Custom News-server V1.06) return 215 instead.
	 * Just accept any 2XX code as success.
	 */
	if ((reply = nntpreply(current_server)) < 200 || reply >= 300) {
	    char *e = lastreply();
	    if (!e) e = "premature disconnect";
	    ln_log(LNLOG_SERR, LNLOG_CSERVER,
		    "Reading new newsgroups failed, reason \"%s\"", e);
	    return -1;
	}
	while ((l = getaline(nntpin)) && (strcmp (l, "."))) {
	    p = l;
	    while (*p && !isspace((unsigned char)*p))
		p++;
	    if (*p)
		*p = '\0';
	    if (gs_match(current_server->group_pcre, l)) {
		merge++;
		insertgroup(l, 1, 1, cur_date);
		prependtolist(&groups, l);
	    }
	}
	if (!l) {		/* timeout */
	    ln_log(LNLOG_SERR, LNLOG_CSERVER,
		    "Reading new newsgroups failed, timeout.");
	    return -1;
	}
	if (merge) {
	    mergegroups();	/* merge groups into active */
	    merge = 0;
	}
	helpptr = groups;
	while (helpptr != NULL) {
	    if (current_server->descriptions) {
		error = 0;
		if (try_xgtitle) {
		    xsnprintf(lineout, SIZE_lineout, "XGTITLE %s\r\n",
			    helpptr->string);
		    putaline();
		    reply = nntpreply(current_server);
		}
		if (!try_xgtitle || reply != 282) {
		    try_xgtitle = 0;
		    xsnprintf(lineout, SIZE_lineout, "LIST NEWSGROUPS %s\r\n",
			      helpptr->string);
		    putaline();
		    reply = nntpreply(current_server);
		    if (reply && (reply != 215))
			error = 1;
		}
		if (!error) {
		    l = getaline(nntpin);
		    if (l && *l && strcmp(l, ".")) {
			p = l;
			while (*p && !isspace((unsigned char)*p))
			    p++;
			while (isspace((unsigned char)*p)) {
			    *p = '\0';
			    p++;
			}
			if (reply == 215 || reply == 282)
			    changegroupdesc(l, *p ? p : NULL);
			do {
			    l = getaline(nntpin);
			    error++;
			} while (l && *l && strcmp(l, "."));
			if (error > 1) {
			    current_server->descriptions = 0;
			    syslog(LOG_NOTICE, "warning: %s does not process "
				   "LIST NEWSGROUPS %s correctly: use nodesc\n",
				   current_server->name, helpptr->string);
			    printf("warning: %s does not process LIST "
				   "NEWSGROUPS %s correctly: use nodesc\n",
				   current_server->name, helpptr->string);
			}
		    }
		}
	    }			/* if ( current_server->descriptions ) */
	    helpptr = helpptr->next;
	}
	freelist(groups);
    } else {
	if (verbose)
	    printf("Getting all newsgroups from %s\n", current_server->name);
	xsnprintf(lineout, SIZE_lineout, "LIST\r\n");
	putaline();
	if (nntpreply(current_server) != 215) {
	    char *e = lastreply();
	    if (!e) e = "premature disconnect";
	    ln_log(LNLOG_SERR, LNLOG_CSERVER,
		    "Reading all newsgroups failed, reason \"%s\".", e);
	    return -1;
	}
	debug--;
	while ((l = getaline(nntpin)) && (strcmp(l, "."))) {
	    p = l;
	    while (*p && !isspace((unsigned char)*p))
		p++;
	    while (isspace((unsigned char)*p)) {
		*p = '\0';
		p++;
	    }
	    if (gs_match(current_server->group_pcre, l)) {
		insertgroup(l, 1, 1, cur_date);
	    }
	}
	mergegroups();
	if (!l) {		/* timeout */
	    ln_log(LNLOG_SERR, LNLOG_CSERVER,
		    "Reading all newsgroups failed, timeout.");
	    return -1;
	}
	if (current_server->descriptions) {
	    xsnprintf(lineout, SIZE_lineout, "LIST NEWSGROUPS\r\n");
	    putaline();
	    l = getaline(nntpin);
	    /* correct reply starts with "215". However, INN 1.5.1 is broken
	       and immediately returns the list of groups */
	    if (l) {
		reply = strtol(l, &p, 10);
		if ((reply == 215) && (*p == ' ' || *p == '\0')) {
		    l = getaline(nntpin);	/* get first description */
		} else if (*p != ' ' && *p != '\0') {
		    int dummy;
		    /* INN 1.5.1: line already contains description */
		    (void)dummy;
		} else {
		    ln_log(LNLOG_SERR, LNLOG_CSERVER,
			    "%s: Reading newsgroups descriptions failed: %s",
			    current_server->name, l);
		    ln_log(LNLOG_SERR, LNLOG_CSERVER,
			    "Workaround: Add \"nodesc = 1\" (without quotes) below the server = %s line.", current_server->name);
		    return -1;
		}
	    } else {
		ln_log(LNLOG_SERR, LNLOG_CSERVER, 
			"Reading newsgroups descriptions failed: server disconnected prematurely.");
		return -1;
	    }
	    while (l && (strcmp(l, "."))) {
		p = l;
		while (*p && !isspace((unsigned char)*p))
		    p++;
		while (isspace((unsigned char)*p)) {
		    *p = '\0';
		    p++;
		}
		changegroupdesc(l, *p ? p : NULL);
		l = getaline(nntpin);
	    }
	    if (!l) {		/* timeout */
		ln_log(LNLOG_SERR, LNLOG_CSERVER,
			"Reading newsgroup descriptions failed, timeout.");
		return -1;
	    }
	}
	debug = debugmode;
    }
    return 0;
}

static int
movetofailed(const char *name) {
    char s[SIZE_s + 1];

    xsnprintf(s, SIZE_s, "%s/failed.postings/%s", spooldir, name);
    ln_log(LNLOG_SERR, LNLOG_CARTICLE,
	    "moving file %s to failed.postings", name);
    if (rename(name, s)) {
	ln_log(LNLOG_SERR, LNLOG_CARTICLE,
		"unable to move failed posting to %s: %m", s);
	return -1;
    } else {
	return 0;
    }
}

/*
 * post all spooled articles
 *
 * if all postings succeed, returns 1
 * if there are no postings to post, returns 1
 * if a posting is strange for some reason, returns 0
 */
static int
postarticles(const struct server *current_server)
{
    struct stat st;
    char *line;
    DIR *d;
    struct dirent *de;
    FILE *f;
    int r, haveid, n;
    char *p, *q;
    int savedir;

    n = 0;

    savedir = open(".", O_RDONLY);
    if (savedir < 0) {
	int e = errno;
	syslog(LOG_ERR, "postarticles: Unable to save current working directory: %m");
	printf("postarticles: Unable to save current working directory: %s\n", strerror(e));
	return 0;
    }

    if (chdir(spooldir) || chdir("out.going")) {
	syslog(LOG_ERR, "Unable to cd to outgoing directory: %m");
	printf("Unable to cd to %s/out.going\n", spooldir);
	fchdir(savedir);
	close(savedir);
	return 0;
    }

    d = opendir(".");
    if (!d) {
	syslog(LOG_ERR, "Unable to opendir out.going: %m");
	printf("Unable to opendir %s/out.going\n", spooldir);
	fchdir(savedir);
	close(savedir);
	return 0;
    }

    while ((de = readdir(d)) != NULL) {
	haveid = 0;
	f = NULL;
	if (!strcmp(".", de->d_name) || !strcmp("..", de->d_name)) {
	    continue;
	}
	p = q = NULL;
	if ((lstat(de->d_name, &st) != 0)) {
	    ln_log(LNLOG_SERR, LNLOG_CARTICLE,
		    "postarticles: cannot stat %s: %m",
		    de->d_name);
	    continue;
	}

	if (!S_ISREG(st.st_mode)) {
	    ln_log(LNLOG_SNOTICE, LNLOG_CARTICLE,
		    "postarticles: %s is not a regular file",
		    de->d_name);
	    movetofailed(de->d_name);
	    continue;
	}

	if (!(st.st_mode & S_IRUSR)) {
	    ln_log(LNLOG_SNOTICE, LNLOG_CARTICLE,
		    "postarticles: skipping %s, not complete",
		    de->d_name);
	    continue;
	}

	f = fopen(de->d_name, "r");
	if (f == NULL) {
	    ln_log(LNLOG_SERR, LNLOG_CARTICLE,
		    "postarticles: cannot open file %s for reading: %m",
		    de->d_name);
	    movetofailed(de->d_name);
	    continue;
	}

	p = fgetheader(f, "Newsgroups:");
	q = fgetheader(f, "Message-ID:");
	if (p == NULL || q == NULL) {
	    ln_log(LNLOG_SERR, LNLOG_CARTICLE,
		    "postarticles: file %s lacks Newsgroups "
		    "and/or Message-ID header (this cannot happen)",
		    de->d_name);
	    movetofailed(de->d_name);
	    goto free_cont;
	}

	if (!current_server->post_anygroup) {
	    if (!isgrouponserver(current_server, p)) {
		ln_log(LNLOG_SDEBUG, LNLOG_CARTICLE,
			"postarticles: file %s: server does not carry Newsgroups: %s",
			de->d_name, p);
		goto free_cont;
	    }
	}
	
	if ((haveid = ismsgidonserver(current_server, q)) != 1) {
	    ln_log(LNLOG_SINFO, LNLOG_CARTICLE, "postarticles: trying to post file %s Message-ID %s",
		    de->d_name, q);

	    xsnprintf(lineout, SIZE_lineout, "POST\r\n");
	    putaline();
	    r = nntpreply(current_server);
	    if (r != 340) {
		char *e = lastreply();
		if (e == NULL) {
		    ln_log(LNLOG_SERR, LNLOG_CARTICLE,
			    "postarticles: server %s disconnected prematurely"
			    " while trying to post file %s)",
			    current_server->name, de->d_name);
		    goto free_ret;
		}

		ln_log(LNLOG_SERR, LNLOG_CARTICLE,
			"postarticles: server %s replied \"%s\" to our POST command, skipping server",
			current_server->name, e);
		goto free_ret;
	    } else {
		/* server replied 340, it is willing to accept our article */
		int postok = 0;
		debug--;
		while ((line = getaline(f)) != NULL) {
		    /* can't use putaline() here because
		       line length of lineout is restricted */
		    if (line[0] == '.')
			fputc('.', nntpout);
		    fputs(line, nntpout);
		    fputs("\r\n", nntpout);
		};
		fflush(nntpout);
		debug = debugmode;
		xsnprintf(lineout, SIZE_lineout, ".\r\n");
		putaline();
		line = getaline(nntpin);
		if (!line) {	/* timeout: posting failed */
		    ln_log(LNLOG_SERR, LNLOG_CARTICLE,
			    "postarticles: server %s disconnected prematurely"
			    " after sending post file %s)",
			    current_server->name, de->d_name);
		    goto free_ret;
		}
		line = critstrdup(line, "postarticles");
		if (strncmp(line, "240", 3) == 0) {
		    postok = 1;
		} else if (ismsgidonserver(current_server, q)) {
		    syslog(LOG_NOTICE, "postarticles: posting gave result of \"%s\", but article is available upstream, assuming OK.", line);
		    printf("postarticles: posting gave result of \"%s\",\nbut article is available upstream, assuming OK.", line);
		    postok = 1;
		}
		if (postok) {
		    if (verbose > 2)
			printf("postarticles: POST of article %s OK\n", de->d_name);
		    n++;
		    if (unlink(de->d_name)) {
			int e = errno;
			syslog(LOG_ERR,
			       "postarticles: unable to unlink posted article %s: %m",
			       de->d_name);
			fprintf(stderr,
				"postarticles: unable to unlink posted article %s: %s",
				de->d_name, strerror(e));
		    }
		} else {
		    ln_log(LNLOG_SERR, LNLOG_CARTICLE,
			    "postarticles: Article file %s Message-ID %s was rejected: \"%s\"",
			    de->d_name, q, line);
		    movetofailed(de->d_name);
		}
		free(line);
	    }
	    haveid = 0;
	} else if (haveid) {
	    syslog(LOG_INFO, "postarticles: Message-ID of %s already in use upstream -- "
		   "article discarded\n", de->d_name);
	    if (verbose > 2)
		printf("postarticles: %s already available upstream\n", de->d_name);
	    unlink(de->d_name);
	}
free_cont:
	fclose(f);
	if (p) free(p);
	if (q) free(q);
	continue;
free_ret:
	if (p) free(p);
	if (q) free(q);
	fchdir(savedir);
	close(savedir);
	closedir(d);
	return 0;
    } /* while de = readdir */
    closedir(d);
    if (verbose)
	printf("%s: %d articles posted\n", current_server->name, n);
    syslog(LOG_INFO, "%s: %d articles posted", current_server->name, n);
    fchdir(savedir);
    close(savedir);
    return 1;
}

static void formatserver(char *d, size_t len, const struct server *serv, const char *suffix)
{
    if (serv->port == 0 || serv->port == 119)
	xsnprintf(d, len, "%s/leaf.node/%s%s", spooldir, serv->name, suffix);
    else
	xsnprintf(d, len, "%s/leaf.node/%s:%u%s", spooldir, serv->name,
		(unsigned int)serv->port, suffix);
}

static int
processupstream(const struct server *serv)
{
    FILE *f;
    DIR *d;
    struct dirent *de;
    struct newsgroup *g;
    int havefile;
    unsigned long newserver = 0;
    char *l;
    char *oldfile;
    struct stat st1, st2;
    int have_st1 = 0;
    char s[SIZE_s+1];

    struct stringlist *ngs = NULL, *a, *b = NULL;

    /* read server info */
    formatserver(s, SIZE_s, serv, "");

    oldfile = critstrdup(s, "processupstream");
    havefile = 0;
    if ((f = fopen(s, "r")) != NULL) {
	if (fstat(fileno(f), &st1)) {
	    int e = errno;
	    syslog(LOG_ERR, "Cannot stat %s: %s", s, strerror(e));
	    fprintf(stderr, "Cannot stat %s: %s", s, strerror(e));
	} else {
	    have_st1 = 1;
	}
	/* a sorted array or a tree would be better than a list */
	ngs = NULL;
	debug--;
	if (verbose > 1)
	    printf("Reading server info from %s\n", s);
	syslog(LOG_INFO, "Reading server info from %s", s);
	while (((l = getaline(f)) != NULL) && (strlen(l))) {
	    a = (struct stringlist *)critmalloc(sizeof(struct stringlist)
						+ strlen(l),
						"Reading server info");
	    strcpy(a->string, l);	/* RATS: ignore */
	    a->next = NULL;
	    if (ngs == NULL)
		ngs = a;
	    else
		b->next = a;
	    b = a;
	}
	havefile = 1;
	debug = debugmode;
	fclose(f);
    }

    xsnprintf(s, SIZE_s, "%s/interesting.groups", spooldir);
    d = opendir(s);
    if (!d) {
	syslog(LOG_ERR, "opendir %s: %m", s);
	free(oldfile);
	return -1;
    }

    formatserver(s, SIZE_s, serv, "~");
    if (stat(s, &st2) == 0 && (!have_st1 || st2.st_mtime >
			       st1.st_mtime) && (f = fopen(s, "r"))) {
	int e = 0;

	/* roll in changes of a previous crash */
	syslog(LOG_INFO, "Merging in %s from previous run", s);
	while (((l = getaline(f)) != NULL)) {
	    char *p, *t = strchr(l, ' ');
	    if (!t || !*++t)
		continue;
	    (void)strtoul(t, &p, 10);
	    if (p && !*p)
		replaceinlist(&ngs, l, (size_t)(t-l));
	}
	(void)fclose(f);	/* read-only file, assume no error */
	xsnprintf(s, SIZE_s, "%s/leaf.node/%s.new", spooldir, serv->name);
	f = fopen(s, "w");
	if (!f) {
	    e = 1;
	} else {
	    a = ngs;
	    while (a && a->string && !ferror(f)) {
		(void)fputs(a->string, f);
		(void)fputc('\n', f);
		a = a->next;
	    }
	    if (fclose(f))
		e = 1;
	}
	if (e) {
	    syslog(LOG_ERR, "open %s: %m", s);
	    (void)unlink(s);
	    return -1;
	}
	if (rename(s, oldfile)) {
	    syslog(LOG_ERR, "rename %s -> %s: %m", s, oldfile);
	    (void)unlink(s);
	    return -1;
	}
    }

    formatserver(s, SIZE_s, serv, "~");
    (void)unlink(s);
    f = fopen(s, "w");
    if (f == NULL) {
	syslog(LOG_ERR, "Could not open %s for writing: %s", s,
	       strerror(errno));
	printf("Could not open %s for writing: %s\n", s, strerror(errno));
    } else {
	/* make sure that at least SERVERINFO~ is complete */
	if (mysetvbuf(f, NULL, _IOLBF, 4096)) {
	    /* try to at least use unbuffered then */
	    mysetvbuf(f, NULL, _IONBF, 0);
	}
    }
    while ((de = readdir(d))) {
	if (isalnum((unsigned char)*(de->d_name))) {
	    g = findgroup(de->d_name);
	    if (g != NULL) {
		xsnprintf(s, SIZE_s, "%s ", g->name);
		l = havefile ? findinlist(ngs, s) : NULL;
		if (l && *l) {
		    char *t;
		    unsigned long srv;
		    l = strchr(l, ' ');
		    srv = strtoul(l, &t, 10);
		    if (t && !*t)
			newserver = getgroup(serv, g, srv);
		    else
			newserver = getgroup(serv, g, 1ul);
		} else {
		    newserver = getgroup(serv, g, 1ul);
		}
		/* run this independent of delaybody mode, because
		 * the admin may have switched delaybody off recently,
		 * and we still want users to be able to retrieve
		 * articles. */
		if (f != NULL) {
		    fprintf(f, "%s %lu\n", g->name, newserver);
		}
	    } else {
		printf("%s not found in groupinfo file\n", de->d_name);
		syslog(LOG_INFO, "%s not found in groupinfo file", de->d_name);
	    }
	}
    }
    closedir(d);
    if (f != NULL) {
	int ren = 1;
	formatserver(s, SIZE_s, serv, "~");
	if (ferror(f))
	    ren = 0;
	if (fflush(f))
	    ren = 0;
	if (fclose(f))
	    ren = 0;
	if (ren) {
	    if (rename(s, oldfile)) {
		int e = errno;
		syslog(LOG_ERR, "cannot rename %s to %s: %s", s,
		       oldfile, strerror(e));
		fprintf(stderr, "cannot rename %s to %s: %s\n", s,
			oldfile, strerror(e));
	    }
	} else {
	    syslog(LOG_ERR, "write error on %s, old version of %s kept",
		   s, oldfile);
	    fprintf(stderr, "write error on %s, old version of %s kept\n",
		    s, oldfile);
	}
    }

    free(oldfile);
    freelist(ngs);
    return 0;
}

static char *
activeread(void)
{
    const char *a = "/active.read";
    size_t l;
    char *t = critmalloc((l = strlen(spooldir)) + strlen(a) + 1, "activeread");
    strcpy(t, spooldir); /* RATS: ignore */
    strcpy(t + l, a); /* RATS: ignore */
    return t;
}

/*
 * checks whether all newsgroups have to be retrieved anew
 * returns 0 if yes, time of last update if not
 * mtime is the time when active was fetched fully
 * atime is the time when active was last updated
 */
static time_t
checkactive(void)
{
    struct stat st;
    char *s = activeread();

    if (stat(s, &st)) {
	free(s);
	return 0;
    }
    if ((now - st.st_mtime) < (timeout_active * SECONDS_PER_DAY)) {
	if (debugmode)
	    syslog(LOG_DEBUG,
		   "Last LIST done %d seconds ago: NEWGROUPS\n",
		   (int)(now - st.st_mtime));
	free(s);
	return st.st_atime;
    } else {
	if (debugmode)
	    syslog(LOG_DEBUG, "Last LIST done %d seconds ago: LIST\n",
		   (int)(now - st.st_mtime));
	free(s);
	return 0;
    }
}

static int
updateactive(void)
{
    struct stat st;
    struct utimbuf buf;
    FILE *f;
    char *s = activeread();
    int rc = 0;

    if (stat(s, &st)) {
	/* active.read probably doesn't exist */
	(void)unlink(s); /* delete it in case it's junk */
	if ((f = fopen(s, "w")) != NULL) {
	    if (fsync(fileno(f))) rc = -1;
	    if (fclose(f)) rc = -1;
	} else {
	    /* f == NULL, open error */
	    rc = -1;
	}
    } else {
	buf.actime = (now < st.st_atime) ? st.st_atime : now;
	/* now < update may happen through HW failures */
	buf.modtime = st.st_mtime;
	if (utime(s, &buf)) {
	    rc = -1;
	}
    }
    if (rc)
	ln_log(LNLOG_SERR, LNLOG_CTOP, "cannot update or create %s: %m", s);
    free(s);
    return rc;
}

static int
killactiveread(void)
{
    int rc = 0;
    char *t = activeread();
    if (unlink(t) && errno != ENOENT) {
	ln_log(LNLOG_SERR, LNLOG_CTOP,
		"cannot delete %s: %m", t);
	rc = -1;
    }
    free(t);
    return rc;
}

static void
error_refetch(const char *e) {
    ln_log(LNLOG_SERR, LNLOG_CTOP,
	    "ERROR: FETCHNEWS MUST REFETCH THE WHOLE ACTIVE FILE NEXT RUN.");
    ln_log(LNLOG_SERR, LNLOG_CTOP, "REASON: %s", e);
}

int
main(int argc, char **argv)
{
    /* the volatile keyboard avoids clobbering by siglongjmp */
    volatile time_t lastrun;
    volatile int rc = 0, skip_servers = 0;
    volatile int anypost = 0;
    struct server *current_server;

    int option, reply;
    pid_t pid;

    verbose = 0;
    usesupplement = 0;
    postonly = 0;

    myopenlog("fetchnews");

    if (!initvars(argv[0]))
	exit(1);

    while ((option = getopt(argc, argv, "Pflnvx:")) != -1) {
	if (option == 'v') {
	    verbose++;
	} else if (option == 'x') {
	    char *nptr, *endptr;
	    nptr = optarg;
	    endptr = NULL;
	    extraarticles = strtoul(nptr, &endptr, 0);
	    if (!nptr || !*nptr || !endptr || *endptr || !extraarticles) {
		usage();
		exit(1);
	    }
	} else if (option == 'l') {
	    usesupplement = -1;	/* don't use supplementary servers */
	} else if (option == 'n') {
	    noexpire = 1;
	} else if (option == 'f') {
	    forceactive = 1;
	} else if (option == 'P') {
	    postonly = 1;
	} else {
	    usage();
	    exit(1);
	}
    }

    /* Set line buffering to ensure that logging gets displayed promptly. */
    if (mysetvbuf(stdout, NULL, _IOLBF, 4096)) {
	/* Try to at least use unbuffered then */
	mysetvbuf(stdout, NULL, _IONBF, 0);
    }

    syslog(LOG_DEBUG, "leafnode %s: verbosity level is %d", version, verbose);
    if (verbose) {
	printf("leafnode %s: verbosity level is %d\n", version, verbose);
	if (verbose > 1 && noexpire) {
	    printf("Don't automatically unsubscribe unread newsgroups.\n");
	}
    }

    now = time(NULL);

    umask(2);

    if (!readconfig(0)) {
	printf("Reading configuration failed, exiting "
	       "(see syslog for more information).\n");
	if (!postonly)
	    unlink(lockfile);
	exit(1);
    }

    if (!postonly) {
	if (try_lock(5)) {
	    printf("Cannot obtain lock file, aborting.\n");
	    exit(1);
	}
	readactive();
    }

    readfilter(filterfile);

    if (forceactive) {
	    ln_log(LNLOG_SNOTICE, LNLOG_CTOP, "Forced active fetch requested from command-line (option -f).");
    }

    /* legacy code by Mark Brown, moved here out of nntpactive */
    {
	struct stat st;
	char s[SIZE_s+1];

	xsnprintf(s, SIZE_s, "%s/leaf.node/groupinfo", spooldir);
	if (stat(s, &st) == 0 && st.st_size < 7) {
	    ln_log(LNLOG_SNOTICE, LNLOG_CTOP, "Groupinfo file too small, forcing active fetch.");
	    forceactive = 1;
	}
    }

    lastrun = 0;
    if (!forceactive) {
	lastrun = checkactive();
	if (!lastrun) {
	    ln_log(LNLOG_SNOTICE, LNLOG_CTOP, "Active has not been fetched completely in previous run");
	    ln_log(LNLOG_SNOTICE, LNLOG_CTOP, "or has never been fetched, forcing active fetch.");
	    forceactive = 1;
	}
    }

    if (forceactive) {
	oldactive = active;
	oldactivesize = activesize;
	active = NULL;
	activesize = 0;
	if (killactiveread()) exit(1);
    }

    if (mysigact(SIGINT, 0, sig_int, SIGALRM) != 0)
	fprintf(stderr, "Can't catch SIGINT.\n");
    if (mysigact(SIGTERM, 0, sig_int, SIGALRM) != 0) 
	fprintf(stderr, "Can't catch SIGTERM.\n");
    if (mysigact(SIGPIPE, 0, sig_int, SIGALRM) != 0)
	fprintf(stderr, "Can't catch SIGPIPE.\n");
    if (sigsetjmp(jmpbuffer, 1) != 0) {
	if (!rc) {
	    rc = 2;
	    if (forceactive)
		error_refetch("caught signal that caused a premature abort.");
	}
	skip_servers = 1;	/* in this case, jump the while ... loop */
    } else {
	canjump = 1;
    }

    /* remove groups that haven't been read in a long time */
    if (!noexpire)
	expire_interesting();

    /* main server loop */
    for (current_server = servers; 
		    !skip_servers && current_server; 
		    current_server = current_server->next) {
	if (verbose) {
	    if (current_server->port)
		printf("Trying to connect to %s:%u ... ",
		       current_server->name, current_server->port);
	    else
		printf("Trying to connect to %s:nntp ... ",
		       current_server->name);
	}
	fflush(stdout);
	reply = nntpconnect(current_server);
	if (reply) {
	    if (verbose) {
		printf("connected.\n");
		if (stat_is_evil)
		    printf("This server software is known to break the STAT <message-ID> command,\n"
			    "using workaround with HEAD instead.\n");
		else 
		    printf("Using STAT <message-ID> command on this server.\n");
	    }
	    if (current_server->username)
		(void)authenticate(current_server);
	    xsnprintf(lineout, SIZE_lineout, "MODE READER\r\n");
	    putaline();
	    reply = nntpreply(current_server);
	    if (reply == 498) {
		syslog(LOG_INFO, "Connection refused by %s",
		       current_server->name);
	    } else {
		if (reply == 200 && current_server->nopost == 0) {
		    anypost = 1;
		    (void)postarticles(current_server);
		} else if (verbose) {
		    printf("Not posting to %s: ", current_server->name);
		    if (reply != 200)
			printf("non-permission ");
		    if (current_server->nopost)
			printf("nopost-set ");
		    printf("\n");
		}
		if (!date_is_evil)
		    check_date(current_server);
		if (!postonly && !current_server->noread) {
		    /* get list of newsgroups or new newsgroups */
		    if (forceactive || current_server->updateactive) {
			if (nntpactive(current_server, lastrun)) {
			    if (forceactive)
				error_refetch("obtaining the active file failed.");
			    rc = 1;
			}
		    } else {
			if (verbose)
			    printf("Not attempting to update newsgroups list for %s\n",
				   current_server->name);
		    }

		    processupstream(current_server);
		}
	    }
	    xsnprintf(lineout, SIZE_lineout, "QUIT\r\n");	/* say it, then just exit :) */
	    putaline();
	    nntpdisconnect();
	    if (verbose)
		printf("Disconnected from %s.\n", current_server->name);
	} else { /* reply = nntpconnect */
	    if (verbose)
		printf("failed.\n");
	    if (forceactive)
		error_refetch("fetching the active list from a server failed.");
	    rc = 2;
	}
	if (usesupplement < 0)
	    break;
    }
    mergegroups(); /* just in case we were interrupted while downloading
		      the list. */

    (void)mysigact(SIGINT, 0, SIG_IGN, 0);		/* do not siglongjmp any more */
    (void)mysigact(SIGTERM, 0, SIG_IGN, 0);	/* do not siglongjmp any more */
    (void)mysigact(SIGPIPE, 0, SIG_IGN, 0);	/* SIGPIPE should not happen below */
    freeservers();

    if (rc != 0 && oldactive) {
	/* restore old active data to keep low/high marks */
	unsigned long i;
	for (i = 0; i < oldactivesize; i++) {
	    insertgroup(oldactive[i].name, 0, 0, 0);
	}
	mergegroups();
	killactiveread();
    }

    freeactive(oldactive);
    oldactive = NULL;
    oldactivesize = 0;

    if (anypost == 0 && skip_servers == 0) {
	const char *e = "WARNING: found no server with posting permission!";
	printf("%s\n", e);
	syslog(LOG_WARNING, "%s", e);
    }

    if (fflush(stdout)) {
	/* to avoid double logging of stuff */
	ln_log(LNLOG_SERR, LNLOG_CTOP, "cannot flush standard output: %m");
    }

    if (!postonly) {
	/* OK, we do a pipe trick so hold the child until the parent
	 * handed over the lock file (this must happen so it does not
	 * inadvertently get removed as stale). The child reads from the
	 * pipe and is blocked until the parent has written to the pipe.
	 * The parent writes to the pipe after it has handed over the
	 * lock.
	 */
	int pfd[2], pipeok;

	pipeok = pipe(pfd);

	/* only update active.read when we could read active from all servers */
	if (writeactive()) {
	    ln_log(LNLOG_SERR, LNLOG_CTOP, "Error writing groupinfo.");
	    if (killactiveread()) {
		error_refetch("cannot write groupinfo file.");
		rc = 1;
	    }
	} else {
	    if (rc == 0 && updateactive())
		if (killactiveread()) {
		    error_refetch("cannot update active.read file.");
		    rc = 1;
		}
	}

#ifdef HAVE_WORKING_FORK
	pid = fork();
#else
	pid = -1; errno = 0;
#endif
	
	switch (pid) {
	    case -1:
#ifdef HAVE_WORKING_FORK
		syslog(LOG_NOTICE, "fork: %m, running on parent schedule.");
#endif
		if (verbose)
		    printf("Updating overview data in the foreground.");
		fixxover();
		unlink(lockfile);
		freeactive(active);
		activesize = 0;
		active = NULL;
		break;

	    case 0:
		(void)setsid();
		if (debugmode)
		    syslog(LOG_DEBUG, "Process forked.");
		if (pipeok == 0) {
		    /* wait for parent to hand over the lock */
		    char buf[4];
		    (void)close(pfd[1]);
		    /* we don't REALLY check for errors here, the worst thing
		     * that could happen (in case of a kernal bug...) to
		     * us is that we start early and delete the parent's
		     * lock file before the parent handed it over -- but
		     * at that time, we'll then exit and everything is
		     * in order, with just one bogus message in the log.
		     */
		    while (read(pfd[0], buf, sizeof(buf)) < 0) {
			if (errno != EAGAIN && errno != EINTR)
			    break;
		    }
		    (void)close(pfd[0]); }

		fixxover();
		if (unlink(lockfile))
		    syslog(LOG_ERR, "unlink(\"%s\"): %m", lockfile);
		freeactive(active);
		activesize = 0;
		active = NULL;
		if (debugmode)
		    syslog(LOG_DEBUG, "Process done.");
		_exit(0);
		break;

	    default:
		{
		    int lock_ok = handover_lock(pid);

		    if (verbose) puts("Started process to update overview data in the background.\nNetwork activity has finished.");
		    if (pipeok == 0) {
			/* tell child it has the lock */
			(void)close(pfd[0]);
			(void)writes(pfd[1], "GO");
			(void)close(pfd[1]);
		    }

		    if (lock_ok) {
			/* could not hand over lock file to child, so wait until it
			   dies */
			syslog(LOG_NOTICE, "could not hand over lockfile to child %lu: %m, "
				"waiting until child is done.",
				(unsigned long)pid);
			(void)waitpid(pid, NULL, 0);
		    } else {
			syslog(LOG_INFO, "child has process ID %lu",
				(unsigned long)pid);
		    }
		    break;
		}
	}
    }

    exit(rc);
}
