/*
 * apply filter file to all files in a newsgroup
 *
 * Written by Cornelius Krasel <krasel@wpxx02.toxi.uni-wuerzburg.de>.
 * Copyright 1999.
 *
 * See file COPYING for restrictions on the use of this software.
 */
 
#include "leafnode.h"
#include <sys/stat.h>
#include <sys/types.h>
#include <ctype.h>
#include <dirent.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <utime.h>

int debug = 0;
int verbose;
int first, last;

static void expiremsgid( void ) {
    int n;
    DIR * d;
    struct dirent * de;
    struct stat st;
    int deleted, kept;

    deleted = kept = 0;

    for ( n=0; n<1000; n++ ) {
	sprintf( s, "%s/message.id/%03d", spooldir, n );
	if ( chdir( s ) ) {
	    if ( errno == ENOENT )
		mkdir( s, 0755 ); /* file system damage again */
	    if ( chdir( s ) ) {
		syslog( LOG_ERR, "chdir %s: %m", s );
		continue;
	    }
	}

	d = opendir( "." );
	if ( !d )
	    continue;
	while ( (de=readdir(d)) ) {
	    if ( !stat( de->d_name, &st ) ) {
		if ( st.st_nlink<2 && !unlink( de->d_name ) )
		    deleted++;
		else if ( S_ISREG(st.st_mode) )
		    kept++;
	    }
	}
	closedir( d );
    }

    if ( kept || deleted ) {
	printf("total: %d articles deleted, %d kept\n", deleted, kept);
	syslog( LOG_INFO, "%d articles deleted, %d kept", deleted, kept );
    }
}

int main( int argc, char * argv[] ) {
    char c[4] = "-\\|/" ; 
    int i, n, error, option;
    char *k, *l;
    FILE *f;
    DIR * d;
    struct dirent * de;
    struct stat st;
    struct utimbuf u;
    struct newsgroup * g;

    if ( !initvars( argv[0] ) )
	exit(1);

    if ( ( argc < 2 ) || ( argc > 3 ) ) {
	printf("Usage: %s newsgroups\n", argv[0] );
	exit(1);
    }

    while ( ( option = getopt( argc, argv, "v" ) ) != -1 ) {
	if ( option == 'v' )
	    verbose++;
	else {
	    printf("Usage: %s newsgroups\n", argv[0] );
	    exit( 1 );
	}
    }

    if ( !readconfig() ) {
	printf( "Reading configuration failed, exiting "
		"(see syslog for more information).\n");
	exit( 2 );
    }
    readconfig();
    if ( filterfile )
	readfilter( filterfile );
    else {
	printf("Nothing to filter -- no filterfile found.\n");
	exit(1);
    }
    if ( lockfile_exists() )
	exit( 1 );
    readactive();

    g = findgroup( argv[optind] );
    if ( !g ) {
	printf("Newsgroups %s not found in active file.\n", argv[optind] );
	unlink( lockfile );
	exit(1);
    }

    g->first = INT_MAX;
    g->last = 0;
    if ( !chdirgroup( g->name, FALSE )) {
	printf("No such newsgroup: %s\n", g->name );
	unlink( lockfile );
	exit(1);
    }
    if ( !(d = opendir( "." ) )) {
	printf("Unable to open directory for newsgroup %s\n", g->name );
	unlink( lockfile );
	exit(1);
    }

    i = 0;
    l = NULL;
    while ( ( de = readdir( d ) ) != NULL ) {
	if ( verbose ) {
	    printf( "%c", c[i%4] );
	    fflush( stdout );
	    i++;
	}
	stat( de->d_name, &st );
	if ( !S_ISREG( st.st_mode ) || !isdigit((int)de->d_name[0]) ) {
	    continue ;
	}
	l = critrealloc( l, st.st_size+1, "Space for article" );
	if ( ( f = fopen( de->d_name, "r" ) ) != NULL ) {
	    fread( l, sizeof(char), st.st_size, f );
	    if ( ( k = strstr( l, "\n\n" ) ) != NULL )
		*k = '\0';	/* cut off body */
	    error = dofilter( l );
	    fclose( f );
	    if ( error ) {
		unlink( de->d_name );
		if ( verbose )
		    printf( "%s deleted\n", de->d_name );
	    }
	    else {
		n = strtol( de->d_name, NULL, 10 );
		if ( n ) {
		    if ( n < g->first )
			g->first = n;
		    if ( n > g->last )
			g->last = n;
		}
		u.actime = st.st_atime;
		u.modtime = st.st_mtime;
		utime( de->d_name, &u );
	    }
	} else
	    printf( "could not open %s\n", de->d_name );
    }
    closedir( d );
    free( l );
    if ( g->first <= g->last )
	writeactive();
    unlink( lockfile );

    if ( verbose )
	printf( "Updating .overview file\n" );
    getxover();
    if ( verbose )
	printf( "Deleting articles from spool\n" );
    expiremsgid();
    exit(0);
}
