/*
libutil -- deal with active file

Written by Arnt Gulbrandsen <agulbra@troll.no> and copyright 1995
Troll Tech AS, Postboks 6133 Etterstad, 0602 Oslo, Norway, fax +47
22646949.
Modified by Cornelius Krasel <krasel@wpxx02.toxi.uni-wuerzburg.de>
and Randolf Skerka <Randolf.Skerka@gmx.de>.
Copyright of the modifications 1997.
Modified by Kent Robotti <robotti@erols.com>. Copyright of the
modifications 1998.
Modified by Markus Enzenberger <enz@cip.physik.uni-muenchen.de>.
Copyright of the modifications 1998.
Modified by Cornelius Krasel <krasel@wpxx02.toxi.uni-wuerzburg.de>.
Copyright of the modifications 1998, 1999.

See file COPYING for restrictions on the use of this software.
*/

#include "leafnode.h"

#include <fcntl.h>
#include <sys/uio.h>
#include <sys/param.h>
#include <ctype.h>
#include <errno.h>
#include <limits.h>
#include <stdlib.h>
#include <netdb.h>
#include <stdio.h>
#include <string.h>
#include <syslog.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <time.h>
#include <unistd.h>
#include <dirent.h>

#define   STRNCMPEQ(s1, s2, n)    (*(s1) == *(s2) && strncmp((s1), (s2), n) == 0)

extern struct state _res;

/* insert or update a newsgroup in the global list */
void insertgroup(const char * name, int first, int last, const char * desc,
		 int age ) {
    struct newsgroup ** a;
    int c;

    if ( !desc )
	desc = "";

    a = &active;
    while(a) {
	if (*a) {
	    c = strcmp((*a)->name, name);
	    if (c<0)
		a = &((*a)->left);
	    else if (c>0)
		a = &((*a)->right);
	    else {
	    	/* update newsgroup description, nothing else */
		if ( (*a)->desc ) {
		    if ( !desc )
			return;
		    if ( ( strcmp( desc, "-x-") == 0 ) ||
			 ( strcmp( desc, "local group" ) == 0 ) )
		        return;
		    free( (*a)->desc );
		}
		(*a)->desc = strdup( desc );
		return;
	    }
	} else {
	    /* create new newsgroup entry */
	    *a = (struct newsgroup *)
		 critmalloc(sizeof(struct newsgroup),
			    "Building newsgroups info list");
	    if (!*a)
		return;
	    (*a)->left = (*a)->right = NULL;
	    (*a)->first = (first) ? first : 1;
	    (*a)->last = (last>=first) ? last : 1;
	    (*a)->age = age;
	    (*a)->name = strdup( name );
	    if ( (*a)->name == NULL ) {
		syslog( LOG_ERR, "Not sufficient memory for newsgroup %s",
			name );
		free(*a);
		return;
	    }
	    (*a)->desc = strdup( desc );
	    return;
	}
    }
}

/*
 * check whether group is represented in interesting.groups
 */
int isinteresting( char * groupname ) {
    struct stat st;

    sprintf( s, "%s/interesting.groups/%s", spooldir, groupname );
    return(stat( s, &st ) + 1);
}

/* find a group by name */
struct newsgroup * findgroup(const char* name) {
    struct newsgroup * a;
    int c;

    a = active;
    while ( a ) {
	c = strcmp( a->name, name );
	if ( c < 0 )
	    a = a->left;
	else if ( c > 0 )
	    a = a->right;
	else
	    return a;
    }
    return a;
}


static void freeactive( struct newsgroup * g ) {
    if ( g ) {
	freeactive( g->right );
	freeactive( g->left );
	free( (char*)g );
    }
}


/* this uses a nifty technique for building a binary tree from a
   sorted list... the basic observation is that if you count from one
   end, with 1 being the rightmost node, then the number of nodes
   between N and the leaf nodes is equal to the number of zero bits
   at the small end of N.  this tree probably doesn't make it obvious:

   1 2 3 4 5 6 7

         1
     1   0   1
   1 0 1 0 1 0 1
   | | | | | | + no zeroes: leaf node
   | | | | | +-- one zero: one step up
   | | | | +---- no zeroes: leaf node
   | | | +------ two zeroes: two steps up
   | | +-------- no zeroes: leaf node
   | +---------- one zero: one step up
   +------------ no zeroes: leaf node

   this can be used to build a smallest-height tree, and with a bit more
   care, a _perfectly_ balanced tree */

void readactive( void ) {
    char * p, *q, *r, *lastline, *desc;
    int first, last;
    time_t age;
    FILE * f;
    struct newsgroup * g;
    struct stat st;

    int line;
    struct newsgroup *levels[32]; /* theoretically finite size :) */
    int l, n, error;

    char * stuff;

    freeactive( active );
    active = NULL;

    strcpy(s, spooldir);
    strcat(s, "/leaf.node/groupinfo");
    if ( stat( s, &st ) ) {
	syslog( LOG_ERR, "can't stat %s: %m", s );
	return;
    } else if ( !S_ISREG( st.st_mode ) ) {
	syslog( LOG_ERR, "%s not a regular file" );
	return;
    }
    stuff = critmalloc( st.st_size+2, "Reading group info" );
    if (( f = fopen( s, "r" )) != NULL ) {
	n = fread( stuff, 1, st.st_size, f );
	if ( n < st.st_size )
	    syslog( LOG_ERR,
		    "Groupinfo file truncated while reading: %d < %d .",
		    n, st.st_size );
	fclose( f );
    }
    else {
	syslog( LOG_ERR, "unable to open %s: %m", s );
	return;
    }

    n = ( n > st.st_size ) ? st.st_size : n ;
	/* to read truncated groupinfo files correctly */
    stuff[n] = '\n';
    stuff[n+1] = '\0';	/* 0-terminate string */

    /* delete spurious 0-bytes */
    while (( p = memchr( stuff, '\0', st.st_size ) ) != NULL )
        *p = ' ';       /* \n might be better, but produces more errors */

    line = 1;
    for( l=0; l<32; levels[l++] = NULL )
	;

    p = stuff;
    error = 0;
    while ( (p < (stuff+n)) && (( q = strchr( p, '\n' )) != NULL ) ) {
	*q = '\0' ;
	lastline = p;
        if (( ( r = strchr( p, ' ' )) != NULL ) && isalnum((int)*p) ) {
            *r = '\0' ;
            p = r + 1 ;
        } else
            error = 1;
        if ( !error ) {
            last = *p ? strtol( p, &r, 10 ) : 1;
            if ( r == p )
                error = 1;
            else {
                p = r;
                first = *p ? strtol( p, &r, 10 ) : 1;
                if ( r == p )
                    error = 1;
                else {
                    p = r;
                    age = *p ? strtol( p, &r, 10 ) : 0;
                    if ( r == p )
                        error = 1;
		    else
			p = r;
                }
            }
        }

        if ( !error ) {
            while (p && *p && isspace((int)*p))
                p++;
            if ( p && *p )
                desc = strdup( p );
            else
                error = 1;
        }

        if ( error ) {
            syslog( LOG_ERR,
		    "Groupinfo file possibly truncated or damaged: %s\n",
		    lastline);
            error = 0;
        } else {
	    g = (struct newsgroup *) critmalloc(sizeof(struct newsgroup),
						"reading groupinfo" );
	    g->right = g->left = NULL;
	    g->first = (first) ? first : 1;
	    g->last = (last>=first) ? last : 1 ;
	    g->age = age;
	    g->name = strdup(lastline);
	    g->desc = desc;

            /* Inserting newsgroup into tree */
            for( l=0; (line&(1<<l))==0; l++ )
                ;
            if ( l ) {
                g->right = levels[l-1];
            }
            if ( active == g->right )
                active = g;
            if ( l<31 && levels[l+1] && !levels[l+1]->left )
                levels[l+1]->left = g;
            levels[l] = g;

            p = strchr( p, '\n' );
            if ( p )
                *p++ = '\0';
            line++;
	}

        p = q + 1;
    }
    free( stuff );

    /* More tree stuff */
    g = NULL;
    for ( l=0; l<31; l++ ) {
	if ( levels[l] ) {
	    if ( g && levels[l] &&
		 levels[l]->left == NULL && levels[l]->right != g ) {
		levels[l]->left = g;
		g = NULL;
	    }
	    if ( !levels[l+1] ||
		 ( levels[l] != levels[l+1]->left &&
		   levels[l] != levels[l+1]->right ) ) {
		if ( g )
		    syslog( LOG_ERR, "2+5=2" );
		g = levels[l];
	    }
	}
    }
}

/*
 * returns 1 if something could be written, 0 otherwise
 */
static int helpwriteactive ( struct newsgroup *g, FILE * f ) {
    if (g) {
        if ( helpwriteactive (g->right, f) ) {
            if ( fprintf(f, "%s %d %d %lu %s\n", g->name, g->last, g->first,
                 g->age, g->desc && *(g->desc) ? g->desc : "-x-" ) ) {
        	if ( helpwriteactive (g->left, f) )
		    return 1;
		else
		    return 0;
	    } else
		return 0;
	} else
	    return 0;
    }
    return 1;
}

void writeactive( void ) {
    FILE * a;
    char c[PATH_MAX];
    int err;

    strcpy(s, spooldir);
    strcat(s, "/leaf.node/groupinfo.new");
    a = fopen( s, "w" );
    if (!a) {
	syslog( LOG_ERR, "cannot open new groupinfo file: %m" );
	return;
    }
    err = helpwriteactive( active, a );
    fclose( a );

    if ( err != 0 ) {
	strcpy(c, spooldir);
        strcat(c, "/leaf.node/groupinfo");
	rename( s, c );
    } else {
	syslog( LOG_ERR, "error writing groupinfo file (disk full?)" );
	unlink(s);
    }
}

/*
 * fake an active file if the groupinfo file is corrupted
 */
void fakeactive( void ) {
    DIR * d;
    struct dirent * de;
    DIR * ng;
    struct dirent * nga;
    int i;

    int first, last;

    strcpy( s, spooldir );
    strcat( s, "/interesting.groups" );
    
    d = opendir( s );
    if ( !d ) {
	syslog( LOG_ERR, "cannot open directory %s: %m", s );
	return;
    }
    while ( (de = readdir(d)) ) {
	if ( isalnum((int)*(de->d_name) ) && chdirgroup(de->d_name, FALSE) ) {
	    /* get first and last article from the directory. This is
	     * the most secure way to get to that information since the
	     * .overview files may be corrupt as well
	     * If the group doesn't exist, just ignore the active entry.
	     */
	    first = INT_MAX;
	    last = 0;

	    ng = opendir( "." );
	    while ( ( nga = readdir( ng ) ) != NULL ) {
		if ( isdigit ((int) *(nga->d_name ) ) ) {
		    i = strtol( nga->d_name, NULL, 10 );
		    if ( i < first )
			first = i;
		    if ( i > last )
			last = i;
		}
	    }
	    if ( first > last ) {
		first = 1;
		last = 1;
	    }
	    closedir( ng );
	    if ( debugmode )
	        syslog( LOG_DEBUG, "parsed directory %s: first %d, last %d",
			de->d_name, first, last );
	    insertgroup( de->d_name, first, last, NULL, 0 );
	}
    }
}
