/*
libutil -- miscellaneous stuff

Written by Arnt Gulbrandsen <agulbra@troll.no> and copyright 1995
Troll Tech AS, Postboks 6133 Etterstad, 0602 Oslo, Norway, fax +47
22646949.
Modified by Cornelius Krasel <krasel@wpxx02.toxi.uni-wuerzburg.de>
and Randolf Skerka <Randolf.Skerka@gmx.de>.
Copyright of the modifications 1997.
Modified by Kent Robotti <robotti@erols.com>. Copyright of the
modifications 1998.
Modified by Markus Enzenberger <enz@cip.physik.uni-muenchen.de>.
Copyright of the modifications 1998.
Modified by Cornelius Krasel <krasel@wpxx02.toxi.uni-wuerzburg.de>.
Copyright of the modifications 1998, 1999.

See file COPYING for restrictions on the use of this software.
*/

#include "leafnode.h"
#include <fcntl.h>
#include <sys/uio.h>
#include <sys/param.h>
#include <ctype.h>
#include <errno.h>
#include <limits.h>
#include <stdlib.h>
#include <netdb.h>
#include <stdio.h>
#include <string.h>
#include <syslog.h>
#include <pwd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <time.h>
#include <unistd.h>
#include <dirent.h>
#include <signal.h>

#ifdef HAVE_FNMATCH
#include <fnmatch.h>
#endif

#ifndef HAVE_SNPRINTF
#include <stdarg.h>
#endif

char fqdn[256];
char s[PATH_MAX+1024]; /* long string, here to cut memory usage */
extern struct state _res;

/* xoverutil global vars */
struct xoverinfo * xoverinfo;
long xfirst, xlast;

/* activutil global vars */
struct newsgroup * active;

/*
 * initialize all global variables
 */
int initvars( char * progname ) {
    struct passwd * pw;

    active = NULL;
    xoverinfo = NULL;
    xfirst = 0;
    xlast  = 0;

    /* config.c stuff does not have to be initialized */

    expire_base = NULL;
    servers     = NULL;

    /* These directories should exist anyway */
    sprintf( s, "%s/interesting.groups", spooldir );
    mkdir( s, 0775 );
    sprintf( s, "%s/leaf.node", spooldir );
    mkdir( s, 0755 );
    sprintf( s, "%s/failed.postings", spooldir );
    mkdir( s, 0775 );
    sprintf( s, "%s/out.going", spooldir );
    mkdir( s, 2755 );
    chmod( s, S_ISUID | S_IRUSR | S_IWUSR | S_IXUSR | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH );

    if ( progname ) {
	pw = getpwnam( "news" );
	if ( !pw ) {
	    fprintf( stderr, "no such user: news\n" );
	    return FALSE;
	}

#ifdef HAVE_SETGID
	setgid( pw->pw_gid );
#else
	setregid( pw->pw_gid, pw->pw_gid );
#endif

#ifdef HAVE_SETUID
	setuid( pw->pw_uid );
#else
	setreuid( pw->pw_uid, pw->pw_uid );
#endif
	if ( getuid() != pw->pw_uid || getgid() != pw->pw_gid ) {
	    fprintf( stderr, "%s: must be run as news or root\n", progname );
	    return FALSE;
	}
    }
    return TRUE;
}

/*
 * check if lockfile exists: if yes (and another instance of fetch is
 * still active) return 1; otherwise generate it. If this fails,
 * also return 1, otherwise return 0
 */
int lockfile_exists( void ) {
    FILE * lf ;
    int  pid ;

    if ( ( lf = fopen( lockfile, "r+" )) != NULL ) {
	if ( fscanf( lf, "%d", &pid ) ) {
	    fclose( lf );
	    if ( kill( pid, 0 ) == -1 && errno == ESRCH ) {
		if ( verbose )
		    printf( "Removing stale lockfile.\n" );
		syslog( LOG_INFO, "Removing stale lockfile: pid %d", pid );
		unlink( lockfile );
	    } else {
		printf( "lockfile %s exists, abort ...\n", lockfile);
		syslog( LOG_ERR, "lockfile %s exists, abort ...",
			lockfile);
		return 1;
	    }
	} else {
	    fclose( lf );
	    syslog( LOG_ERR, "unable to read lockfile %s, abort ...", lockfile );
	    return 1;
	}
    } else if ( errno != ENOENT ) {
	syslog( LOG_ERR, "unable to open lockfile %s: %d %m", lockfile, errno );
    }
    if ( ( lf = fopen( lockfile, "w" )) != NULL ) {
	fprintf( lf, "%d", (int)getpid() );
	fclose( lf );
	return 0;
    } else {
	printf( "Could not open lockfile %s for writing, abort program ...\n",
		lockfile );
	syslog( LOG_ERR, "Could not open lockfile %s for writing: %m",
		lockfile );
	return 1;
    }
}


/*
 * check whether "groupname" is represented in interesting.groups without
 * touching the file
 */
int isinteresting( const char *groupname ) {
    DIR * d;
    struct dirent * de;

    sprintf( s, "%s/interesting.groups", spooldir );
    d = opendir( s );
    if ( !d ) {
    	syslog( LOG_ERR, "Unable to open directory %s: %m", s );
	printf( "Unable to open directory %s\n", s );
	return FALSE;
    }

    while( ( de = readdir( d ) ) != NULL ) {
    	if ( strcasecmp( de->d_name, groupname ) == 0 ) {
	    closedir( d );
	    return TRUE;
	}
    }
    closedir( d );
    return FALSE;
}

/* no good but this server isn't going to be scalable so shut up */
const char * lookup ( const char *msgid ) {
    static char * name = NULL;
    static unsigned int namelen = 0; 
    unsigned int r;
    unsigned int i;

    if (!msgid || !*msgid)
	return NULL;

    i = strlen(msgid)+strlen(spooldir)+30;

    if (!name) {
	name = (char *)malloc(i);
	namelen = i;
    } else if (i > namelen) {
	name = (char *)realloc(name, i);
	namelen = i;
    }

    if (!name) {
	syslog( LOG_ERR, "malloc(%d) failed", i );
	exit(1);
    }

    strcpy( name, spooldir );
    strcat( name, "/message.id/000/" );
    i = strlen( name );
    strcat( name, msgid );

    r = 0;
    do {
	if ( name[i] == '/' )
	    name[i] = '@';
	else if ( name[i] == '>' )
	    name[i+1] = '\0';
	r += (int)(name[i]);
	r += ++i;
    } while ( name[i] );

    i = strlen( spooldir )+14; /* to the last digit */
    r = (r%999)+1;
    name[i--] = '0'+(char)(r%10); r /= 10;
    name[i--] = '0'+(char)(r%10); r /= 10;
    name[i] = '0'+(char)(r);
    return name;
}



/*
 * replacement for malloc, syslogs allocation failures
 * and exits with the error message
 */
char * critmalloc(size_t size, const char* message) {
    char * a;

    a = malloc(size);
    if (!a) {
	syslog(LOG_ERR, "malloc(%d) failed: %s", (int)size, message);
	fprintf(stderr, "malloc(%d) failed: %s\n", (int)size, message);
	exit(1);
    }
    return a;
}

/*
 * replacement for realloc, syslogs allocation failures
 * and exits with the error message
 */
char * critrealloc(char *a, size_t size, const char* message) {
    a = realloc(a, size);
    if (!a) {
	syslog(LOG_ERR, "realloc(%d) failed: %s", (int)size, message);
	fprintf(stderr, "realloc(%d) failed: %s\n", (int)size, message);
	exit(1);
    }
    return a;
}

#define LM_SIZE 65536

static int makedir( char * d ) {
    char * p;
    char * q;

    if (!d || *d != '/' || chdir("/"))
	return 0;
    q = d;
    do {
	*q = '/';
	p = q;
	q = strchr( ++p, '/' );
	if (q)
	    *q = '\0';
	if (!chdir(p))
	    continue; /* ok, I do use it sometimes :) */
	if (errno==ENOENT)
	    if (mkdir(p, 0775)) {
		syslog( LOG_ERR, "mkdir %s: %m", d );
		exit( 1 );
	    }
	if (chdir(p)) {
	    syslog( LOG_ERR, "chdir %s: %m", d );
	    exit( 1 );
	}
    } while ( q );
    return 1;
}
	

/* chdir to the directory of the argument if it's a valid group */
int chdirgroup( const char *group, int creatdir ) {
    char *p;

    if (group && *group) {
	strcpy(s, spooldir);
	p = s + strlen(s);
	*p++ = '/';
	strcpy(p, group);
	while(*p) {
	    if (*p=='.')
		*p = '/';
	    else
		*p = tolower((unsigned char)*p);
	    p++;
	}
	if ( !chdir(s) )
	    return 1;		/* chdir successful */
	if ( creatdir )
	    return makedir( s );
    }
    return 0;
}

/* get the fully qualified domain name of this box into fqdn */

void whoami( void ) {
    struct hostent * he;

    if (!gethostname(fqdn, 255) && (he = gethostbyname(fqdn))!=NULL) {
	strncpy( fqdn, he->h_name, 255 );
	if (strchr(fqdn, '.') == NULL) {
	    char ** alias;
	    alias = he->h_aliases;
	    while( alias && *alias )
		if (strchr(*alias, '.') && (strlen(*alias)>strlen(fqdn)))
		    strncpy( fqdn, *alias, 255 );
		else
		    alias++;
	    }
    } else
	*fqdn = '\0';
}

/*
 * append string "newentry" to stringlist "list". "lastentry" is a
 * pointer pointing to the last entry in "list" and must be properly
 * intialized.
 */
void appendtolist( struct stringlist ** list, struct stringlist ** lastentry,
    char * newentry ) {

    struct stringlist *ptr;

    ptr = (struct stringlist *)critmalloc( sizeof( struct stringlist ) +
          strlen( newentry ), "Allocating space in stringlist" );
    strcpy( ptr->string, newentry );
    ptr->next = NULL;
    if ( *list == NULL )
        *list = ptr;
    else
        (*lastentry)->next = ptr;
    *lastentry = ptr;
}

/*
 * find a string in a stringlist
 * return pointer to string if found, NULL otherwise
 */
char * findinlist( struct stringlist * haystack, char * needle ) {
    struct stringlist * a;

    a = haystack;
    while ( a && a->string ) {
	if ( strncmp( needle, a->string, strlen(needle) ) == 0 )
	    return a->string;
	a = a->next;
    }
    return NULL;
}

/*
 * free a list
 */
void freelist( struct stringlist * list ) {
    struct stringlist * a;

    while ( list ) {
	a = list->next;
	free( list );
	list = a;
    }
}

/* next few routines implement a mapping from message-id to article
   number, and clearing the entire space */

struct msgidtree {
    struct msgidtree * left;
    struct msgidtree * right;
    int art;
    char msgid[1];
};

static struct msgidtree * head; /* starts as NULL */

void insertmsgid( const char * msgid, int art ) {
    struct msgidtree ** a;
    int c;

    if ( strchr( msgid, '@' ) == 0 )
	return;

    a = &head;
    while (a) {
	if (*a) {
	    /* comparing only by msgid is uncool because the tree becomes
	       very unbalanced */
	    c = strcmp(strchr((*a)->msgid, '@'), strchr(msgid, '@'));
	    if ( c == 0 )
		c = strcmp((*a)->msgid, msgid);
	    if (c<0)
		a = &((*a)->left);
	    else if (c>0)
		a = &((*a)->right);
	    else {
		return;
	    }
	} else {
	    *a = (struct msgidtree *)
		 critmalloc(sizeof(struct msgidtree) + strlen(msgid),
			    "Building expiry database");
	    (*a)->left = (*a)->right = NULL;
	    strcpy((*a)->msgid, msgid);
	    (*a)->art = art;
	    return;
	}
    }
}

int findmsgid( const char* msgid ) {
    struct msgidtree * a;
    int c;
    char * domainpart;

    /* domain part differs more than local-part, so try it first */

    domainpart = strchr( msgid, '@' );
    if ( domainpart == NULL )
	return 0;

    a = head;
    while (a) {
	c = strcmp(strchr(a->msgid, '@'), domainpart);
	if ( c == 0 )
	    c = strcmp(a->msgid, msgid);
	if ( c < 0 )
	    a = a->left;
	else if ( c > 0 )
	    a = a->right;
	else
	    return a->art;
    }
    return 0;
}

static void begone( struct msgidtree * m ) {
    if ( m ) {
	begone( m->right ) ;
	begone( m->left );
	free( (char *) m );
    }
}

void clearidtree( void ) {
    if ( head ) {
	begone( head->right ) ;
	begone( head->left );
	free( (char *) head );
    }
    head = NULL;
}


const char* rfctime(void) {
    static char date[128];
    const char * months[] = { "Jan", "Feb", "Mar", "Apr",
	"May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };
    const char * days[] = { "Sun", "Mon", "Tue", "Wed",
	"Thu", "Fri", "Sat" };
    time_t now;
    struct tm gmt;
    struct tm local;
    int    hours, mins;
#ifndef HAVE_GMTOFF
    extern long int timezone;
#endif

    /* get local and Greenwich times */
    now = time(0);
    gmt	  = *(gmtime(&now));
    local = *(localtime(&now));

#ifdef HAVE_GMTOFF
    hours = local.tm_gmtoff/60/60;
    mins  = local.tm_gmtoff/60%60;
#else
    /* this is empirical */
    hours = -timezone/60/60+((local.tm_isdst>0)?1:0);
    mins  = -timezone/60%60;
#endif

    /* finally print the string */
    sprintf(date, "%3s, %d %3s %4d %02d:%02d:%02d %+03d%02d",
	    days[local.tm_wday], local.tm_mday, months[local.tm_mon],
	    local.tm_year+1900, local.tm_hour, local.tm_min, local.tm_sec,
	    hours, mins);

    return (date);
}


#ifndef HAVE_STRDUP
char * strdup(const char *s) {
    char * s1;
    s1 = malloc(strlen(s)+1);
    if (!s1)
	return 0;
    strcpy (s1,s);
    return s1;
}
#endif

#ifndef HAVE_SNPRINTF
/*
 * poor man's snprintf() - for systems which don't have their own
 *
 * This version of snprintf() currently supports only %s, %c, %d, %u, %ld, %lu.
 * It also behaves differently from standard snprintf() in that if the
 * output is truncated, it will return -1. Otherwise, it will return the
 * number of printed chars.
 *
 * Copyright (c) Cornelius Krasel 2000.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

int snprintf( char *str, size_t n, const char *format, ... ) {
    int rval;
    va_list ap;
    const char *p;
    char *q;
    int flag  = 0;
    int lflag = 0;	/* checking for longs */
    int i     = 1;	/* because the terminating \0 also counts */
    size_t len;
    char   buf[30];	/* buffer for converting longs and ints */

    int  d;
    long l;
    char *s;
    char c;

    va_start( ap, format );
    p = format;
    q = str;
    while ( p && *p && ( i < n ) ) {
    	if ( ( *p == '%' )  && !flag ) {
	    lflag = 0;
	    flag  = 1;
	    p++;
	}
	else if ( flag ) {
	    switch (*p)  {
		case 's': {
		    s = va_arg( ap, char * );
		    len = strlen( s );
		    if ( len > (n-i) )
			len = n - i;
		    *q = '\0';
		    strncat( q, s, len );
		    p++;
		    q += len;
		    i += len;
		    flag = 0;
		    break;
		}
		case 'u': {
		    if ( lflag ) {
			l = va_arg( ap, unsigned long );
			sprintf( buf, "%lu", l );
		    }
		    else {
			d = va_arg( ap, unsigned int );
			sprintf( buf, "%u", d );
		    }
                    len = strlen( buf );
                    if ( len > (n-i) )
                        len = n-i;
                    *q = '\0';
                    strncat( q, buf, len );
                    q += len;
                    i += len;
		    lflag = 0;
		    flag  = 0;
		    p++;
		    break;
		}
		case 'd': {
		    if ( lflag ) {
			l = va_arg( ap, long );
			sprintf( buf, "%ld", l );
		    }
		    else {
			d = va_arg( ap, int );
			sprintf( buf, "%d", d );
		    }
                    len = strlen( buf );
                    if ( len > (n-i) )
                        len = n-i;
                    *q = '\0';
                    strncat( q, buf, len );
                    q += len;
                    i += len;
		    lflag = 0;
		    flag  = 0;
		    p++;
		    break;
		}
		case 'l': {
		    /* next argument will be long */
		    lflag = 1;
		    p++;
		    break;
		}
		case 'c': {
		    c = va_arg( ap, char );
		    lflag = 0;
		    flag  = 0;
		    i++;
		    *q++ = c;
		    p++;
		    break;
		}
		case '%': {
		    lflag = 0;
		    flag  = 0;
		    i++;
		    *q++ = *p++;
		    break;
		}
	    }
	}
	else {
	    lflag = 0;
	    i++;
	    *q++ = *p++;
	}
    }
    va_end( ap );
    *q = '\0';
    if ( i >= n )
        return -1;
    else
	return (i-1);
}

#endif

/*
 * Lars Wirzenius: read a line into memory, with no max length
 * return a pointer to the line, or NULL in case of error
 *
 * strip \r at EOL
 */
char *getaline(FILE *f) {
    static char *buf;       /* buffer for line */
    static size_t size;     /* size of buffer */
    size_t len;             /* # of chars stored into buf before '\0' */
    char * p;
    int i;

    len = 0;
    if (!size)
	size = 256;
    if (!buf)
	buf = critmalloc( size, "reading line" );

    while ((p=fgets(buf+len, size-len, f))) {
	/* replace NUL bytes (which shouldn't be there anyway)
	 * with spaces
	 */
	i = size - len;
	while ( *p != '\n' && --i > 0 ) {
	    if ( *p == '\0' )
		*p = ' ';
	    p++;
	}
	len += strlen(buf+len);
	if ( buf[len-1] == '\n' )
	    break;		/* the whole line has been read */

	size += size;		/* exponential growth of buffer */
	buf = critrealloc(buf, size, "reading line" );
    }

    if ( len == 0 )
	return NULL;

    if (len && (buf[len-1] == '\n')) { /* go back on top of the newline */
	--len;
	if (len && (buf[len-1] == '\r')) /* also delete CR */
	    --len;
    }

    buf[len] = '\0';        /* unconditionally terminate string,
                               possibly overwriting newline */

    if ( debug )
        syslog( LOG_DEBUG, "<%s\n", buf );
    return buf;
}

int ngmatch(const char* pattern, const char* str) {
#ifdef HAVE_FNMATCH
    if ( strchr( pattern, '?' ) || !strchr( pattern, '*' ) ||
	 !strchr( pattern, '[' ) ) {
	/* pattern contains something wildcard-like */
	if ( fnmatch( pattern, str, 0 ) == 0 )
	    return 0;
	else
	    return 1;
    }
    else {
	/* no wildcards - we can do case-insensitive comparison */
	if ( strcasecmp( pattern, str ) == 0 )
	    return 0;
	else
	    return 1;
    }
#else
    /* poor man's fnmatch() - but it is case-insensitive! */
    int len = strlen(pattern);
    if (pattern[len - 1] == '*') {
	/* If pattern is terminated with '*', treat it as *. */
	/* Therefore, compare a part of string before '*'. */
	if (strncasecmp(pattern, str, len - 1) == 0)
	    return 0;
    } else {
	/* Otherwise, treat string as just string. */
	if (strcasecmp(pattern, str) == 0)
	    return 0;
    }
    return 1;
#endif
}
