/*	$Id: ldapvacation.c,v 1.47 2006/03/12 09:34:04 mbalmer Exp $	*/

/*
 * Copyright (c) 2005,2006 Marc Balmer <marc@msys.ch>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

/*
 * ldapvacation is based on Vacation
 * Copyright (c) 1983  Eric P. Allman
 * Berkeley, California
 */

#include <sys/param.h>
#include <sys/stat.h>

#include <ctype.h>
#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <ldap.h>
#include <paths.h>
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <time.h>
#include <unistd.h>

#include "ldapvacation.h"
#include "pathnames.h"

#ifdef __linux__
#include "strlfunc.h"
#include "strtonum.h"
#endif

/*
 * LDAPVACATION -- return a message to the sender when on vacation.
 *
 * This program is invoked as a message receiver.  It returns a
 * message specified by the user to whomever sent the mail, taking
 * care not to return a message too often to prevent "I am on
 * vacation" loops.
 */

#define	MAXLINE	1024			/* max line from mail header */

#define MAXLEN		256
#define MAXSTR		256

#define LDAPHOST	"localhost"
#define LDAPPORT	389
#define LDAPURL		NULL
#define LDAPQUERY	"(&(objectClass=vmailAccount)(vacationEnabled=TRUE)(uid=$u))"
#define MAILATTR	"mailLocalAddress"

#if (__linux__)
#define SECSPERMIN      60
#define MINSPERHOUR     60
#define HOURSPERDAY     24
#define DAYSPERWEEK     7
#define SECSPERHOUR     (SECSPERMIN * MINSPERHOUR)
#define SECSPERDAY      ((long) SECSPERHOUR * HOURSPERDAY)
#else
#include <tzfile.h>
#endif

LDAP	*ld;
char	*ldaphost;
unsigned int	 ldapport;
char	*ldapurl;
char	*searchbase;
char	*binddn;
char	*bindpasswd;
int	 ldap_use_tls;
char	*query;
char	*cfgfile;
char	*mailattr;
char	*vacationSent;
int	 log_facility;
int	 verbose;
char	*spamheader;

extern char *__progname;

/* This should really use our queues. */

typedef struct alias {
	struct alias	*next;
	char		*name;
} ALIAS;
ALIAS *names;

char from[MAXLINE];
char subj[MAXLINE];
time_t interval;
int ldap_has_interval;
char *dn;

int junkmail(void);
int nsearch(char *, char *);
void readheaders(void);
int recent(void);
void sendmessage(char *, volatile char *);
void setinterval(time_t);
void setreply(void);
void usage(void);
extern void ldapvacation_init(void);

/*
 * readheaders --
 *	read mail headers
 */
void
readheaders(void)
{
	char buf[MAXLINE], *p;
	int tome, cont;
	ALIAS *cur;

	cont = tome = 0;
	while (fgets(buf, sizeof(buf), stdin) && *buf != '\n')
		switch (*buf) {
		case 'F':		/* "From " */
		case 'f':
			cont = 0;
			if (!strncasecmp(buf, "From ", 5)) {
				for (p = buf + 5; *p && *p != ' '; ++p)
					;
				*p = '\0';
				(void)strlcpy(from, buf + 5, sizeof(from));
				if ((p = strchr(from, '\n')))
					*p = '\0';
				if (junkmail())
					exit(0);
			}
			break;
		case 'R':		/* "Return-Path:" */
		case 'r':
			cont = 0;
			if (strncasecmp(buf, "Return-Path:",
			    sizeof("Return-Path:")-1) ||
			    (buf[12] != ' ' && buf[12] != '\t'))
				break;
			for (p = buf + 12; *p && isspace(*p); ++p)
				;
			if (strlcpy(from, p, sizeof(from)) >= sizeof(from)) {
				syslog(LOG_NOTICE,
				    "Return-Path %s exceeds limits", p);
				exit(1);
			}
			if ((p = strchr(from, '\n')))
				*p = '\0';
			if (junkmail())
				exit(0);
			break;
		case 'P':		/* "Precedence:" */
		case 'p':
			cont = 0;
			if (strncasecmp(buf, "Precedence", 10) ||
			    (buf[10] != ':' && buf[10] != ' ' &&
			    buf[10] != '\t'))
				break;
			if (!(p = strchr(buf, ':')))
				break;
			while (*++p && isspace(*p))
				;
			if (!*p)
				break;
			if (!strncasecmp(p, "junk", 4) ||
			    !strncasecmp(p, "bulk", 4) ||
			    !strncasecmp(p, "list", 4))
				exit(0);
			break;
		case 'S':		/* Subject: */
		case 's':
			cont = 0;
			if (strncasecmp(buf, "Subject:",
			    sizeof("Subject:")-1) ||
			    (buf[8] != ' ' && buf[8] != '\t'))
				break;
			for (p = buf + 8; *p && isspace(*p); ++p)
				;
			if (strlcpy(subj, p, sizeof(subj)) >= sizeof(subj)) {
				syslog(LOG_NOTICE,
				    "Subject %s exceeds limits", p);
				exit(1);
			}
			if ((p = strchr(subj, '\n')))
				*p = '\0';
			break;
		case 'C':		/* "Cc:" */
		case 'c':
			if (strncasecmp(buf, "Cc:", 3))
				break;
			cont = 1;
			goto findme;
		case 'T':		/* "To:" */
		case 't':
			if (strncasecmp(buf, "To:", 3))
				break;
			cont = 1;
			goto findme;
		case 'X':		/* X-Spam-Status: spam" */
		case 'x':
			if (spamheader != NULL && !strncasecmp(buf, spamheader, strlen(spamheader)))
				exit(0);
			break;
		default:
			if (!isspace(*buf) || !cont || tome) {
				cont = 0;
				break;
			}
findme:			for (cur = names; !tome && cur; cur = cur->next)
				tome += nsearch(cur->name, buf);
		}

	if (!tome)
		exit(0);
	if (!*from) {
		syslog(LOG_INFO, "no initial \"From\" or \"Return-Path\"line.");
		exit(0);
	}
}

/*
 * nsearch --
 *	do a nice, slow, search of a string for a substring.
 */
int
nsearch(char *name, char *str)
{
	size_t len;

	for (len = strlen(name); *str; ++str)
		if (!strncasecmp(name, str, len))
			return(1);
	return(0);
}

/*
 * junkmail --
 *	read the header and return if automagic/junk/bulk/list mail
 */
int
junkmail(void)
{
	static struct ignore {
		char	*name;
		size_t	 len;
	} ignore[] = {
		{ "-request", 8 },
		{ "postmaster", 10 },
		{ "uucp", 4 },
		{ "mailer-daemon", 13 },
		{ "mailer", 6 },
		{ "-relay", 6 },
		{ NULL, 0 }
	};
	struct ignore *cur;
	int len;
	char *p;

	/*
	 * This is mildly amusing, and I'm not positive it's right; trying
	 * to find the "real" name of the sender, assuming that addresses
	 * will be some variant of:
	 *
	 * From site!site!SENDER%site.domain%site.domain@site.domain
	 */
	if (!(p = strchr(from, '%'))) {
		if (!(p = strchr(from, '@'))) {
			if ((p = strrchr(from, '!')))
				++p;
			else
				p = from;
			for (; *p; ++p)
				;
		}
	}
	len = p - from;
	for (cur = ignore; cur->name; ++cur)
		if (len >= cur->len &&
		    !strncasecmp(cur->name, p - cur->len, cur->len))
			return(1);
	return(0);
}

/*
 * recent --
 *	find out if user has gotten a vacation message recently.
 *	use bcopy for machines with alignment restrictions
 */
int
recent(void)
{
	time_t then, next;
	char recent_query[MAXLINE];
	LDAPMessage *result, *e;
	char *attrs[] = {	"vacationSent",
				NULL};
	char **vals;
	char *s;				
	int retval = 0;

	int n;
	size_t l;
	
	next = interval == -1 ? (time_t)SECSPERDAY * (time_t)DAYSPERWEEK : interval;
	
	if (*query == '(')
		snprintf(recent_query, sizeof(recent_query), "(&%s(vacationSent=%s *))", query, from);
	else
		snprintf(recent_query, sizeof(recent_query), "(&(%s)(vacationSent=%s *))", query, from);

	if (ldap_search_s(ld, searchbase, LDAP_SCOPE_SUBTREE, recent_query, attrs, 0, &result) == LDAP_SUCCESS) {
		if (ldap_count_entries(ld, result)) {
			e = ldap_first_entry(ld, result);
			vals = ldap_get_values(ld, e, "vacationSent");
			for (n = 0, l = strlen(from); vals[n] != NULL; n++) {
				if (!strncasecmp(vals[n], from, l)) {
					if ((vacationSent = strdup(vals[n])) == NULL) {
						syslog(LOG_ERR, "memory allocation error");
						exit(1);
					}
					if ((s = strchr(vals[n], ' '))) {
						then = (time_t) atol(++s);

						if (next == (time_t) LONG_MAX ||
							then + next > time(NULL)) {
							retval = 1;
							break;
						}
					}
				}
			}
			ldap_value_free(vals);
		}
		ldap_msgfree(result);
	}
	return retval;
}

/*
 * setreply --
 *	store that this user knows about the vacation.
 */
void
setreply(void)
{
	LDAPMod mod;
	LDAPMod *mods[2];
	char *vacationvals[2];
	char entry[MAXLINE];

	vacationvals[1] = NULL;
	mod.mod_type = "vacationSent";
	mod.mod_values = vacationvals;
	mods[0] = &mod;
	mods[1] = NULL;

	if (vacationSent != NULL) {
		vacationvals[0] = vacationSent;
		mod.mod_op = LDAP_MOD_DELETE;
		if (ldap_modify_s(ld, dn, mods) != LDAP_SUCCESS) {
			syslog(LOG_ERR, "Can't delete old vacation sender for %s", dn);
			exit(1);
		}
	}

	snprintf(entry, sizeof(entry), "%s %ld", from, (long) time(NULL));

	vacationvals[0] = entry;
	mod.mod_op = LDAP_MOD_ADD;
	if (ldap_modify_s(ld, dn, mods) != LDAP_SUCCESS) {
		syslog(LOG_ERR, "Can't add vacation sender for %s", dn);
		exit(1);
	}
}

/*
 * sendmessage --
 *	exec sendmail to send the vacation msg to sender
 */
void
sendmessage(char *myname, volatile char *msg)
{
	FILE *sfp;
	char cmd[MAXLINE];
	char *s;

#if 0
	int pvect[2], i;

	The original vacation program uses vfork() to start sendmail.
	After problems using this from maildrop, I switched to popen().
	
	if (pipe(pvect) < 0) {
		syslog(LOG_ERR, "pipe: %m");
		exit(1);
	}
	i = vfork();
	if (i < 0) {
		syslog(LOG_ERR, "fork: %m");
		exit(1);
	}
	if (i == 0) {
		dup2(pvect[0], 0);
		close(pvect[0]);
		close(pvect[1]);
		execl(_PATH_SENDMAIL, "sendmail", "-v", "-f", myname, "--",
		    from, (char *)NULL);
		syslog(LOG_ERR, "can't exec %s: %m", _PATH_SENDMAIL);
		_exit(1);
	}
	close(pvect[0]);
	sfp = fdopen(pvect[1], "w");
	if (sfp == NULL) {
		close(pvect[1]);
		return;
	}
#endif

	snprintf(cmd, sizeof(cmd), "%s -f %s -- %s", _PATH_SENDMAIL, myname, from);

	if (verbose)
		syslog(LOG_INFO, "send vacation msg from %s to %s using command '%s'", myname, from, cmd);

	sfp = popen(cmd, "w");
	if (sfp == NULL) {
		syslog(LOG_ERR, "cant open pipe");
		exit(1);
	}
	/* XXX: Since we pull the message from LDAP, use UTF-8 encoding */
	fprintf(sfp, "Content-Type: text/plain; charset=UTF-8\n");
	fprintf(sfp, "Content-transfer-encoding: 8bit\n");
	fprintf(sfp, "To: %s\n", from);

	while ((s = strstr((char *) msg, "$SUBJECT")) != NULL) {
		*s = '\0';
		fputs((char *) msg, sfp);
		fputs(subj, sfp);
		msg = s + 8;
	}
	fputs((char *) msg, sfp);

	pclose(sfp);
}

__dead void
usage(void)
{
	fprintf(stderr, "usage: %s [-tVvZ[Z]] [-b searchbase] [-C configfile] [-D binddn] "
	    "[-d dest] [-f from] [-h ldaphost] [-L ldapport] "
	    "[-m mailattr] [-q query] [-u ldapurl] [-w bindpasswd] [-x spamheader]\n", __progname);
	exit(1);
}

void
exithandler(void)
{
	closelog();
	ldap_unbind(ld);
}

int
main(int argc, char *argv[])
{
	int ch;
	int version;
	char *s;
	char *p;
	char *uid;
	ALIAS *cur;
	struct passwd *pw;
	char *attrs[] = {	"vacationMsg",
			"vacationInterval",
			"vacationEnabled",
			NULL,	/* mailattr will be put here */
			NULL
		};
	LDAPMessage *result, *e;
	char *myname;
	size_t size;
	char **vals;
	const char *errstr;

	/* Set initial values */

	verbose = 0;

	log_facility = LOG_MAIL;
	ldaphost = NULL;
	ldapport = 0;
	ldapurl = NULL;
	searchbase = NULL;
	binddn = NULL;
	bindpasswd = NULL;
	query = NULL;
	cfgfile = _PATH_CONFIG;
	ldap_use_tls = -1;
	interval = -1;
	uid = NULL;
	mailattr = NULL;
	myname = NULL;
	vacationSent = NULL;
	spamheader = NULL;
	
	/* Process the commandline */

	while ((ch = getopt(argc, argv, "b:C:Dd:f:h:m:p:q:tu:Vvw:x:?")) != -1) {
		switch (ch) {
		case 'b':
			searchbase = optarg;
			break;
		case 'C':
			cfgfile = optarg;
			break;
		case 'D':
			binddn = optarg;
			break;
		case 'd':
			uid = optarg;
			break;
		case 'f':
			if (strlen(optarg)) {
				strlcpy(from, optarg, sizeof(from));
				if (junkmail())
					exit(0);
			}
			break;
		case 'h':
			ldaphost = optarg;
			break;
		case 'm':
			mailattr = optarg;
			break;
		case 'p':
			ldapport = (unsigned int)strtonum(optarg, 1, 65535, &errstr);
			if (errstr)
				errx(1, "port number is %s:%s", errstr, optarg);
			break;
		case 'q':
			query = optarg;
			break;
		case 't':
			ldap_use_tls = 1;
			break;
		case 'u':
			ldapurl = optarg;
			break;
		case 'V':
			printf("%s %s\n", __progname, VERSION);
			exit(0);
			/* FALLTHROUGH */
		case 'v':
			verbose = 1;
			break;
		case 'w':
			bindpasswd = optarg;
			break;
		case 'x':
			if (*optarg == 'x' || *optarg == 'X')
				spamheader = optarg;
			else {
				fprintf(stderr, "Spam header must start with 'x' or 'X'");
				exit(1);
			}
			break;
		case 'Z':
			switch (ldap_use_tls) {
			case -1:
				ldap_use_tls = 1;
				break;
			case 1:
				ldap_use_tls = 2;
				break;
			default:
				usage();
				/* NOTREACHED */
			}
			break;
		default:
			usage();
			/* NOTREACHED */
		}
	}

	argc -= optind;
	argv += optind;

	/* Read config file */

	ldapvacation_init();

	if (uid == NULL) {
		if (!(pw = getpwuid(getuid()))) {
			syslog(LOG_ERR,
			    	"no such user uid %u.", getuid());
			exit(1);
		}

		uid = pw->pw_name;
	}

	/* Set default values if some variables are not set */

	if (ldaphost == NULL)
		ldaphost = LDAPHOST;
	if (ldapport == 0)
		ldapport = LDAPPORT;
	if (ldapurl == NULL)
		ldapurl = LDAPURL;
	if (ldap_use_tls == -1)
		ldap_use_tls = 0;
	if (query == NULL)
		query = strdup(LDAPQUERY);
	if (mailattr == NULL)
		mailattr = MAILATTR;

	attrs[3] = mailattr;

	if (searchbase == NULL) {
		syslog(LOG_ERR, "No LDAP basedn set");
		exit(1);
	}

	if (binddn == NULL) {
		syslog(LOG_ERR, "No bind DN set");
		exit(1);
	}

	openlog(__progname, LOG_CONS | LOG_NDELAY | LOG_PID | verbose ? LOG_PERROR : 0, log_facility);

	if (ldapurl == NULL) {
		if ((ld = ldap_init(ldaphost, ldapport)) == NULL) {
			syslog(LOG_ERR, "No directory server at %s, %m", ldaphost);
			exit(1);
		}
	} else {
		if (ldap_initialize(&ld, ldapurl)) {
			syslog(LOG_ERR, "No directory server at %s, %m", ldapurl);
			exit(1);
		}
	}

	version = LDAP_VERSION3;
	if (ldap_set_option(ld, LDAP_OPT_PROTOCOL_VERSION, &version) != LDAP_OPT_SUCCESS) {
		syslog(LOG_ERR, "Failed to set LDAP version 3 protocol");
		exit(1);
	}

	if (ldap_use_tls > 0 && ldap_start_tls_s(ld, NULL, NULL) != LDAP_SUCCESS) {
		syslog(LOG_ERR, "Failed to start TLS for LDAP");
		if (ldap_use_tls > 1)
			exit(1);
	}

	if (ldap_simple_bind_s(ld, binddn, bindpasswd) != LDAP_SUCCESS) {
		syslog(LOG_ERR, "Failed to bind to directory server as '%s'", binddn);
		memset(bindpasswd, 0L, strlen(bindpasswd));
		exit(1);
	}
	bzero(bindpasswd, strlen(bindpasswd));

	if (verbose)
		syslog(LOG_INFO, "bound to LDAP server as %s", binddn);

	if (atexit(exithandler)) {
		exithandler();
		syslog(LOG_ERR, "Can't install exit handler");
		exit(1);
	}

	if ((s = strstr(query, "$u")) != NULL) {
		size = strlen(query) + strlen(uid) - 1;

		if ((p = malloc(size)) == NULL) {
			syslog(LOG_ERR, "memory allocation error");
			exit(1);
		}

		bzero(p, size);

		*s = 0;
		strlcpy(p, query, size);
		strlcat(p, uid, size);
		strlcat(p, s+2, size);
		query = p; 
	}

	if (verbose)
		syslog(LOG_INFO, "LDAP query: %s", query);

	if (ldap_search_s(ld, searchbase, LDAP_SCOPE_SUBTREE, query, attrs, 0, &result) == LDAP_SUCCESS) {
		if (!ldap_count_entries(ld, result))
			exit(0);	

		dn = NULL;

		if ((e = ldap_first_entry(ld, result)) != NULL)
			dn = ldap_get_dn(ld, e);

		if (dn == NULL) {
			syslog(LOG_ERR, "Can't get dn");
			exit(1);
		}

		if (verbose)
			syslog(LOG_INFO, "DN: %s", dn);

		if ((vals = ldap_get_values(ld, e, "vacationInterval")) != NULL) {
			interval = (time_t) atol(vals[0]);
			ldap_value_free(vals);
		}

		if ((vals = ldap_get_values(ld, e, mailattr)) != NULL) {
			for (; *vals != NULL; vals++) {
				if (myname == NULL)
					myname = *vals;
				if ((cur = (ALIAS *) malloc(sizeof(ALIAS))) == NULL) {
					syslog(LOG_ERR, "memory error");
					exit(1);
				}
				if (verbose)
					syslog(LOG_INFO, "%s: %s", mailattr, *vals);
					
				cur->name = *vals;
				cur->next = names;
				names = cur;
			}			
		}

		if (myname == NULL)
			myname = uid;

		if (!(cur = malloc(sizeof(ALIAS))))
			exit(1);
		cur->name = uid;
		cur->next = names;
		names = cur;

		readheaders();

		if (!recent()) {
			if (verbose)
				syslog(LOG_INFO, "not recent");
			if ((vals = ldap_get_values(ld, e, "vacationMsg")) != NULL) {
				sendmessage(myname, vals[0]);
				setreply();
				ldap_value_free(vals);
			}
		}
	} else if (verbose)
		syslog(LOG_INFO, "query failed");

	ldap_unbind(ld);
	return 0;
}
