/*
 * Copyright (c) 2001-2002 The Trustees of Indiana University.  
 *                         All rights reserved.
 * Copyright (c) 1998-2001 University of Notre Dame. 
 *                         All rights reserved.
 * Copyright (c) 1994-1998 The Ohio State University.  
 *                         All rights reserved.
 * 
 * This file is part of the LAM/MPI software package.  For license
 * information, see the LICENSE file in the top level directory of the
 * LAM/MPI source distribution.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	GDB
 *
 *	$Id: tkill.c,v 6.14.2.2 2002/10/09 19:49:59 brbarret Exp $
 *
 *	Function:	- kills OTB environment
 *			- reads process IDs from the kill file, aka
 *			  reset file, aka lock file for historical reasons
 *			- makes a very good attempt at ensuring the
 *			  death of each pid in the file
 */

#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <dirent.h>

#include <args.h>
#include <debug.h>
#include <terror.h>
#include <laminternal.h>

#define TIMEOUT		10000000	/* timeout on process death */


/*
 * global variables
 */
int		fl_debug;	        /* debugging option */

/*
 * local functions
 */
static void lam_rmdir(char *dir);


int
main(int argc, char *argv[])
{
	FILE		*fp_kill;	/* kill file ptr */
	int		fd_kill;	/* kill file desc. */
	int		fl_pretend;	/* pretend option */
	int		fl_verbose;	/* verbose option */
	int		n;		/* favourite counter */
	int		pid;		/* lots of space to hold a PID */
	unsigned int	usec;		/* microseconds to sleep */
	char		*f_kill;	/* kill file */
	char		*f_sock;	/* socket file */
	char		*f_iosock;	/* io daemon socket file */
	char            extra[1024];    /* place for ASCII messages */

/*
 * Initialize option parser.
 */
	validopts("dhvN");
	followed("fb");

	if ((do_args(&argc, argv)) || (errno = (argc == 1) ? 0 : EUSAGE)) {
	    show_help("tkill", "usage", NULL);
	    exit(errno);
	}

	/* Root can only run this program is "-N" is supplied (i.e., if it
	 * was launched from recon */

	fl_pretend = opt_taken('N');
	if (!fl_pretend && (getuid() == 0 || geteuid() == 0)) {
	  show_help(NULL, "deny-root", NULL);
	  exit(EACCES);
	}

	if (opt_taken('h')) {
	    show_help("tkill", "usage", NULL);
	    exit(0);
	}

	fl_debug = opt_taken('d');
	fl_verbose = opt_taken('v') || fl_debug;
	if (opt_taken('b')) {
	  if (fl_debug)
	    printf("tkill: setting killname to %s\n", getparam('b'));
	  set_killname(getparam('b'));
	  if (fl_debug)
	    printf("tkill: got killname back: %s\n", killname());
	}
/*
 * Remove the socket file.
 */
	DBUG("tkill: removing socket file ...\n");
	f_sock = sockname();

	if (f_sock == 0) {
	  show_help("lam-temp-files", "badness", "tkill/sockname", NULL);
	  exit(errno);
	}

	if (!fl_pretend) {
	    DBUG("tkill: socket file: %sd\n", f_sock);
	    if (unlink(f_sock)) {
		if (errno != ENOENT) {
		  snprintf(extra, 1024, "unlink(\"%s\")", f_sock);
		  show_help(NULL, "system-call-fail", extra, NULL);
		}
	    }
	}
/*
 * Remove the IO daemon socket file.
 */
	DBUG("tkill: removing IO daemon socket file ...\n");
	f_iosock = iosockname();

	if (f_iosock == 0) {
	  show_help("lam-temp-files", "badness", "tkill/iosockname", NULL);
	  exit(errno);
	}

	if (!fl_pretend) {
	  DBUG("tkill: IO daemon socket file: %s\n", f_iosock);
	    if (unlink(f_iosock)) {
		if (errno != ENOENT) {
		  snprintf(extra, 1024, "unlink(\"%s\")", f_iosock);
		  show_help(NULL, "system-call-fail", extra, NULL);
		}
	    }
	}
/*
 * Get the kill filename.
 */
	if (opt_taken('f')) {
	    f_kill = getparam('f');
	} else {
	    f_kill = killname();

	    if (f_kill == 0) {
	      show_help(NULL, "lib-call-fail", "killname", NULL);
	      exit(errno);
	    }
	}
/*
 * Dump debugging information.
 */
	DBUG("tkill: f_kill = \"%s\"\n", f_kill);
/*
 * Try to open the file.
 */
	if ((fd_kill = open(f_kill, O_RDWR, 0)) < 0) {

	    if (errno == ENOENT) {
		VERBOSE("tkill: nothing to kill: \"%s\"\n", f_kill);
		exit(0);
	    } else {
	      snprintf(extra, 1024, "open(\"%s\", O_RDWR)", f_kill);
	      show_help(NULL, "system-call-fail", extra, NULL);
	      exit(errno);
	    }
	}

	fp_kill = fdopen(fd_kill, "r");

	if (! fp_kill) {
	  snprintf(extra, 1024, "fdopen(\"%s\", \"r\")", f_kill);
	  show_help(NULL, "lib-call-fail", extra, NULL);
	  exit(errno);
	}

	VERBOSE("tkill: killing LAM...\n");

	n = fscanf(fp_kill, "%d", &pid);

	while (n > 0) {
	    DBUG("tkill: killing PID %d ...", pid);

	    if (fl_debug) 
	      fflush(stdout);
/*
 * Send SIGHUP to the process.
 */
	    if (fl_pretend) {
		DBUG("\n");
		n = fscanf(fp_kill, "%d", &pid);
		continue;
	    }

	    if (kill(pid, SIGHUP)) {

		if (errno == ESRCH) {
		    DBUG(" already dead");
		} else {
		    DBUG("\n");
		    perror("tkill (kill)");
		    exit(errno);
		}
	    } else {
		usec = 2;

		while ((kill(pid, SIGHUP) == 0) && (usec <= TIMEOUT)) {
		    microsleep(usec);
		    usec *= 2;
		}
	
		if (usec > TIMEOUT) {
		    DBUG(" trying -9 ...");
		    kill(pid, SIGKILL);
		    microsleep(500000);

		    if (kill(pid, SIGHUP)) {
			DBUG(" killed");
		    } else {
			DBUG(" cannot kill");

			snprintf(extra, 1024, "%d", pid);
			show_help("tkill", "cannot-kill", extra, NULL);
		    }
		} else {
		    DBUG(" killed");
		}
	    }
		
	    DBUG("\n");
	    n = fscanf(fp_kill, "%d", &pid);
	}

	if (fl_pretend) {
	    exit(0);
	}
/*
 * Cleanup all registered objects.
 */
	lam_cleanup_objects();

/*
 * Remove the session directory.  Go remove all entries in it first.
 */
	lam_rmdir(f_kill);

	free(f_kill);
	free(f_sock);
	free(f_iosock);
	return(0);
}



/*
 * Remove all files in a directory and then remove the directory
 */
static void
lam_rmdir(char *dir)
{
  int ret, badness;
  DIR *dp;
  struct dirent *entry;
  char *s, *name = strdup(dir);

  s = strrchr(name, '/');
  *s = '\0';
  DBUG("tkill: removing session directory \"%s\"\n", name);

  /* Try to go to that directory */

  do {
    ret = chdir(name);
    if (ret == -1) {
      if (errno == EINTR)
	continue;
      else if (errno == ENOENT) {
	free(name);
	return;
      } else {
	show_help(NULL, "unable-rmdir", "tkill", name, NULL);
	free(name);
	return;
      }
    }
  } while (ret != 0);

  /* First, remove all files in the session directory */

  dp = opendir(name);
  if (dp == NULL) {
    show_help(NULL, "unable-rmdir", "tkill", name, NULL);
    free(name);
    return;
  }
  badness = 0;
  while ((entry = readdir(dp)) != NULL) {
    if (strcmp(entry->d_name, ".") == 0 ||
	strcmp(entry->d_name, "..") == 0)
      continue;

    DBUG("tkill: removing extraneous file \"%s\"\n", entry->d_name);
    do {
      ret = remove(entry->d_name);
      if (ret == -1) {
	if (errno == EINTR)
	  continue;
	else
	  badness++;
      }
    } while (ret != 0);
  }
  closedir(dp);

  /* Did we remove everything successfully? */

  if (badness != 0) {
    show_help(NULL, "unable-rmdir", "tkill", name, NULL);
    free(name);
    return;
  }

  /* Move back one directory so that we can remove the target dir */

  do {
    ret = chdir("..");
    if (ret == -1) {
      if (errno == EINTR)
	continue;
      else {
	show_help(NULL, "unable-rmdir", "tkill", name, NULL);
	free(name);
	return;
      }
    }
  } while (ret != 0);

  /* Now go remove the directory itself */

  DBUG("tkill: removing the directory\n");
  do {
    ret = rmdir(name);
    if (ret == -1) {
      if (errno == EINTR)
	continue;
      else {
	show_help(NULL, "unable-rmdir", "tkill", name, NULL);
	free(name);
	return;
      }
    }
  } while (ret != 0);

  free(name);
}
