/*
 * Copyright (c) 2001-2002 The Trustees of Indiana University.  
 *                         All rights reserved.
 * Copyright (c) 1998-2001 University of Notre Dame. 
 *                         All rights reserved.
 * Copyright (c) 1994-1998 The Ohio State University.  
 *                         All rights reserved.
 * 
 * This file is part of the LAM/MPI software package.  For license
 * information, see the LICENSE file in the top level directory of the
 * LAM/MPI source distribution.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	JRV/RBD/GDB
 *
 *	$Id: inetexec.c,v 6.32.2.1 2002/10/09 19:48:50 brbarret Exp $
 * 
 *	Function:	- run program on remote UNIX machine
 *	Accepts:	- hostname
 *			- username (or NULL)
 *			- argv structure
 *	Returns:	- 0 or LAMERROR
 */

#include <lam_config.h>

#include <errno.h>
#include <stdio.h>
#include <string.h>
#if HAVE_STRINGS_H
#include <strings.h>
#endif
#include <stdlib.h>
#include <ctype.h>
#include <pwd.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/types.h>
#if LAM_NEED_SYS_SELECT_H
#include <sys/select.h>
#endif
#include <sys/wait.h>
#include <sys/param.h>

#include <args.h>
#include <typical.h>
#include "sfh.h"

#ifndef MAXPATHLEN
#define MAXPATHLEN	1024
#endif


/*
 * static functions
 */
static int ioexecvp(char **cmdv, int showout, char *outbuff, int outbuffsize);
static void add_rsh(int *argc, char ***argv);

int
inetexec(char *host, char *username, char **argv, char *prefix,
	 int fl_fast)
{
	char		**cmdv;			/* rsh command argv */
	int		cmdc;			/* rsh command argc */
	int		fl_csh;			/* csh-flavoured flag */
	int		fl_bash;		/* bash-flavoured flag */
	int		i;			/* favorite counter */
	char            printable[BUFSIZ];      /* command to exec */
	char            remote_host[BUFSIZ];    /* username@hostname */
	char            *cmdv0;                 /* Copy of cmdv[0] */
	char            shellpath[MAXPATHLEN];	/* return of 'echo $SHELL' */
	struct passwd   *p;
/*
 * Get the user's shell by executing 'echo $SHELL' on remote machine,
 * but only if they asked for it via not specifying fl_fast.  Since
 * *only* the bourne and korn shells require this, and *most* users
 * don't use plain old bourne/korn (bash does *not* require explicitly
 * running .profile), only do this if specifically requested, 'cause
 * we can ditch one of the 2 rsh's, and double the speed of things
 * like lamboot, recon, and wipe.
 */
	/* Build the username/hostname to pass to the error routine */
	
	memset(remote_host, 0, 512);
	if (username && *username)
	  snprintf(remote_host, 512, "%s@%s", username, host);
	else
	  strncat(remote_host, host, 512);
	remote_host[511] = '\0';

	if (!fl_fast) {
	  cmdc = 0;
	  cmdv = 0;
	  
	  add_rsh(&cmdc, &cmdv);
	  argvadd(&cmdc, &cmdv, host);
	  argvadd(&cmdc, &cmdv, "-n");
	  
	  if (username && *username) {
	    argvadd(&cmdc, &cmdv, "-l");
	    argvadd(&cmdc, &cmdv, username);
	  }
	  
	  argvadd(&cmdc, &cmdv, "echo $SHELL");
	  
	  /* Assemble a string for possible error usage -- argv gets
	     freed in ioexecvp */
	  
	  printable[0] = '\0';
	  for (i = 0; i < cmdc; i++) {
	    strncat(printable, cmdv[i], BUFSIZ);
	    strncat(printable, " ", BUFSIZ);
	  }
	  
	  if (prefix) {
	    fprintf(stderr, "%s: attempting to execute \"", prefix);
	    for (i = 0; i < cmdc; i++) {
	      if (i > 0)
		fprintf(stderr, " ");
	      fprintf(stderr, "%s", cmdv[i]);
	    }
	    fprintf(stderr, "\"\n");
	  }
	  
	  cmdv0 = strdup(cmdv[0]);
	  if (ioexecvp(cmdv, 0, shellpath, sizeof(shellpath))) {
	    if (errno == EFAULT)
	      show_help("boot", "remote-stderr", remote_host, cmdv0, 
			"echo $SHELL", printable, NULL);
	    else
	      show_help("boot", "remote-shell-fail", remote_host, cmdv0, 
			"echo $SHELL", printable, NULL);
	    
	    free(cmdv0);
	    return (LAMERROR);
	  }
	  
	  /*
	   * Did we get valid output?
	   */
	  if (strlen(shellpath) == 0) {
	    show_help("boot", "no-shell", remote_host, cmdv0, "echo $SHELL", 
		      printable, NULL);
	    free(cmdv0);
	    return (LAMERROR);
	  }
	  free(cmdv0);
	  
	  /* 
	   * Perl chomp 
	   */
	  if (shellpath[strlen(shellpath) - 1] == '\n')
	    shellpath[strlen(shellpath) - 1] = '\0';
	  if (prefix) {
	    fprintf(stderr, "%s: got remote shell %s\n", prefix, shellpath);
	  }
	  fl_csh = (strstr(shellpath, "csh") != 0) ? TRUE : FALSE;
	  fl_bash = (strstr(shellpath, "bash") != 0) ? TRUE : FALSE;
	} 

	/* If we didn't ask for the bourne shell, look up the user's
           local shell and assume they have the same shell on the
           remote hoted; fill in values for fl_csh and fl_bash */

	else {
	  p = getpwuid(getuid());
	  if (p == NULL)
	    return LAMERROR;
	  if (prefix) {
	    fprintf(stderr, 
		    "%s: -b used, assuming same shell on remote nodes\n", 
		    prefix);
	    fprintf(stderr, "%s: got local shell %s\n",
		    prefix, p->pw_shell);
	  }

	  fl_csh = (strstr(p->pw_shell, "csh") != 0) ? TRUE : FALSE;
	  fl_bash = (strstr(p->pw_shell, "bash") != 0) ? TRUE : FALSE;
	}

/*
 * Remotely execute the command using "rsh".
 */
	cmdc = 0;
	cmdv = 0;

	add_rsh(&cmdc, &cmdv);
	argvadd(&cmdc, &cmdv, host);
	argvadd(&cmdc, &cmdv, "-n");

	if (username && *username) {
	  argvadd(&cmdc, &cmdv, "-l");
	  argvadd(&cmdc, &cmdv, username);
	}

#if LAM_RSH_NEED_MINUSMINUS
/*
 * Stop rsh from interpreting options for the remote command as
 * rsh options.
 */
        argvadd(&cmdc, &cmdv, "--");
#endif

/*
 * If the user's shell is not based on "csh" or "bash", force the
 * interpretation of the user's .profile script in order to initialize
 * the paths. This works for "sh" and "ksh".
 */
	if (!(fl_csh || fl_bash)) 
	  argvadd(&cmdc, &cmdv, "(. ./.profile;");
	for (i = 0; argv[i]; ++i)
	  argvadd(&cmdc, &cmdv, argv[i]);
	if (!(fl_csh || fl_bash)) 
	  argvadd(&cmdc, &cmdv, ")");

	if (prefix) {
	  fprintf(stderr, "%s: attempting to execute \"", prefix);
	  for (i = 0; i < cmdc; i++) {
	    if (i > 0)
	      fprintf(stderr, " ");
	    fprintf(stderr, "%s", cmdv[i]);
	  }
	  fprintf(stderr, "\"\n");
	}

	/* Build the sample command to pass to the error routine.
           Have to do this ahead of time, because ioexecvp frees the
           argv array. */
	
	printable[0] = '\0';
	for (i = 0; i < cmdc; i++) {
	  strncat(printable, cmdv[i], BUFSIZ);
	  strncat(printable, " ", BUFSIZ);
	}

	cmdv0 = strdup(cmdv[0]);
	i = ioexecvp(cmdv, 1, (char *) 0, 0);

	/* Do we need to print an error message? */

	if (i) {

	  if (errno == EFAULT)
	    show_help("boot", "remote-stderr", remote_host, cmdv0,
		      argv[0], printable, NULL);
	  else
	    show_help("boot", "remote-boot-fail", remote_host, cmdv0,
		      argv[0], printable, NULL);

	  free(cmdv0);
	  return(LAMERROR);
	}

	free(cmdv0);
	return 0;
}

/*
 *	ioexecvp
 *
 *	Function:	- execute command (similar to cnfexec)
 *			- can direct command stdout to buffer and/or stdout
 *			- stderr is checked and passed through
 *	Accepts		- command argv
 *			- print stdout flag
 *			- ptr to buffer (for stdout data)
 *			- size of buffer
 *	Returns		- 0 or LAMERROR
 */
static int
ioexecvp(char **cmdv, int showout, char *outbuff, int outbuffsize)
{
	int		kidstdout[2];		/* child stdout pipe */
	int		kidstderr[2];		/* child stderr pipe */
	int		ret;			/* read() return value */
	int		err;			/* error indicator */
	int		status;			/* exit status */
	int		pid;			/* child process id */
	char		*ptr = 0;		/* buffer pointer */
	fd_set          readset;                /* fd's for read select */
	fd_set          errset;                 /* fd's for error select */
	int             nfds = 1;               /* num fd's in readset */
	char            temp[256];              /* string holding space */
	int             want_out = 0;           /* want stdout in select */
	int             stdout_err = 0;
	int             stderr_err = 0;
	int             i;
/*
 * Create child stdout/stderr pipes and fork the child process (command).
 */
	if (pipe(kidstdout) || pipe(kidstderr)) return(LAMERROR);

	if ((pid = fork()) < 0) {
		return(LAMERROR);
	}

	else if (pid == 0) {				/* child */

		if ((dup2(kidstderr[1], 2) < 0) ||
				(dup2(kidstdout[1], 1) < 0)) {
			perror(cmdv[0]);
			exit(errno);
		}

		if (close(kidstdout[0]) || close(kidstderr[0]) ||
				close(kidstdout[1]) || close(kidstderr[1])) {
			perror(cmdv[0]);
			exit(errno);
		}

/*
 * Ensure that we close all other file descriptors
 */
		for (i = 3; i < FD_SETSIZE; i++)
		  close(i);

		execvp(cmdv[0], cmdv);
		exit(errno);
	}

	if (close(kidstdout[1]) || close(kidstderr[1])) return(LAMERROR);

	argvfree(cmdv);
/* 
 * We must be able to monitor both stdout and stderr; it is possible
 * that we may be trying to capture the stdout but also need to
 * monitor output on stderr (e.g., recon, lamboot).  So make a FD_SET
 * with potentially both of the file descriptors and do a select on
 * it.
 */
	FD_ZERO(&readset);
	FD_SET(kidstderr[0], &readset);
	nfds = kidstderr[0] + 1;
	if (showout || (outbuff != 0)) {
	  ptr = outbuff;
	  FD_SET(kidstdout[0], &readset);
	  nfds = (nfds > kidstdout[0] + 1) ? nfds : kidstdout[0] + 1;
	  want_out = 1;
	}

	err = 0;
	while (err == 0 && nfds > 0) {
	  /*
	   * Check to see if select() gets interrupted.
	   */
	  errset = readset;
	  ret = select(nfds, &readset, NULL, &errset, NULL);
          if (ret == -1) {
	    if (errno == EINTR)
	      continue;
	    else {
	      /* 
	       * Need to simply break on error instead of returning so
               * that we can still reap the child properly 
	       */
	      err = LAMERROR;
	      break;
	    }
	  }

	  /*
	   * Check for error condition on stderr.  Don't need to close
	   * it here -- it will get closed unconditionally later.
	   */
	  if (FD_ISSET(kidstderr[0], &errset) != 0) {
	    stderr_err = 1;
	  }
	  /*
	   * See if there was something on stderr 
	   */
	  if (FD_ISSET(kidstderr[0], &readset) != 0) {
	    while (1) {
	      ret = read(kidstderr[0], temp, 256);
	      /* Error? */
              if (ret < 0) {
		if (errno == EINTR)
		  continue;
		else {
		  stderr_err = 1;
		  break;
		}
	      }
	      /* Good bytes */
	      else if (ret > 0) {
		write(2, temp, ret);
		fflush(stderr);
		
		errno = EFAULT;
		err = LAMERROR;
		break;
	      } 
	      /* Zero bytes */
	      else {
		/* This is likely to indicate that this pipe has closed */
		stderr_err = 1;
		break;
	      }
	    }
	  }

	  /*
	   * Check for error condition on stdout.  Don't need to close
	   * it here -- it will get closed unconditionally later.
	   */
	  if (FD_ISSET(kidstdout[0], &errset) != 0) {
	    stdout_err = 1;
	  }
	  /*
	   * See if there is something on stdout (and if we care)
	   */
	  if ((showout || (outbuff != 0)) && 
	      FD_ISSET(kidstdout[0], &readset) != 0) {
	    while (1) {
	      ret = read(kidstdout[0], temp, 256);
	      /* Error? */
              if (ret < 0) {
		if (errno == EINTR)
		  continue;
		else {
		  stdout_err = 1;
		  err = LAMERROR;
		  break;
		}
	      }
	      /* Good bytes */
	      else if (ret > 0) {
		if (outbuffsize > 0) {
		  memcpy(ptr, temp, (ret > outbuffsize) ? outbuffsize : ret);
		  /* Doesn't matter if we overshoot here */
		  outbuffsize -= ret;
		  ptr += ret;
		}
		if (showout) {
		  write(1, temp, ret);
		  fflush(stdout);
		}
	      } 
	      /* Zero bytes */
	      else {
		stdout_err = 1;
		break;
	      }
	    }
	  }

	  /* 
	   * Reset stderr, 'cause we're always interested in that,
	   * unless it errored out
	   */
	  nfds = 0;
	  if (!stderr_err) {
	    FD_SET(kidstderr[0], &readset);
	    nfds = kidstderr[0] + 1;
	  }

	  /*
	   * See if we want to reset stdout 
	   */
	  if (!stdout_err && (want_out || outbuffsize > 0)) {
	    FD_SET(kidstdout[0], &readset);
	    nfds = (nfds > kidstdout[0] + 1) ? nfds : kidstdout[0] + 1;
	  }
	}
/*
 * Close the pipes of the parent process.
 */
	if (close(kidstdout[0]) || close(kidstderr[0])) {
	  err = LAMERROR;
	}
/*
 * Wait for the command to exit.
 */
	do {
	  if (waitpid(pid, &status, 0) < 0) {
	    return(LAMERROR);
	  }
	} while (! WIFEXITED(status));
	
	if (WEXITSTATUS(status)) {
	  errno = WEXITSTATUS(status);
	  
	  if (errno == 1) {
	    errno = EUNKNOWN;
	  }
	  
	  return(LAMERROR);
	}
	
	return(err);
}


/*
 *	add_rsh
 *
 *	Function:	- add the "rsh" command to the argc/argv array
 *			- use value in LAMRSH environment variable (if exists)
 *			- or use default LAM_RSH string (from lam_config.h)
 *			- be sure to parse into individual words to add nicely
 *	Accepts		- ptr to length of array
 *			- prt to array
 */
static void
add_rsh(int *cmdc, char ***cmdv)
{
  char		*prsh_orig;		/* override default rsh cmd */
  char		*prsh;		        /* override default rsh cmd */
  
  prsh_orig = getenv("LAMRSH");
  if (prsh_orig == 0) prsh_orig = LAM_RSH;

  /*
   * Split into individual terms 
   */
  prsh = strdup(prsh_orig);
  *cmdv = sfh_argv_break(prsh, ' ');
  *cmdc = sfh_argv_count(*cmdv);
  free(prsh);
}
