/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres and Arun Rodrigues with Brian Barrett,
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 * 
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	GDB/RBD
 *
 *	$Id: wipe.c,v 6.25 2001/03/01 00:14:31 jsquyres Exp $
 * 
 *	Function:	- topology shutdown tool
 *			- kills Trollius on a network of OTB nodes
 *			- topology given using host file syntax
 *			- uses tkill
 *      Options:        - see man page and help menu
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>

#include <args.h>
#include <debug.h>
#include <lamnet.h>
#include <net.h>
#include <terror.h>
#include <laminternal.h>
#include <etc_misc.h>
#include <boot.h>
#include <sfh.h>

/*
 * static variables
 */
static int fl_verbose;	/* verbose mode */
static int fl_debug;	/* debug mode */

/*
 * static functions
 */
static char *findbhost(int cmdc, char **cmdv, char **env);


int
main(int argc, char *argv[], char *env[])
{
	FILE		*fp;		/* boot schema file pointer */
	char		*bhost;		/* boot schema filename */
	char		**cmdv;		/* cmd arguments */
	char		*batchid;	/* batch job id */
	int		cmdn;		/* # cmd arguments */
	int		global_ret;	/* global error status */
	int		i;
	int		fl_fast;        /* look for .profile */
	int		iorigin;	/* index of origin node */
	int		limit;		/* limit # nodes to wipe */
	int		nlamnet;	/* lamnet size */
	int		badhost;	/* bad host index */
	int		r, j, success = 1;
	struct lamnode	*lamnet;	/* network description array */

	/* Ensure that we are not root */

	if (getuid() == 0 || geteuid() == 0) {
	  show_help(NULL, "deny-root", NULL);
	  exit(EACCES);
	}

/*
 * Parse the command line.
 */
	validopts("bdhvHV");
	followed("n");

	if (do_args(&argc, argv)) {
	  show_help("wipe", "usage", NULL);
	  exit(EUSAGE);
	}

	if ((errno = (argc <= 2) ? 0 : EUSAGE)) {
	  show_help("wipe", "usage", NULL);
	  exit(errno);
	}

	if (opt_taken('h')) {
	  show_help("wipe", "usage", NULL);
	  exit(0);
	}

	fl_fast = opt_taken('b');
	fl_debug = opt_taken('d');
	fl_verbose = opt_taken('v') || fl_debug;

	if (opt_taken('V')) {
	  lam_show_version(1);
	  exit(0);
	} else {
		if (! opt_taken('H')) {
		  lam_show_version(0);
		}
	}
/* 
 * Locate the system file.
 */
	bhost = findbhost(argc, argv, env);
/*
 * Open the system file.
 */
	DBUG("wipe: opening hostfile %s\n", bhost);
	fp = fopen(bhost, "r");
	if (fp == 0) {
	  show_help("boot", "open-hostfile", "wipe", bhost, NULL);
	  exit(errno);
	}
/*
 * Parse the system file.
 */
	if (bhostparse(fp, &lamnet, &nlamnet)) {
	  show_help("boot", "bhostparse", "wipe", bhost, NULL);
	  exit(errno);
	}
	if (fl_debug) {
	  printf("wipe: found the following hosts:\n");
	  for (j = 0; j < nlamnet; j++)
	    printf("wipe:   n%d %s\n", j, lamnet[j].lnd_hname);
	}
/*
 * Close the system file.
 */
	if (fclose(fp)) {
	  show_help("boot", "close-hostfile", "wipe", bhost, NULL);
	  exit(errno);
	}
/*
 * Locate the host nodes.
 */
	if (lamnet_findhosts(lamnet, nlamnet, &badhost)) {

		if (errno == EBADHOST) {
		  show_help("boot", "resolv-hostname", "wipe",
			    lamnet[badhost].lnd_hname, bhost, NULL);
		  exit(LAM_EEXIT);
		} else {
		  char buffer[16];
		  sprintf(buffer, "%d", errno);
		  show_help("boot", "resolv-unknown", "wipe", bhost,
			    buffer, NULL);
		}
		exit(errno);
	}
/*
 * Find the origin node.
 */
	iorigin = lamnet_findorig(lamnet, nlamnet);

	if (iorigin >= 0) {
		lamnet[iorigin].lnd_type |= NT_ORIGIN | NT_ME;
	}

	DBUG("wipe: found %d host node(s)\n", nlamnet);
	if (iorigin >= 0) {
	  DBUG("wipe: origin node is %d (%s)\n", iorigin, 
	     lamnet[iorigin].lnd_hname);
	} else {
	  DBUG("wipe: killing LAM from a non-member machine\n");
	}
/*
 * Build the tkill command.
 */
	cmdn = 0;
	cmdv = 0;
	argvadd(&cmdn, &cmdv, DEFTTKILL);

	if (fl_debug) {
		argvadd(&cmdn, &cmdv, "-d");
	}

	if (opt_taken('n')) {
		intparam('n', &limit);
	} else {
		limit = -1;
	}
/*
 * If we're running ounder a batch system, we have to propogate the
 * socket name to all the remote tkill instances.
 */
	batchid = get_batchid();
	if (strlen(batchid) > 0) {
	  argvadd(&cmdn, &cmdv, "-b");
	  argvadd(&cmdn, &cmdv, batchid);
	}
/*
 * Loop over all host nodes.
 */
	global_ret = 0;

	for (i = 0; (i < nlamnet) && limit; ++i) {

		if (limit > 0) --limit;

		VERBOSE("Executing %s on n%d (%s)...\n", DEFTTKILL,
				lamnet[i].lnd_nodeid, lamnet[i].lnd_hname);

                if (fl_debug) {
		  printf("wipe: attempting to launch \"");
		  for (j = 0; j < cmdn; j++) {
		    if (j > 0)
		      printf(" ");
		    printf("%s", cmdv[j]);
		  }
		  printf("\" ");
		}

		if (lamnet[i].lnd_type & NT_ORIGIN) {
		        DBUG("(local execution)\n");
			r = _lam_few(cmdv);

			if (r) {
				errno = r;
			}
		} else {
		        DBUG("(remote execution)\n");
			r = inetexec(lamnet[i].lnd_hname,
				     lamnet[i].lnd_uname, cmdv, 
				     (fl_debug ? "wipe" : NULL),
				     fl_fast);
		}

		if (r) {
			fprintf(stderr, "wipe: %s failed on n%d (%s)\n",
					DEFTTKILL, lamnet[i].lnd_nodeid,
					lamnet[i].lnd_hname);

			if (errno != EUNKNOWN) {
				terror("wipe");
			} else
			  show_help(NULL, "unknown", NULL);

			global_ret = errno;
			success = 0;
		}
	}

	if (success) {
	  DBUG("wipe completed successfully\n");
	} else {
	  DBUG("wipe did NOT complete successfully\n");
	}

	argvfree(cmdv);
	return(global_ret);
}

/*
 *	findbhost
 *
 *	Function:	- locates boot schema file
 *	Accepts:	- argc of remaining command line
 *			- argv of remaining command line
 *	Returns:	- full pathname of boot schema file
 */
static char *
findbhost(int cmdc, char **cmdv, char **env)
{
	char		*bhost;
	char		*full;		/* temporary for full pathname */
	char		**pathv;
	int		pathc;
/*
 * Set the directories for the boot schema file search.
 */
	pathc = 0;
	pathv = 0;
	argvadd(&pathc, &pathv, "");
	argvadd(&pathc, &pathv, "$TROLLIUSHOME/etc");
	argvadd(&pathc, &pathv, "$LAMHOME/etc");
	argvadd(&pathc, &pathv, LAM_SYSCONFDIR);
/*
 * Set the boot schema file names.
 */
	if (cmdc == 2) {
		bhost = cmdv[1];
	} else if ((bhost = getenv("LAMBHOST"))) {
	} else if ((bhost = getenv("TROLLIUSBHOST"))) {
	} else {
		bhost = DEFFBHOST;
	}
/*
 * Search the directories.
 */
	full = sfh_path_findv(bhost, pathv, R_OK, env);

	if (full == 0) {
		fprintf(stderr, "wipe: cannot locate \"%s\"\n", bhost);
		exit(errno);
	}

	DBUG("wipe: boot schema file: %s\n", full);
	return(full);
}
