/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres and Arun Rodrigues with Brian Barrett,
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 * 
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	NJN
 *
 *	$Id: wwait.c,v 1.2 2001/02/03 20:51:50 arodrig6 Exp $
 *
 *	Function:	- one sided window wait
 *	Accepts:	- window
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <rpisys.h>

/*@
MPI_Win_wait - Completes an RMA exposure epoch

Input Parameter:
+ win -  window object (handle) 

Notes:

Completes an RMA exposure epoch started by a call to MPI_WIN_POST on
win. This call matches calls to MPI_WIN_COMPLETE(win) issued by each
of the origin processes that were granted access to the window during
this epoch. The call to MPI_WIN_WAIT will block until all matching
calls to MPI_WIN_COMPLETE have occurred. This guarantees that all
these origin processes have completed their RMA accesses to the local
window. When the call returns, all these RMA accesses will have
completed at the target window.

.N fortran
.N IMPI_MPI2

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_WIN
.N MPI_ERR_EPOCH

.seealso MPI_Put(3), MPI_Get(3), MPI_Win_create(3), MPI_Win_start(3), MPI_Win_complete(3), MPI_Win_fence(3), MPI_Win_free(3), MPI_Win_get_group(3), MPI_Win_get_group(3), MPI_Accumulate(3)

.N WEB
  @*/


int MPI_Win_wait(MPI_Win win)
{
	MPI_Comm	comm;
	int		err;			/* error code */

	lam_initerr_m();
	lam_setfunc_m(BLKMPIWINWAIT);
/*
 * Check arguments.
 */
	if (win == MPI_WIN_NULL) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPIWINWAIT, lam_mkerr(MPI_ERR_WIN, 0)));
	}

	comm = win->w_comm;

	if (!(win->w_flags & LAM_WFPOSTED)) {
		return(lam_errfunc(comm,
			BLKMPIWINWAIT, lam_mkerr(MPI_ERR_EPOCH, 0)));
	}
/*
 * Loop until everybody has completed.
 */
	_mpi_req_blkclr_m();
	_mpi_req_blkset_m(win->w_req);

	while (win->w_ncomplete > 0) {

		err = lam_osd_blkadv(win);
		if (err != MPI_SUCCESS) {
			return(lam_errfunc(comm, BLKMPIWINWAIT, err));
		}
	}
/*
 * Complete all remote one-sided requests in the window.
 */
	err = lam_osd_complete(win, LAM_RQFOSTARG);
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPIWINWAIT, err));
	}

	win->w_flags &= ~LAM_WFPOSTED;

        lam_resetfunc_m(BLKMPIWINWAIT);
	return(MPI_SUCCESS);
}
