/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres and Arun Rodrigues with Brian Barrett,
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 * 
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	NJN
 *
 *	$Id: tcreatehvector.c,v 1.3 2001/02/03 19:50:13 bbarrett Exp $
 *
 *	Function:	- create extended hvector derived datatype
 *	Accepts:	- # blocks
 *			- block length
 *			- block stride (in bytes)
 *			- element datatype
 *			- new datatype (out)
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <errno.h>
#include <stdlib.h>

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>

/*@
    MPI_Type_create_hvector - Create extended hvector derived datatype

Input Parameters:
+ count - Number of blocks (nonnegative integer)
. lengths - Number of elements in each block (nonnegative integer) 
. stride - Number of bytes between start of each block (integer) 
- oldtype - Old datatype (handle)

Output Parameter:
+ newtype - New datatype (handle) 

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_TYPE
.N MPI_ERR_COUNT
.N MPI_ERR_ARG
.N MPI_ERR_OTHER

.seealso MPI_Type_create_hindexed(3), MPI_Type_create_struct(3), MPI_Get_address(3)

.N WEB
@*/

int
MPI_Type_create_hvector(int count, int length, MPI_Aint stride,
			MPI_Datatype oldtype, MPI_Datatype *newtype)
{
	MPI_Datatype	ntype;			/* new datatype */

	lam_initerr_m();
	lam_setfunc_m(BLKMPITCREATEHVECTOR);
/*
 * Check the arguments.
 */
	if (oldtype == MPI_DATATYPE_NULL) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPITCREATEHVECTOR, lam_mkerr(MPI_ERR_TYPE, 0)));
	}

	if (count < 0) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPITCREATEHVECTOR, lam_mkerr(MPI_ERR_COUNT, 0)));
	}

	if ((length < 0) || (newtype == 0)) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPITCREATEHVECTOR, lam_mkerr(MPI_ERR_ARG, 0)));
	}
/*
 * Create the new datatype.
 */
	ntype = (MPI_Datatype) malloc(sizeof(struct _dtype));
	if (ntype == 0) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPITCREATEHVECTOR, lam_mkerr(MPI_ERR_OTHER, errno)));
	}

	oldtype->dt_refcount++;

	LAM_ZERO_ME(*ntype);
	lam_type_setdefaults(ntype);
	
	ntype->dt_format = LAM_DTHVECTORCREAT;
	ntype->dt_count = count;
	ntype->dt_length = length;
	ntype->dt_stride = stride;
	ntype->dt_dtype = oldtype;

	if (count > 0) {
		lam_dtblock(ntype, oldtype, length, 0);

		if (count > 1 && stride != ntype->dt_size) {
			ntype->dt_flags &= ~LAM_DTNOPACK;
		}

		ntype->dt_nelem *= count;
		ntype->dt_size *= count;

		stride *= count - 1;

		if (stride < 0) {
			ntype->dt_datalow += stride;
			ntype->dt_lower += stride;
		} else {
			ntype->dt_dataup += stride;
			ntype->dt_upper += stride;
		}

		lam_dtalign(ntype);
	}
	else {
		ntype->dt_flags = LAM_DTNOPACK | LAM_DTNOXADJ;
		ntype->dt_size = 0;
	}

	*newtype = ntype;

	lam_resetfunc_m(BLKMPITCREATEHVECTOR);
	return(MPI_SUCCESS);
}
