/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres and Arun Rodrigues with Brian Barrett,
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 * 
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	NJN
 *
 *	$Id: tcreatehindex.c,v 1.3 2001/02/03 19:50:13 bbarrett Exp $
 *
 *	Function:	- create extended hindexed derived datatype
 *	Accepts:	- # blocks
 *			- block length array
 *			- block displacement array (in bytes)
 *			- element datatype
 *			- new datatype (out)
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <errno.h>
#include <stdlib.h>

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>

/*@
    MPI_Type_create_hindexed - Create extended hindexed derived datatype

Input Parameters:
+ count - Number of blocks
. lengths - Number of elements in each block (array of nonnegative integers) 
. displacements - Byte displacement of each block (array of integer)
- oldtype - Old datatype (handle)

Output Parameter:
+ newtype - New datatype (handle) 

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_TYPE
.N MPI_ERR_COUNT
.N MPI_ERR_ARG
.N MPI_ERR_OTHER

.seealso MPI_Type_create_hvector(3), MPI_Type_create_struct(3), MPI_Get_address(3)

.N WEB
@*/

int
MPI_Type_create_hindexed(int count, int *lengths, MPI_Aint *disps,
			 MPI_Datatype oldtype, MPI_Datatype *newtype)
{
	MPI_Datatype	ntype;			/* new datatype */
	int		i;			/* favourite index */

	lam_initerr();
	lam_setfunc(BLKMPITCREATEHINDEX);
/*
 * Check the arguments.
 */
	if (oldtype == MPI_DATATYPE_NULL) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPITCREATEHINDEX, lam_mkerr(MPI_ERR_TYPE, 0)));
	}

	if (count < 0) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPITCREATEHINDEX, lam_mkerr(MPI_ERR_COUNT, 0)));
	}

	if ((newtype == 0) ||
			((count > 0) && ((lengths == 0) || (disps == 0)))) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPITCREATEHINDEX, lam_mkerr(MPI_ERR_ARG, 0)));
	}
/*
 * Create the new datatype.
 */
	ntype = (MPI_Datatype) malloc(sizeof(struct _dtype));
	if (ntype == 0) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPITCREATEHINDEX, lam_mkerr(MPI_ERR_OTHER, errno)));
	}

	LAM_ZERO_ME(*ntype);
	lam_type_setdefaults(ntype);

	if (count > 0) {
		ntype->dt_disps = (MPI_Aint *) malloc((unsigned) (count *
					(sizeof(MPI_Aint) + sizeof(int))));

		if (ntype->dt_disps == 0) {
			free((char *) ntype);
			return(lam_errfunc(MPI_COMM_WORLD, BLKMPITCREATEHINDEX,
					lam_mkerr(MPI_ERR_OTHER, errno)));
		}

		ntype->dt_lengths = (int *) (((char *) ntype->dt_disps) +
						(count * sizeof(MPI_Aint)));
	}

	for (i = 0; i < count; ++i) {
		ntype->dt_disps[i] = disps[i];

		if ((ntype->dt_lengths[i] = lengths[i]) < 0) {
			free((char *) ntype->dt_disps);
			free((char *) ntype);
			return(lam_errfunc(MPI_COMM_WORLD, BLKMPITCREATEHINDEX,
						lam_mkerr(MPI_ERR_ARG, 0)));
		}
	}

	ntype->dt_format = LAM_DTHINDEXEDCREAT;
	ntype->dt_count = count;
	ntype->dt_dtype = oldtype;
	oldtype->dt_refcount++;

	if (count > 0) {
		for (i = 0; i < count; ++i) {
			lam_dtblock(ntype, oldtype, lengths[i], disps[i]);
		}

		if (ntype->dt_size > 0 && ntype->dt_lower != 0) {
			ntype->dt_flags &= ~LAM_DTNOPACK;
		}

		lam_dtalign(ntype);
	}
	else {
		ntype->dt_flags = LAM_DTNOPACK | LAM_DTNOXADJ;
		ntype->dt_size = 0;
	}

	*newtype = ntype;

	lam_resetfunc(BLKMPITCREATEHINDEX);
	return(MPI_SUCCESS);
}
