/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres and Arun Rodrigues with Brian Barrett,
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 * 
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	RBD
 *
 *	$Id: recvinit.c,v 6.7 2000/10/25 23:33:58 jsquyres Exp $
 *
 *	Function:	- create a receive request
 *	Accepts:	- buffer
 *			- count
 *			- datatype
 *			- source
 *			- tag
 *			- communicator
 *			- request (out)
 *	Returns: 	- MPI_SUCCESS or error code
 */

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <rpisys.h>
#if LAM_WANT_IMPI
#include <impi.h>
#endif

/*@
    MPI_Recv_init - Builds a handle for a receive

Input Parameters:
+ buf - initial address of receive buffer (choice) 
. count - number of elements received (integer) 
. dtype - type of each element (handle) 
. src - rank of source or 'MPI_ANY_SOURCE' (integer) 
. tag - message tag or 'MPI_ANY_TAG' (integer) 
- comm - communicator (handle) 

Output Parameter:
. req - communication request (handle) 

Notes:

The communication initialized by this function will not be started
until 'req' is given to 'MPI_Start' or 'MPI_Startall'.  

The communication is then not guaranteed to progress or complete until
'req' has been given to one of the test or wait functions ('MPI_Test',
'MPI_Testall', 'MPI_Testany', 'MPI_Testsome', 'MPI_Wait',
'MPI_Waitall', 'MPI_Waitany', 'MPI_Waitsome').

Once any of the test or wait functions indicate that the communication
has finished, the communication cycle can be started again with
'MPI_Start' or 'MPI_Startall'.

However, once the communication has finished and it is no longer going
to be used, 'req' should be freed with 'MPI_Request_free'.

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_COUNT
.N MPI_ERR_TYPE
.N MPI_ERR_RANK
.N MPI_ERR_TAG
.N MPI_ERR_COMM
.N MPI_ERR_EXHAUSTED

.seealso MPI_Recv, MPI_Start, MPI_Startall, MPI_Test, MPI_Testall, MPI_Testany, MPI_Testsome, MPI_Wait, MPI_Waitall, MPI_Waitany, MPI_Waitsome

.N ACK
@*/
int MPI_Recv_init(void *buf, int count, MPI_Datatype dtype, 
		  int src, int tag, MPI_Comm comm, 
		  MPI_Request *req)
{
	int		err;			/* error code */

	lam_initerr_m();
	lam_setfunc_m(BLKMPIRINIT);

	if ((tag < 0 && tag != MPI_ANY_TAG) || tag > LAM_MAXTAG) {
		return(lam_errfunc(comm, BLKMPIRINIT,
				lam_mkerr(MPI_ERR_TAG, 0)));
	}

	*req = MPI_REQUEST_NULL;
	err = _mpi_req_build(buf, count, dtype, src, tag, comm,
				LAM_RQIRECV, req);

	if (err != MPI_SUCCESS) 
	  return(lam_errfunc(comm, BLKMPIRINIT, err));

	(*req)->rq_marks |= (LAM_RQFPERSIST | LAM_RQFMAND);
#if LAM_WANT_IMPI
	IMPI_Register_datasync_init(src, tag, comm, &((*req)->rq_shadow));
#endif

	lam_resetfunc_m(BLKMPIRINIT);
	return(MPI_SUCCESS);
}
