/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres and Arun Rodrigues with Brian Barrett,
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 * 
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	RBD
 *
 *	$Id: packsize.c,v 6.3 1999/12/31 21:27:34 jsquyres Exp $
 *
 *	Function:	- get size needed to pack data
 *	Accepts:	- count
 *			- datatype
 *			- communicator
 *			- ptr to size
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>

/*@

MPI_Pack_size - Returns the upper bound on the amount of space needed
to pack a message

Input Parameters:
+ count - count argument to packing call (integer) 
. dtype - datatype argument to packing call (handle) 
- comm - communicator argument to packing call (handle) 

Output Parameter:
. psize - upper bound on size of packed message, in bytes (integer) 

Notes: 

The MPI standard document describes this in terms of 'MPI_Pack', but
it applies to both the 'MPI_Pack' and 'MPI_Unpack' functions.  That
is, the value 'size' is the maximum that is needed by either
'MPI_Pack' or 'MPI_Unpack'.

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_COMM
.N MPI_ERR_COUNT
.N MPI_ERR_TYPE
.N MPI_ERR_ARG

.seealso MPI_Pack, MPI_Unpack

.N ACK
@*/
int MPI_Pack_size(int count, MPI_Datatype dtype, 
		  MPI_Comm comm, int *psize)
{
	lam_initerr();
	lam_setfunc(BLKMPIPACKSIZE);
/*
 * Check the arguments.
 */
	if (comm == MPI_COMM_NULL) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPIPACKSIZE, lam_mkerr(MPI_ERR_COMM, 0)));
	}

	if (count < 0) {
		return(lam_errfunc(comm,
			BLKMPIPACKSIZE, lam_mkerr(MPI_ERR_COUNT, 0)));
	}

	if (dtype == MPI_DATATYPE_NULL) {
		return(lam_errfunc(comm,
			BLKMPIPACKSIZE, lam_mkerr(MPI_ERR_TYPE, 0)));
	}

	if (psize == 0) {
		return(lam_errfunc(comm,
			BLKMPIPACKSIZE, lam_mkerr(MPI_ERR_ARG, 0)));
	}
/*
 * Return the size.
 */
	*psize = count * dtype->dt_size;

	lam_resetfunc(BLKMPIPACKSIZE);
	return(MPI_SUCCESS);
}
