/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres and Arun Rodrigues with Brian Barrett,
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 * 
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	RBD/NJN
 *
 *	$Id: mpietc.c,v 6.6 2000/10/25 23:33:18 jsquyres Exp $
 *
 *	Function:	- misc. shared MPI-related functions
 *			- used by libmpi and MPI tools
 */

#include <app_mgmt.h>
#include <mpi.h>
#include <mpisys.h>
#include <rpisys.h>

/*
 * macros
 */
#define FLIPCID(x)	(-((x) + 1))

/*
 *	lam_pt2coll
 *
 *	Function:	- change context ID from pt2pt to collective
 *	Accepts:	- pt2pt context ID
 *	Returns:	- collective context ID
 */
int
lam_pt2coll(int cid)
{
  if (cid >= 0) 
    cid = FLIPCID(cid);
  return(cid);
}

/*
 *	lam_coll2pt
 *
 *	Function:	- change context ID from collective to pt2pt
 *	Accepts:	- collective context ID
 *	Returns:	- pt2pt context ID
 */
int
lam_coll2pt(int cid)
{
  if (cid < 0) 
    cid = FLIPCID(cid);
  return(cid);
}

#if LAM_WANT_IMPI
/*
 *	lam_pt2impicoll
 *
 *	Function:	- change context ID from pt2pt to impi collective
 *	Accepts:	- pt2pt context ID
 *	Returns:	- impi collective context ID
 */
int
lam_pt2impicoll(int cid)
{
  /* Per IMPI section 4.3, the context ID for IMPI collectives is the
      pt2pt ID + 1 */

  if (cid >= 0 && cid % 3 == 0)
    cid++;
  return(cid);
}

/*
 *	lam_impicoll2pt
 *
 *	Function:	- change context ID from impi collective to pt2pt
 *	Accepts:	- impi collective context ID
 *	Returns:	- pt2pt context ID
 */
int
lam_impicoll2pt(int cid)
{
  /* Per IMPI section 4.3, the context ID for IMPI collectives is the
      pt2pt ID + 1 */

  if (cid > 0 && cid % 3 == 1)
    cid--;
  return(cid);
}


/*
 *	lam_pt2impidatasync
 *
 *	Function:	- change context ID from pt2pt to impi datasync
 *	Accepts:	- pt2pt context ID
 *	Returns:	- impi datasync context ID
 */
int
lam_pt2impidatasync(int cid)
{
  if (cid >= 0 && cid % 3 == 0)
    cid += 2;

  return(cid);
}


/*
 *	lam_pt2impidatasync
 *
 *	Function:	- change context ID from pt2pt to impi syncack
 *	Accepts:	- pt2pt context ID
 *	Returns:	- impi syncack context ID
 */
int
lam_pt2impisyncack(int cid)
{
  if (cid >= 0 && cid % 3 == 0) {
    cid += 2;
    cid = FLIPCID(cid);
  }

  return(cid);
}


#endif

/*
 *	lam_mkcoll
 *
 *	Function:	- switch communicator to collective context ID
 *	Accepts:	- communicator
 */
void
lam_mkcoll(MPI_Comm comm)
{
  if (comm) 
    comm->c_contextid = lam_pt2coll(comm->c_contextid);
}

/*
 *	lam_mkpt
 *
 *	Function:	- switch communicator to pt2pt context ID
 *	Accepts:	- communicator
 */
void
lam_mkpt(MPI_Comm comm)
{
  /* To preserve naming schemes, we have to do a little extra work in
     the IMPI case -- figure out which collective context ID scheme
     was used, and then do the opposite of it to return it to pt2pt */

  if (comm) {
#if LAM_WANT_IMPI
    if (comm->c_shadow != NULL)
      comm->c_contextid = lam_impicoll2pt(comm->c_contextid);
    else
      comm->c_contextid = lam_coll2pt(comm->c_contextid);
#else
    comm->c_contextid = lam_coll2pt(comm->c_contextid);
#endif
  }
}

#if LAM_WANT_IMPI
/*
 *	lam_mkimpicoll
 *
 *	Function:	- switch communicator to impi collective context ID
 *	Accepts:	- communicator
 */
void
lam_mkimpicoll(MPI_Comm comm)
{
  if (comm) 
    comm->c_contextid = lam_pt2impicoll(comm->c_contextid);
}
#endif

/*
 *	lam_unsyncsql
 *
 *	Function:	- extract SQL sync. information
 *	Accepts:	- event
 *			- type
 *			- tag (out)
 *			- context ID (out)
 *			- ACK flag (out)
 *			- destination index (out)
 *			- source node (out)
 *			- source index (out)
 */
void
lam_unsyncsql(int event, int type, int *ptag, int *pcid, int *pack,
	      int *pdidx, int *psnode, int *psidx)
{
  *pdidx = event & 0x7FFF;
  *pack = event & 0x10000;
  *pcid = (event >> 18) & 0xFFF;
  if (*pcid & 0x800) 
    *pcid |= 0xFFFFF000;
  
  *ptag = type & 0xFFFF;
  if (*ptag & 0x8000) 
    *ptag |= 0xFFFF0000;
  
  type >>= 16;
  *psnode = type & 0x3FF;
  *psidx = (type >> 10) & 0x3F;
  
  if (*psnode == 0x3FF) 
    *psnode = MPI_ANY_SOURCE;
  if (*psidx == 0x3F) 
    *psidx = MPI_ANY_SOURCE;
}

/*
 *	lam_rank_in_group
 *
 *	Function:	- find rank of process in a group
 *	Accepts:	- group
 *			- process
 *	Returns:	- rank in group else MPI_UNDEFINED
 */
int
lam_rank_in_group(MPI_Group group, struct _proc *proc)
{
  int		i;
  struct _gps	*gps;
  
  gps = &proc->p_gps;
  
  for (i = 0; i < group->g_nprocs; i++) {
    if (LAM_FULL_GPSCMP(&group->g_procs[i]->p_gps, gps) == 0) {
      return(i);
    }
  }
  
  return(MPI_UNDEFINED);
}
