/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres and Arun Rodrigues with Brian Barrett,
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 * 
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 * 
 * Additional copyrights may follow.
 * 
 *
 *	$Id: lamthreads.c,v 6.3 2001/03/01 02:38:52 jsquyres Exp $
 *
 *	Function:	- LAM interface for threads
 */

#include <lam_config.h>

#include <stdio.h>
#include <mpi.h>
#include <lamthreads.h>


/*
 * Exported varaibles
 */
int lam_thread_level = MPI_THREAD_SINGLE;
lam_thread_t lam_main_thread;
lam_mutex_t lam_mpi_mutex;
lam_mutex_t lam_updown_mutex;


/*
 *	lam_thread_init
 *
 *	Function:	- initialize the LAM thread interface
 */
void
lam_thread_init(int requested, int *pprovided)
{
#if LAM_HAVE_THREADS
/*
 * Set the provided level, and the global supported variable
 */
  if (requested < MPI_THREAD_MULTIPLE)
    lam_thread_level = *pprovided = requested;
  else
    lam_thread_level = *pprovided = MPI_THREAD_SERIALIZED;

/*
 * Initialize the global mutexes
 */
  lam_mutex_init(&lam_mpi_mutex);
  lam_mutex_init(&lam_updown_mutex);

#else /* LAM_HAVE_THREADS */
/*
 * If we don't have threads, set the return code to be MPI_THREAD_SINGLE
 */
  *pprovided = MPI_THREAD_SINGLE;

/*
 * Set the main thread variable -- even if we don't have threads, so
 * that MPI_IS_THREAD_MAIN will return something predicable, at least.
 */
  lam_main_thread = lam_thread_self();
#endif
}


/*
 *	lam_thread_self
 *
 *	Function:	- get thread specific identifier
 *      Returns:        - thread specific identifier
 */
lam_thread_t
lam_thread_self(void)
{
#if LAM_HAVE_SOL_THREADS
  return thr_self();
#elif LAM_HAVE_PTHREADS
  return pthread_self();
#else
  return 0;
#endif
}


/*
 *	lam_thread_compare
 *
 *	Function:	- compare two thread IDs
 *	Returns:	- 1 if they are the same, 0 otherwise
 *			- 1 if threading is not enabled (on assumption 
 *			  that there is only one thread)
 */
int
lam_thread_compare(lam_thread_t a, lam_thread_t b)
{
#if LAM_HAVE_SOL_THREADS
  return (a == b);
#elif LAM_HAVE_PTHREADS
  return !pthread_equal(a, b);
#else
  return 1;
#endif
}


/*
 *	lam_thread_finalize
 *
 *	Function:	- finalize the LAM thread interface
 */
void
lam_thread_finalize(void)
{
#if LAM_HAVE_THREADS
  if (lam_thread_level > MPI_THREAD_SINGLE) {
    lam_mutex_finalize(&lam_updown_mutex);
    lam_mutex_finalize(&lam_mpi_mutex);
  }
#endif
}
