/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres and Arun Rodrigues with Brian Barrett,
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 * 
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	RBD
 *
 *	$Id: graphnbr.c,v 6.3 1999/12/31 21:27:27 jsquyres Exp $
 *
 *	Function:	- get neighbours of a process
 *	Accepts:	- communicator
 *			- rank
 *			- size of neighbours array 
 *			- neighbours array (returned value)
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>

/*@

MPI_Graph_neighbors - Returns the neighbors of a node associated 
                      with a graph topology

Input Parameters:
+ comm - communicator with graph topology (handle) 
. rank - rank of process in group of comm (integer) 
- maxnbrs - size of array neighbors (integer) 

Output Parameters:
. nbrs - ranks of processes that are neighbors to specified process
(array of integer)

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_TOPOLOGY
.N MPI_ERR_COMM
.N MPI_ERR_ARG
.N MPI_ERR_RANK

.N ACK
@*/
int MPI_Graph_neighbors(MPI_Comm comm, int rank, 
			int maxnbrs, int *nbrs)
{
	int		nnbrs;
	int		i;
	int		*p;

	lam_initerr();
	lam_setfunc(BLKMPIGRAPHNBR);
/*
 * Check the arguments.
 */
	if (comm == MPI_COMM_NULL) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPIGRAPHNBR, lam_mkerr(MPI_ERR_COMM, 0)));
	}

	if (LAM_IS_INTER(comm)) {
		return(lam_errfunc(comm,
			BLKMPIGRAPHNBR, lam_mkerr(MPI_ERR_COMM, 0)));
	}

	if (!LAM_IS_GRAPH(comm)) {
		return(lam_errfunc(comm, BLKMPIGRAPHNBR,
				lam_mkerr(MPI_ERR_TOPOLOGY, 0)));
	}

	if ((rank < 0) || (rank >= comm->c_group->g_nprocs)) {
		return(lam_errfunc(comm,
			BLKMPIGRAPHNBR, lam_mkerr(MPI_ERR_RANK, 0)));
	}

	if ((maxnbrs < 0) || ((maxnbrs > 0) && (nbrs == 0))) {
		return(lam_errfunc(comm,
			BLKMPIGRAPHNBR, lam_mkerr(MPI_ERR_ARG, 0)));
	}
/*
 * Fill the neighbours.
 */
	nnbrs = comm->c_topo_index[rank];
	p = comm->c_topo_edges;

	if (rank > 0) {
		i = comm->c_topo_index[rank - 1];
		nnbrs -= i;
		p += i; 
	}

	for (i = 0; (i < maxnbrs) && (i < nnbrs); ++i, ++p) {
		*nbrs++ = *p;
	}

	lam_resetfunc(BLKMPIGRAPHNBR);
	return(MPI_SUCCESS);
}
