/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres and Arun Rodrigues with Brian Barrett,
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 * 
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 * 
 * Additional copyrights may follow.
 * 
 *
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	RBD
 *
 *	$Id: graphmap.c,v 6.3 1999/12/31 21:27:27 jsquyres Exp $
 *
 *	Function:	- map processes to graph coordinates
 *	Accepts:	- old communicator
 *			- # processes
 *			- node index array
 *			- array of edges
 *			- ptr caller's new rank
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>

/*@

MPI_Graph_map - Maps process to graph topology information

Input Parameters:
+ comm - input communicator (handle) 
. nnodes - number of graph nodes (integer) 
. index - integer array specifying the graph structure, see
'MPI_GRAPH_CREATE'
- edges - integer array specifying the graph structure 

Output Parameter:
. pnewrank - reordered rank of the calling process; 'MPI_UNDEFINED' if
the calling process does not belong to graph (integer)
 
.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_TOPOLOGY
.N MPI_ERR_COMM
.N MPI_ERR_ARG

.N ACK
@*/
int MPI_Graph_map(MPI_Comm comm, int nnodes, int *index, 
		  int *edges, int *pnewrank)
{
	int		errcode;
	int		myrank;

	lam_initerr();
	lam_setfunc(BLKMPIGRAPHMAP);
/*
 * Check the arguments.
 */
	if (comm == MPI_COMM_NULL) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPIGRAPHMAP, lam_mkerr(MPI_ERR_COMM, 0)));
	}

	if (LAM_IS_INTER(comm)) {
		return(lam_errfunc(comm,
			BLKMPIGRAPHMAP, lam_mkerr(MPI_ERR_COMM, 0)));
	}

	if ((nnodes < 1) || (index == 0) || (edges == 0) || (pnewrank == 0)) {
		return(lam_errfunc(comm,
			BLKMPIGRAPHMAP, lam_mkerr(MPI_ERR_ARG, 0)));
	}
/*
 * Assign new rank.
 */
	errcode = MPI_Comm_rank(comm, &myrank);
	if (errcode != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPIGRAPHMAP, errcode));
	}

	*pnewrank =
		((myrank < 0) || (myrank >= nnodes)) ? MPI_UNDEFINED : myrank;

	lam_resetfunc(BLKMPIGRAPHMAP);
	return(MPI_SUCCESS);
}
