/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres and Arun Rodrigues with Brian Barrett,
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 * 
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 * 
 * Additional copyrights may follow.
 * 
 *
 *	$Id: freemem.c,v 6.2 2000/08/10 17:36:00 jsquyres Exp $
 *
 *	Function:	- free memory allocated from MPI_Alloc_mem
 *	Accepts:	- ptr to memory to free
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>

/*@

MPI_Free_mem - Free memory that was previously allocated by MPI_ALLOC_MEM

Input Parameter:
. base - pointer to memory to free

Notes:

For the TCP, SYSV, and USYSV RPIs, this function is simply a wrapper
around 'free'.

However, future RPIs (e.g., for Myrinet and VIA-based RPIs) will
provide "special" memory deallocation routines that will deallocate
memory that was used directly for DMA (i.e., allocated by
'MPI_ALLOC_MEM').

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_ARG
.N MPI_ERR_NO_MEM

.N ACK
@*/
int MPI_Free_mem(void *base)
{
  lam_initerr();
  lam_setfunc(BLKMPIFREEMEM);
  
  if (base == 0) {
    return(lam_errfunc(MPI_COMM_WORLD,
		       BLKMPIALLOCMEM, lam_mkerr(MPI_ERR_BASE, 0)));
  }
  
  if (_rpi_c2c_free_mem(base) != MPI_SUCCESS) {
    return(lam_errfunc(MPI_COMM_WORLD,
		       BLKMPIALLOCMEM, lam_mkerr(MPI_ERR_BASE, 0)));
  }

  lam_resetfunc(BLKMPIFREEMEM);
  return(MPI_SUCCESS);
}
