/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres and Arun Rodrigues with Brian Barrett,
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 * 
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	RBD/GDB
 *
 *	$Id: cancel.c,v 6.4 1999/12/31 21:27:17 jsquyres Exp $
 *
 *	Function:	- marks immediate request for cancellation
 *	Accepts:	- ptr request
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <rpisys.h>

/*@
    MPI_Cancel - Cancels a communication request

Input Parameter:
. preq - communication request (handle) 

Notes:

Cancel has only been implemented for receive requests; it is a no-op
for send requests.  The primary expected use of MPI_Cancel is in
multi-buffering schemes, where speculative 'MPI_Irecv' calls are made.
When the computation completes, some of these receive requests may
remain; using 'MPI_Cancel' allows the user to cancel these unsatisfied
requests.

Cancelling a send operation is `much` more difficult, in large part
because the send will usually be at least partially complete (the
information on the tag, size, and source are usually sent immediately
to the destination).  As such, LAM/MPI has not implemented the
canceling of send operations.

.N NULL

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_REQUEST
.N MPI_ERR_ARG

.N ACK
@*/
int MPI_Cancel(MPI_Request *preq)
{
	MPI_Request	req;

	lam_initerr();
	lam_setfunc(BLKMPICANCEL);
/*
 * Check the arguments.
 */
	if (preq == 0) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPICANCEL, lam_mkerr(MPI_ERR_ARG, 0)));
	}

	if ((req = *preq) == MPI_REQUEST_NULL) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPICANCEL, lam_mkerr(MPI_ERR_ARG, 0)));
	}

	if (req->rq_state == LAM_RQSINIT) {
		return(lam_errfunc(req->rq_comm,
			BLKMPICANCEL, lam_mkerr(MPI_ERR_REQUEST, 0)));
	}
/*
 * In the start state, nothing has happened, so cancel is easy.
 * Move the request to the done state.
 */
	if (req->rq_state == LAM_RQSSTART) {
		req->rq_flags |= LAM_RQFCANCEL;
		req->rq_status.MPI_SOURCE = MPI_CANCEL_SOURCE;
		req->rq_state = LAM_RQSDONE;
		lam_rq_nactv--;
	}
/*
 * In the active state, it may still be possible to cancel a send operation.
 * This is an immense pain for many C2C implementations and the same
 * resource cleanup goal can be achieved by having the receiver suck
 * in the message.  Cancelling an active send is no problem for lamd.
 * We just make a remote service call to bufferd and ask that the MPI
 * sequence number from this source be nuked.  We set the cancel flag
 * based on the return code.  I could add this in half an hour.  Screw it.
 */
	else if ((req->rq_state == LAM_RQSACTIVE) &&
			(req->rq_marks & LAM_RQFDEST)) {
	}
/*
 * In the done state, the cancel fails.
 */
	else {
	}

	lam_resetfunc(BLKMPICANCEL);
	return(MPI_SUCCESS);
}
