/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres and Arun Rodrigues with Brian Barrett,
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 * 
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	RBD
 *
 *	$Id: bufattach.c,v 6.3 1999/12/31 21:27:17 jsquyres Exp $
 *
 *	Function:	- attach buffer to process
 *	Accepts:	- buffer address
 *			- buffer size
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>

/*@
  MPI_Buffer_attach - Attaches a user-defined buffer for sending

Input Parameters:
+ buf - initial buffer address (choice) 
- size - buffer size, in bytes (integer) 

Notes:

The size given should be the sum of the sizes of all outstanding
Bsends that you intend to have, plus a few hundred bytes for each
Bsend that you do.  For the purposes of calculating size, you should
use 'MPI_Pack_size'.  In other words, in the code

.vb
     MPI_Buffer_attach( buffer, size );
     MPI_Bsend( ..., count=20, datatype=type1,  ... );
     ...
     MPI_Bsend( ..., count=40, datatype=type2, ... );
.ve

the value of 'size' in the 'MPI_Buffer_attach' call should be greater
than the value computed by

.vb
     MPI_Pack_size(20, type1, comm, &s1);
     MPI_Pack_size(40, type2, comm, &s2);
     size = s1 + s2 + 2 * MPI_BSEND_OVERHEAD;
.ve    

'MPI_Bsend_overhead' gives the maximum amount of space that may be
used in the buffer for use by the 'Bsend' routines in using the
buffer.  This value is in 'mpi.h' (for C) and 'mpif.h' (for Fortran).

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_BUFFER
.N MPI_ERR_INTERN

.seealso: MPI_Buffer_detach, MPI_Bsend

.N ACK
@*/
int MPI_Buffer_attach(void *buf, int size)
{
	int		err;			/* error code */

	lam_initerr();
	lam_setfunc(BLKMPIBUFATTACH);

	err = lam_bufattach(buf, size);
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPIBUFATTACH, err));
	}

	lam_resetfunc(BLKMPIBUFATTACH);
	return(MPI_SUCCESS);
}
