/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres and Arun Rodrigues with Brian Barrett,
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 * 
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	JRV/RBD
 *
 *	$Id: allgather.c,v 6.8 2001/02/28 00:15:05 jsquyres Exp $
 *
 *	Function:	- gather buffers at all process ranks
 *	Accepts:	- send buffer
 *			- send count
 *			- send datatype
 *			- recv buffer
 *			- recv count
 *			- recv datatype
 *			- communicator
 *	Returns:	- MPI_SUCCESS or an MPI error code
 */

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <lam_config.h>

/*@

MPI_Allgather - Gathers data from all tasks and distribute it to all 

Input Parameters:
+ sbuf - starting address of send buffer (choice) 
. count - number of elements in send buffer (integer) 
. sdtype - data type of send buffer elements (handle) 
. rbuf - number of elements received from any process (integer) 
. rcount - data type of receive buffer elements (handle) 
- comm - communicator (handle) 

Output Parameter:
. rbuf - address of receive buffer (choice) 

Notes:
 The MPI standard (1.0 and 1.1) says 

 "The jth block of data sent from each proess is received by every
 process and placed in the jth block of the buffer recvbuf."

 This is misleading; a better description is

 "The block of data sent from the jth process is received by every
 process and placed in the jth block of the buffer recvbuf."

 This text was suggested by Rajeev Thakur.

.N IMPI

.N fortran

.N Errors
.N MPI_ERR_COMM
.N MPI_ERR_INTERCOMM
.N MPI_ERR_COUNT
.N MPI_ERR_TYPE
.N MPI_ERR_BUFFER

.N ACK
@*/
int MPI_Allgather(void *sbuf, int scount, MPI_Datatype sdtype, 
		  void *rbuf, int rcount, MPI_Datatype rdtype, 
		  MPI_Comm comm)
{
	int		err;			/* error code */
	int		size;			/* group size */

	lam_initerr();
	lam_setfunc(BLKMPIALLGATHER);
/*
 * Check for invalid arguments.
 */
	if ((comm == MPI_COMM_NULL) || LAM_IS_INTER(comm)) {
		return(lam_errfunc(comm, BLKMPIALLGATHER,
				lam_mkerr(MPI_ERR_COMM, 0)));
	}

	if ((sdtype == MPI_DATATYPE_NULL) || (rdtype == MPI_DATATYPE_NULL)) {
		return(lam_errfunc(comm, BLKMPIALLGATHER,
				lam_mkerr(MPI_ERR_TYPE, 0)));
	}

	if ((scount < 0) || (rcount < 0)) {
		return(lam_errfunc(comm, BLKMPIALLGATHER,
				lam_mkerr(MPI_ERR_COUNT, 0)));
	}

	LAM_TRACE(lam_tr_cffstart(BLKMPIALLGATHER));
/*
 * Gather and broadcast.
 */
	MPI_Comm_size(comm, &size);

	err = MPI_Gather(sbuf, scount, sdtype, rbuf, rcount, rdtype, 0, comm);
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPIALLGATHER, err));
	}

	err = MPI_Bcast(rbuf, rcount * size, rdtype, 0, comm);
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPIALLGATHER, err));
	}

	LAM_TRACE(lam_tr_cffend(BLKMPIALLGATHER, -1, comm, sdtype, scount));

	lam_resetfunc(BLKMPIALLGATHER);
	return(MPI_SUCCESS);
}
