/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres and Arun Rodrigues with Brian Barrett,
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 * 
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	GDB
 *
 *	$Id: ktrace.c,v 6.3 1999/05/25 17:50:38 kmeyer1 Exp $
 * 
 *	Function:	- runtime trace package
 *			- an internal package, hence does not check or
 *			  return many error conditions
 */

#include <stdlib.h>
#include <unistd.h>

#include <kio.h>
#include <ktrace.h>
#include <net.h>
#include <mpitrace.h>
#include <portable.h>
#include <trreq.h>

/*
 * external variables
 */
extern struct kio_t	_kio;			/* Kernel Input/Output block */
extern int		errno;			/* global error variable */

/*
 * local functions
 */
static void		tralloc();		/* allocate trace buffers */

/*
 * global functions
 */
void			lam_trexit();		/* trace cleanup */
void			lam_trinit();		/* initialize traces */
void			lam_trunload();		/* unload traces to traced */

/*
 *	lam_trinit
 *
 *	Function:	- initializes trace system
 */
void
lam_trinit()

{
	int		i;			/* favourite index */
/*
 * Initialize trace buffer header array.
 * Its size limits the total size of trace data per process
 * since each individual trace buffer is MAXNMSGLEN long.
 */
	_kio.ki_tid = getpid();
	_kio.ki_trcurr = 0;
	_kio.ki_trhead = (struct trhead *) malloc((unsigned)
		(TRHMAX * sizeof(struct trhead)));

	if (! _kio.ki_trhead) {
		kexit(errno);
	}

	for (i = 0; i < TRHMAX; ++i) {
		_kio.ki_trhead[i].trh_length = 0;
		_kio.ki_trhead[i].trh_buf = 0;
	}
/*
 * Vector SIGTRACE to unload trace data to the local daemon.
 * Register a cleanup function.
 */
	ksignal(SIGTRACE, lam_trunload);
	atkexit(lam_trexit, (char *) 0);
/*
 * Allocate the first few trace buffers.
 */
	tralloc(0);
}

/*
 *	tralloc
 *
 *	Functions:	- allocates memory for the next few trace buffers
 *			- next TRHQUAN buffers are allocated
 *			- assumes TRHQUAN evenly divides TRHMAX
 *	Accepts:	- index of first trace buffer ptr
 */
static void
tralloc(idx)

int			idx;

{
	int		i;
	int		imax;
	char		*p;

	p = malloc((unsigned) (MAXNMSGLEN * TRHQUAN));
	if (! p) kexit(errno);

	imax = idx + TRHQUAN;

	for (i = idx; i < imax; ++i) {
		_kio.ki_trhead[i].trh_buf = p;
		p += MAXNMSGLEN;
	}
}

/*
 *	lam_trunload
 *
 *	Function:	- unloads used trace buffers to local trace daemon
 */
void
lam_trunload()

{
	int		i;

	for (i = 0; i < TRHMAX; ++i) {

		if (_kio.ki_trhead[i].trh_length > 0) {

			if (lam_rtrstore(LOCAL, TRRUNTIME, _kio.ki_tid,
					_kio.ki_trhead[i].trh_buf,
					_kio.ki_trhead[i].trh_length) < 0) {
				kexit(errno);
			}

			_kio.ki_trhead[i].trh_length = 0;
		}
	}

	_kio.ki_trcurr = 0;
}

/*
 *	lam_trpoint
 *
 *	Function:	- determines where the next trace can be stored
 *			- calls tralloc, if necessary
 *			- may permanently turn off tracing if all buffers fill
 *	Accepts:	- size of next trace
 *	Returns:	- ptr to next trace location
 */
char *
lam_trpoint(size)

int			size;

{
	struct trhead	*p;

	p = _kio.ki_trhead + _kio.ki_trcurr;
/*
 * If the current trace buffer is filled, move to the next one.
 */
	if ((p->trh_length + size) > MAXNMSGLEN) {
/*
 * Flush the trace buffers if we've used up the last one.
 */
		if (_kio.ki_trcurr == (TRHMAX - 1)) {
			lam_trunload();
			p = _kio.ki_trhead + _kio.ki_trcurr;
		} else {
			_kio.ki_trcurr += 1;
			p++;
/*
 * If the next trace buffer is unallocated, allocate it plus a few more.
 */
			if (p->trh_buf == 0) {
				tralloc(_kio.ki_trcurr);
			}
		}
	}
/*
 * Increment the trace buffer length before passing the address to be stored.
 */
	p->trh_length += size;
	return(p->trh_buf + (p->trh_length - size));
}

/*
 *	lam_trexit
 *
 *	Function:	- cleanup trace system prior to process exit
 */
void
lam_trexit()

{
	int		i;
/*
 * Unload all traces to the local daemon.
 */
	lam_trunload();
/*
 * Free all trace buffers.
 */
	for (i = 0; i < TRHMAX; i += TRHQUAN) {

		if (_kio.ki_trhead[i].trh_buf != 0) {
			free(_kio.ki_trhead[i].trh_buf);
		}
	}
/*
 * Free the trace header array.
 */
	free((char *) _kio.ki_trhead);
}
