/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres and Arun Rodrigues with Brian Barrett,
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 * 
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 * 
 * Additional copyrights may follow.
 * 
 *
 *	$Id: lamlog.c,v 6.2 2001/03/29 16:48:53 jsquyres Exp $
 * 
 *	Function:	- thin wrappers around syslog functionality
 */

#include <lam_config.h>

#include <stdio.h>
#if __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif
#include <stdlib.h>
#include <syslog.h>
#include <string.h>

#include <lamlog.h>

/*
 * Local state variable
 */

static int fl_syslog = 0;


/*
 * lamopenlog
 *
 * Open the syslog and set fl_syslog to 1 so that future calls to
 * lamlog will go to the syslog.
 */
void
lamopenlog(char *ident)
{
  openlog(ident, LOG_PID, LOG_INFO);
  fl_syslog = 1;
}


/*
 * lamlog
 *
 * Send something to the syslog if fl_syslog is 1
 */
void
#if LAM_WANT_PROTOS
lamlog(char *format, ...)
#else
lamlog(format, va_alist)
char *format;
va_dcl
#endif
{
  va_list arglist;
  int i, len = 0;
  char *sarg;
  int iarg;
  long larg;
  double darg;
  float farg;

  if (fl_syslog) {
#if __STDC__
    va_start(arglist, format);
#else
    va_start(arglist);
#endif

    /* First, construct how long of a string we'll need for all the args */

    len = strlen(format);
    for (i = 0; i < strlen(format); ++i) {
      if (format[i] == '%' && i + 1 < strlen(format) && format[i + 1] != '%') {
	++i;
	switch(format[i]) {
	case 's':
	  sarg = va_arg(arglist, char*);
	  /* If there's an arg, get the strlen, otherwise we'll use (null) */
	  if (sarg != NULL)
	    len += strlen(sarg);
	  else
	    len += 5;
	  break;

	case 'd':
	case 'i':
	  iarg = va_arg(arglist, int);
	  /* Alloc for minus sign */
	  if (iarg < 0)
	    ++len;
	  /* Now get the log10 */
	  do {
	    ++len;
	    iarg /= 10;
	  } while (iarg != 0);
	  break;

	case 'x':
	case 'X':
	  iarg = va_arg(arglist, int);
	  /* Now get the log16 */
	  do {
	    ++len;
	    iarg /= 16;
	  } while (iarg != 0);
	  break;

	case 'f':
	  farg = va_arg(arglist, int);
	  /* Alloc for minus sign */
	  if (farg < 0) {
	    ++len;
	    farg = -farg;
	  }
	  /* Alloc for 3 decimal places + '.' */
	  len += 4;
	  /* Now get the log10 */
	  do {
	    ++len;
	    farg /= 10.0;
	  } while (farg != 0);
	  break;

	case 'g':
	  darg = va_arg(arglist, int);
	  /* Alloc for minus sign */
	  if (darg < 0) {
	    ++len;
	    darg = -darg;
	  }
	  /* Alloc for 3 decimal places + '.' */
	  len += 4;
	  /* Now get the log10 */
	  do {
	    ++len;
	    darg /= 10.0;
	  } while (darg != 0);
	  break;

	case 'l':
	  /* Get %ld %lx %lX %lf */
	  if (i + 1 < strlen(format)) {
	    ++i;
	    switch(format[i]) {
	    case 'x':
	    case 'X':
	      larg = va_arg(arglist, int);
	      /* Now get the log16 */
	      do {
		++len;
		larg /= 16;
	      } while (larg != 0);
	      break;

	    case 'f':
	      darg = va_arg(arglist, int);
	      /* Alloc for minus sign */
	      if (darg < 0) {
		++len;
		darg = -darg;
	      }
	      /* Alloc for 3 decimal places + '.' */
	      len += 4;
	      /* Now get the log10 */
	      do {
		++len;
		darg /= 10.0;
	      } while (darg != 0);
	      break;

	    case 'd':
	    default:
	      larg = va_arg(arglist, int);
	      /* Now get the log10 */
	      do {
		++len;
		larg /= 10;
	      } while (larg != 0);
	      break;
	    }
	  }

	default:
	  break;
	}
      }
    }
    va_end(arglist);

    /* Now we have the length.  Alloc a buffer long enough.  Throw
       some extra dudging in there for % escape sequences that we
       didn't handle. Unfortunately, we can't count on vsnprintf being
       on all systems.  :-( */

    sarg = malloc(len + 128);
#if __STDC__
    va_start(arglist, format);
#else
    va_start(arglist);
#endif
    vsprintf(sarg, format, arglist);
    va_end(arglist);

    syslog(LOG_INFO, sarg);
    free(sarg);
  }
}


/*
 * lamcloselog
 *
 * Closes the syslog, if it was open
 */
void
lamcloselog()
{
  if (fl_syslog) {
    fl_syslog = 0;
    closelog();
  }
}
