/***************************************************************************
                          kxe_treeview.cpp  -  description
                             -------------------
    begin                : Thu Sep 20 2001
    copyright            : (C) 2001 by The KXMLEditor Team
    email                : OleBowle@gmx.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "kxe_treeview.h"
#include "kxe_treeviewitem.h"
#include "dlgsearch.h"
#include "dlgconfiguration.h"
#include "qdom_add.h"

#include <klocale.h>
#include <kdebug.h>
#include <kxmlgui.h>
#include <kxmlguiclient.h>
#include <kpopupmenu.h>

#include <qdom.h>
#include <qcursor.h>

KXE_TreeView::KXE_TreeView( KXMLGUIClient * pGUIClient, QWidget * pParent, const char * pszName )
	: KListView(pParent,pszName),
	  m_pGUIClient(pGUIClient),
	  m_nBookmarkedItems(0)
{
	setSorting(-1); // no sorting

	addColumn(i18n("Qualified name"));

	setSelectionMode(QListView::Single);

	connect( this, SIGNAL(selectionChanged(QListViewItem*)), this, SLOT(slotSelectionChanged(QListViewItem*)) );

	setReadWrite(false);
}

void KXE_TreeView::setReadWrite( bool fReadWrite )
{
	setItemsRenameable( fReadWrite );
	setRenameable( 0, fReadWrite );
}

//////////////////////////////////////////////////////////////
// configuration slots
//////////////////////////////////////////////////////////////

void KXE_TreeView::slotConfigurationChanged( const DlgConfiguration * const pNewConfiguration )
{
	setRootIsDecorated( pNewConfiguration->getTreeViewRootDecorated() );
	KXE_TreeViewItem::s_iDefaultExpandLevel = pNewConfiguration->getTreeViewExpandToLevel();
	
	// if the element display mode has been changed, the treeview has to be updated
	if ( KXE_TreeViewItem::s_iElemDisplayMode != pNewConfiguration->getTreeViewElemDisplayMode() )
	{
		KXE_TreeViewItem::s_iElemDisplayMode = pNewConfiguration->getTreeViewElemDisplayMode();

		if ( pNewConfiguration->getTreeViewElemDisplayMode() == 0 )
		{
			if ( columns() > 1 )
				removeColumn(1);
		}
		else if ( columns() < 2 )
			addColumn( i18n("Attributes") );

		KXE_TreeViewItem * pItem = static_cast<KXE_TreeViewItem*> (firstChild());
		while (pItem)
		{
			pItem->setTexts();
			pItem = pItem->nextItem();
		}
	}

	// setAcceptDrops(bEnable); TODO later, when read/write KPart
	// viewport()->setAcceptDrops(bEnable);
}

//////////////////////////////////////////////////////////////
// action slots
//////////////////////////////////////////////////////////////

void KXE_TreeView::slotEditDeselect()
{
	clearSelection();
}

void KXE_TreeView::slotViewNodeUp()
{
	// get selected item from tree view
	QListViewItem * pSelItem = selectedItem();
	if ( ! pSelItem )
	{
		kdDebug() << "KXE_TreeView::slotViewNodeUp no item selected" << endl;
		return;
	}

	// get parent item
	QListViewItem * pParentItem = pSelItem->parent();

	// select parent item in tree view
	if (pParentItem)
	{
		setCurrentItem(pParentItem);
		ensureItemVisible(pParentItem);
	}
}

void KXE_TreeView::slotViewExpNode( int nLevel )
{
	// get selected item from tree view (if any)
	QListViewItem * pSelItem = selectedItem();
	if ( ! pSelItem )
	{
		kdDebug() << "KXE_TreeView::slotViewExpNode no item selected" << endl;
		return;
	}

	// expand node
	KXE_TreeViewItem * pSelTreeItem = static_cast <KXE_TreeViewItem*> (pSelItem);
	pSelTreeItem->expandSubTree(nLevel);
}

void KXE_TreeView::slotViewColNode( int nLevel )
{
	// get selected item from tree view (if any)
	QListViewItem * pSelItem = selectedItem();
	if ( ! pSelItem )
	{
		kdDebug() << "KXE_TreeView::slotViewColNode no item selected" << endl;
		return;
	}

	// expand node
	KXE_TreeViewItem * pSelTreeItem = static_cast <KXE_TreeViewItem*> (pSelItem);
	pSelTreeItem->collapseSubTree(nLevel);
}

void KXE_TreeView::slotBookmarksToggle()
{
	// get selected item from tree view
	KXE_TreeViewItem * pSelItem = static_cast <KXE_TreeViewItem*> (selectedItem());
	if ( ! pSelItem )
	{
		kdDebug() << "KXE_TreeView::slotBookmarksToggle: no item selected" << endl;
		return;
	}

	// toggle bookmark on selected item
	if ( pSelItem->toggleBookmark() )
		m_nBookmarkedItems++;
	else
		m_nBookmarkedItems--;
}

void KXE_TreeView::slotBookmarksPrev()
{
	if ( childCount() < 1 )
	{
		kdDebug() << "KXE_TreeView::slotBookmarksPrev: internal error - this tree view is empty" << endl;
		return;
	}

	// get selected item from tree view
	KXE_TreeViewItem * pSelItem = static_cast <KXE_TreeViewItem*> (selectedItem());
	if ( ! pSelItem )                                     // If there is no item selected we take
	{                                                     // the last root items last grand child.
		QListViewItem * pTmpItem = firstChild(); // Take first child and
		while ( pTmpItem->nextSibling() )        // find last child by
			pTmpItem = pTmpItem->nextSibling();    // traversing all childs

		pSelItem = static_cast <KXE_TreeViewItem*> (pTmpItem); // this is the last root item
		while ( pSelItem->lastChild() )      // find its last
			pSelItem = pSelItem->lastChild();  // grand child

		if ( pSelItem->isBookmarked() )                     // We have to check its
		{                                                   // bookmarked-status
			selectItem(pSelItem);                             // and select it, in case
			return;                                           // it is bookmarked.
		}
	}

	// Search items above the selected one
	while ( (pSelItem = pSelItem->prevItem()) != 0 )
	{
		if ( pSelItem->isBookmarked() )
		{
			selectItem(pSelItem);
			return;
		}
	}
}

void KXE_TreeView::slotBookmarksNext()
{
	if ( childCount() < 1 )
	{
		kdDebug() << "KXE_TreeView::slotBookmarksNext: internal error - this tree view is empty" << endl;
		return;
	}

	// get selected item from tree view
	KXE_TreeViewItem * pSelItem = static_cast <KXE_TreeViewItem*> (selectedItem());
	if ( ! pSelItem )
	{                                                            // If there is no item selected
		pSelItem = static_cast <KXE_TreeViewItem*> (firstChild()); // we take the first root item,
		if ( pSelItem->isBookmarked() )                            // but we have to check its
		{                                                          // bookmarked-status
			selectItem(pSelItem);                                    // and select it, in case
			return;                                                  // it is bookmarked.
		}
	}

	// Search items below the selected one
	while ( (pSelItem = pSelItem->nextItem()) != 0 )
	{
		if ( pSelItem->isBookmarked() )
		{
			selectItem(pSelItem);
			return;
		}
	}
}

void KXE_TreeView::selectItem( KXE_TreeViewItem * const pItem )
{
	if ( ! pItem  )
	{
		kdDebug() << "KXE_TreeView::selectItem: the given pointer is a null pointer" << endl;
		return;
	}

	setSelected( pItem, true );
	setCurrentItem( pItem );
	ensureItemVisible( pItem );
}

bool KXE_TreeView::selectNode( const QDomNode & node )
{
	if ( node.isNull() )
	{
		kdError() << "KXE_TreeView::selectNode: the given node is an empty one" << endl;
		return false;
	}

	KXE_TreeViewItem * pItem = findCorrespondingItem(node);

	if ( ! pItem ) // can't find the corresponding item
	{
		kdError() << "KXE_TreeView::selectNode can't find an item to the given node." << endl;
		return false;
	}

	selectItem(pItem);
	return true;
}

QDomNode * KXE_TreeView::getSelectedNode() const
{
	// get selected item from tree view
	QListViewItem * pSelItem = selectedItem();
	if ( ! pSelItem )
		return 0;

	KXE_TreeViewItem * pSelTreeItem = static_cast <KXE_TreeViewItem *> (pSelItem);
	return pSelTreeItem->xmlNode();
}

QString KXE_TreeView::getSelectedPath() const
{
	// get selected item from tree view
	QListViewItem * pSelItem = selectedItem();
	if ( ! pSelItem )
		return QString();

	KXE_TreeViewItem * pSelTreeItem = static_cast <KXE_TreeViewItem *> (pSelItem);
	return domTool_getPath( * pSelTreeItem->xmlNode() );
}

void KXE_TreeView::contentsMousePressEvent( QMouseEvent * pEvent )
{
	KListView::contentsMousePressEvent(pEvent);

	if ( pEvent->button() == RightButton )
	{
		QWidget * pTmpWidget = 0;

		QListViewItem * pItem = itemAt( contentsToViewport(pEvent->pos()) );
		if (pItem)
		{
			KXE_TreeViewItem * pTreeItem = static_cast <KXE_TreeViewItem*> (pItem);
			switch( pTreeItem->xmlNode()->nodeType() )
			{
				case  QDomNode::ElementNode:
					pTmpWidget = m_pGUIClient->factory()->container( "popupXmlElement", m_pGUIClient );
					break;
				case	QDomNode::TextNode:
				case	QDomNode::CDATASectionNode:
				case	QDomNode::CommentNode:
					pTmpWidget = m_pGUIClient->factory()->container( "popupXmlContent", m_pGUIClient );
					break;
				case	QDomNode::ProcessingInstructionNode:
					pTmpWidget = m_pGUIClient->factory()->container( "popupXmlProcInstr", m_pGUIClient );
					break;
				default:
					kdDebug() << "KXE_TreeView::contentsMousePressEvent unknown item type" << endl;
					return;
			}
		}
		else
			pTmpWidget = m_pGUIClient->factory()->container( "popupXmlTree", m_pGUIClient );

		if ( ! pTmpWidget || ! pTmpWidget->inherits("KPopupMenu") )
		{
			kdDebug() << "KXE_TreeView::contentsMousePressEvent wrong container widget" << endl;
			return;
		}

		KPopupMenu * pMenu = static_cast <KPopupMenu*> (pTmpWidget);
		pMenu->popup( QCursor::pos() );
	}

}

void KXE_TreeView::slotSelectionChanged( QListViewItem * pItem )
{
	if ( ! pItem )
	{
		kdDebug() << "KXE_TreeView::slotSelectionChanged given pointer is null" << endl;
		return;
	}

	QDomNode selectedNode = * ( static_cast <KXE_TreeViewItem*> (pItem) )->xmlNode(); // uses QDomNode copy constructor

	// choose appropriate object kind
	switch ( selectedNode.nodeType() )
	{
		case  QDomNode::ElementNode:
			emit sigSelectionChanged( selectedNode.toElement() );
			break;

		case	QDomNode::TextNode:
		case	QDomNode::CDATASectionNode:
		case	QDomNode::CommentNode:
			emit sigSelectionChanged( selectedNode.toCharacterData() );
			break;

		case	QDomNode::ProcessingInstructionNode:
			emit sigSelectionChanged( selectedNode.toProcessingInstruction() );
			break;

			default:
				kdDebug() << "KXE_TreeView::slotSelectionChanged unknown object type selected" << endl;
				return;
		}
}

//////////////////////////////////////////////////////////////
// update slots
//////////////////////////////////////////////////////////////

void KXE_TreeView::slotUpdateNodeCreated( const QDomNode & node )
{
	if ( node.isNull() )
	{
		kdError() << "KXE_TreeView::slotUpdateNodeCreated the given node is an empty one." << endl;
		return;
	}

	if ( childCount() == 0 ) // no root item, so the given node
	{                        // has to be the documents root node
		if ( node.ownerDocument().documentElement() != node ) // it seems, it is not :-(
		{
			kdError() << "KXE_TreeView::slotUpdateNodeCreated the given node must be the root element, but isn't." << endl;
			return;
		}

		KXE_TreeViewItem * pRootItem = new KXE_TreeViewItem( node, this );
		if ( ! rootIsDecorated() )
			pRootItem->setOpen(true);
	}
	else   // this tree view has items, so the given node seems to have a parent node
	{
		if ( node.parentNode().isNull() ) // let's test it
		{
			kdError() << "KXE_TreeView::slotUpdateNodeCreated the given node has no parent node (but should)." << endl;
			return;
		}

		// To create the new item, we need (1st) the item corresponding to the parent node of the given one.
		QDomNode parentNode = node.parentNode();
		// Because the currently selected item is very likely (in many cases) the correct one, try it first.
		KXE_TreeViewItem * pParentItem = static_cast<KXE_TreeViewItem*> (selectedItem());
		if ( (!pParentItem) || ( *(pParentItem->xmlNode()) != parentNode ) )
		{                                                   // no strike :-(
			pParentItem = findCorrespondingItem(parentNode);  // do it the "long" way
		}

		if ( ! pParentItem ) // can't find the corresponding item
		{
			kdError() << "KXE_TreeView::slotUpdateNodeCreated can't find an item to the given nodes parent node." << endl;
			return;
		}

		// Now we need (2nd) the item corresponding to the previous sibling of the given one,
		//                   because, the new item has to be inserted behind the given one.
		QDomNode prevNode = node.previousSibling();
		if ( prevNode.isNull() )
		{                         // it seems to be the first child node, so create a first child item
			new KXE_TreeViewItem( node, pParentItem );
		}
		else
		{
			KXE_TreeViewItem * pPrevItem = findCorrespondingItem(prevNode);
			if ( ! pParentItem ) // can't find the corresponding item :-(
			{
				kdError() << "KXE_TreeView::slotUpdateNodeCreated can't find an item to the given nodes previous sibling." << endl;
				return;
			}
			                                    // everything's alright, let's create the new item
			new KXE_TreeViewItem( node, pParentItem, pPrevItem );
		}

	}
}

void KXE_TreeView::slotUpdateNodeChanged( const QDomNode & node )
{
	if ( node.isNull() )
	{
		kdError() << "KXE_TreeView::slotUpdateNodeChanged the given node is an empty one." << endl;
		return;
	}

	// To change the item, we have to find it.
	// Because the currently selected item is very likely (in many cases) the correct one, try it first.
	KXE_TreeViewItem * pItem = static_cast<KXE_TreeViewItem*> (selectedItem());
	if ( (!pItem) || ( *(pItem->xmlNode()) != node ) )  // no strike :-(
		pItem = findCorrespondingItem(node);              // do it the "long" way

	if ( ! pItem ) // can't find the corresponding item
	{
		kdError() << "KXE_TreeView::slotUpdateNodeChanged can't find an item to the given node." << endl;
		return;
	}

	pItem->setTexts(); // update the item
}

void KXE_TreeView::slotUpdateNodeDeleted( const QDomNode & node )
{
	if ( node.isNull() )
	{
		kdError() << "KXE_TreeView::slotUpdateNodeDeleted the given node is an empty one." << endl;
		return;
	}

	// To remove the item, we have to find it.
	// Because the currently selected item is very likely (in many cases) the correct one, try it first.
	KXE_TreeViewItem * pItem = static_cast<KXE_TreeViewItem*> (selectedItem());
	if ( (!pItem) || ( *(pItem->xmlNode()) != node ) )  // no strike :-(
		pItem = findCorrespondingItem(node);              // do it the "long" way

	if ( ! pItem ) // can't find the corresponding item
	{
		kdError() << "KXE_TreeView::slotUpdateNodeDeleted can't find an item to the given node." << endl;
		return;
	}

	delete pItem;
}

void KXE_TreeView::slotUpdateClear()
{
	clear();
}

//////////////////////////////////////////////////////////////
// misc functions
//////////////////////////////////////////////////////////////

KXE_TreeViewItem * KXE_TreeView::findCorrespondingItem( const QDomNode & node )
{
	KXE_TreeViewItem * pItem = static_cast<KXE_TreeViewItem*> (firstChild());
	while ( pItem )
	{
		if ( *(pItem->xmlNode()) == node )
			return pItem;
		pItem = pItem->nextItem();
	}

	return 0;
}