/* This file original part of the KDE libraries
    Modified for use in KXicq ( www.kxicq.org )
    Copyright (C) 1997 Torben Weis (weis@kde.org)

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
*/
#include <kxengine.h>
#include <qapplication.h>
#include <config.h>

#include <stdio.h>
#include <errno.h>
#ifdef STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_SYSENT_H
#include <sysent.h>
#endif
#include <sys/types.h>
#include <sys/stat.h>
#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif
#ifdef HAVE_SYS_SOCKET_H
// on Linux/libc5, this includes linux/socket.h where SOMAXCONN is defined
#include <sys/socket.h>
#endif
// Play it safe, use a reasonable default, if SOMAXCONN was nowhere defined.
#ifndef SOMAXCONN
#warning Your header files do not seem to support SOMAXCONN
#define SOMAXCONN 5
#endif

#include <sys/resource.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <fcntl.h>
#include <signal.h>

#include "kxsock.h"

#ifndef UNIX_PATH_MAX
#define UNIX_PATH_MAX 108 // this is the value, I found under Linux
#endif

int gethostbyname_r_portable(const char *szHostName, struct hostent *h)                                                                                                           
{                                                                                                                                                                                 
// Linux                                                                                                                                                                          
#if defined(__GLIBC__)                                                                                                                                                            
  struct hostent *h_buf;                                                                                                                                                          
  char temp[1024];                                                                                                                                                                
  int herror = 0;                                                                                                                                                                 
  gethostbyname_r(szHostName, h, temp, 1024, &h_buf, &herror);                                                                                                                    
  return herror;                                                                                                                                                                  
// Solaris                                                                                                                                                                        
#elif defined(sun)                                                                                                                                                                
  struct hostent *h_buf;                                                                                                                                                          
  char temp[1024];                                                                                                                                                                
  int herror = 0;                                                                                                                                                                 
  h_buf = gethostbyname_r(szHostName, h, temp, 1024, &herror);                                                                                                                    
  return herror;                                                                                                                                                                  
// not sure about this one (actually pretty sure it's wrong)                                                                                                                      
// who uses OSF anyway?                                                                                                                                                           
#elif defined(__osf__)                                                                                                                                                            
  h = gethostbyname(szHostName);                                                                                                                                                  
  return h_error;                                                                                                                                                                 
// Default to thread unsafe version                                                                                                                                               
#else                                                                                                                                                                             
#warning "I don't know how to do reentrant gethostbyname on this machine."                                                                                                        
#warning "Using thread-unsafe version."                                                                                                                                           
  struct hostent *h_buf;                                                                                                                                                          
  h_buf = gethostbyname(szHostName);                                                                                                                                              
  if (h_buf != NULL) memcpy(h, h_buf, sizeof(struct hostent));                                                                                                                    
  return (h == NULL ? h_errno : 0);                                                                                                                                               
#endif                                                                                                                                                                            
}    

kxSocket::kxSocket( const char *_host, unsigned short int _port, int _timeout ) :
  sock( -1 ), readNotifier( 0L ), writeNotifier( 0L )
{
    timeOut = _timeout;
    domain = PF_INET;
    connect( _host, _port );
}

kxSocket::kxSocket( const char *_host, unsigned short int _port ) :
  sock( -1 ), readNotifier( 0L ), writeNotifier( 0L )
{
    timeOut = 30;
    domain = PF_INET;
    connect( _host, _port );
}
kxSocket::kxSocket( Q_UINT32 _host, unsigned short int _port ) :
  sock( -1 ), readNotifier( 0L ), writeNotifier( 0L )
{
    timeOut = 60;
    domain = PF_INET;
    connect( _host, _port );
}

kxSocket::kxSocket( const char *_path ) :
  sock( -1 ), readNotifier( 0L ), writeNotifier( 0L )
{
  domain = PF_UNIX;
  connect( _path );
}

void kxSocket::enableRead( bool _state )
{
  if ( _state )
    {
	  if ( !readNotifier  )
		{
		  readNotifier = new QSocketNotifier( sock, QSocketNotifier::Read );
		  QObject::connect( readNotifier, SIGNAL( activated(int) ), this, SLOT( slotRead(int) ) );
		}
	  else
	    readNotifier->setEnabled( true );
    }
  else if ( readNotifier )
	readNotifier->setEnabled( false );
}

void kxSocket::enableWrite( bool _state )
{
  if ( _state )
    {
	  if ( !writeNotifier )
		{
		  writeNotifier = new QSocketNotifier( sock, QSocketNotifier::Write );
		  QObject::connect( writeNotifier, SIGNAL( activated(int) ), this, 
							SLOT( slotWrite(int) ) );
		}
	  else
	    writeNotifier->setEnabled( true );
    }
  else if ( writeNotifier )
	writeNotifier->setEnabled( false );
}

void kxSocket::slotRead( int )
{
  char buffer[2];
  
  int n = recv( sock, buffer, 1, MSG_PEEK );
  if ( n <= 0 )
	emit closeEvent( this );
  else
	emit readEvent( this );
}

void kxSocket::slotWrite( int )
{
  emit writeEvent( this );
}

/*
 * Initializes a sockaddr structure. Do this after creating a socket and
 * before connecting to any other socket. Here you must specify the
 * host and port you want to connect to.
 */
bool kxSocket::init_sockaddr( const char *hostname, unsigned short int port )
{
  if ( domain != PF_INET )
    return false;
  
  struct hostent *hostinfo;
  server_name.sin_family = AF_INET;
  server_name.sin_port = htons( port );
  hostinfo = gethostbyname( hostname );
  
  if ( !hostinfo )
    {
	  warning("Unknown host %s.\n",hostname);
	  return false;	
    }
  server_name.sin_addr = *(struct in_addr*) hostinfo->h_addr;    
  
  return true;
}
/*
 * Initializes a sockaddr structure. Do this after creating a socket and
 * before connecting to any other socket. Here you must specify the
 * host and port you want to connect to.
 */
bool kxSocket::init_sockaddr( Q_UINT32 hostname, unsigned short int port )
{
  if ( domain != PF_INET )
    return false;
  
  server_name.sin_family = AF_INET;
  server_name.sin_port = htons( port );
  server_name.sin_addr.s_addr = hostname;
  
  return true;
}

/*
 * Connects the PF_UNIX domain socket to _path.
 */
bool kxSocket::connect( const char *_path )
{
  if ( domain != PF_UNIX )
    fatal( "Connecting a PF_INET socket to a PF_UNIX domain socket\n");
  
  unix_addr.sun_family = AF_UNIX;
  int l = strlen( _path );
  if ( l > UNIX_PATH_MAX - 1 )
  {      
    warning( "Too long PF_UNIX domain name '%s'\n",_path);
    return false;
  }  
  strcpy( unix_addr.sun_path, _path );

  sock = ::socket(PF_UNIX,SOCK_STREAM,0);
  if (sock < 0)
	return false;
  
  if ( 0 > ::connect( sock, (struct sockaddr*)(&unix_addr), 
					  sizeof( unix_addr ) ) )
  {
      ::close( sock );
      sock = -1;
      return false;
  }

  return true;
}

/*
 * Connects the socket to _host, _port.
 */
bool kxSocket::connect( const char *_host, unsigned short int _port )
{
  if ( domain != PF_INET )
    fatal( "Connecting a PF_UNIX domain socket to a PF_INET domain socket\n");

  sock = ::socket(PF_INET,SOCK_STREAM,0);
  if (sock < 0)
	return false;
  
  if ( !init_sockaddr( _host, _port) )
	{
	  ::close( sock );
	  sock = -1;
	  return false;
	}

  fcntl(sock,F_SETFL,(fcntl(sock,F_GETFL)|O_NDELAY));

  errno = 0;
  if (::connect(sock, (struct sockaddr*)(&server_name), sizeof(server_name))){
      if(errno != EINPROGRESS && errno != EWOULDBLOCK){
          ::close( sock );
          sock = -1;
          return false;
      }
  }else
      return true;

  fd_set rd, wr;
  struct timeval timeout;
  int ret = 0, n;

  n = timeOut;
  FD_ZERO(&rd);
  FD_ZERO(&wr);
  FD_SET(sock, &rd);
  FD_SET(sock, &wr);
//  printf("timeout=%d\n", n);
  while(n--){
      timeout.tv_usec = 0;
      timeout.tv_sec = 1;

//      struct rlimit rlp;
//      getrlimit(RLIMIT_NOFILE, &rlp); // getdtablesize() equivalent. David Faure.

      ret = select(sock+1, (fd_set *)&rd, (fd_set *)&wr, (fd_set *)0,
                   (struct timeval *)&timeout);
      // if(ret)
      //    return(true);

      switch (ret)
      {
	  case 0: break; // Timeout
	  case 1: case 2: return(true); // Success
	  default: // Error
	      ::close(sock);
	      sock = -1;
	      return false;
      }

      qApp->processEvents();
      qApp->flushX();
  }
  warning("Timeout connecting socket...\n");
  ::close( sock );
  sock = -1;
  return false;
}
/*
 * Connects the socket to _host, _port.
 */
bool kxSocket::connect( Q_UINT32 _host, unsigned short int _port )
{
  if ( domain != PF_INET )
    fatal( "Connecting a PF_UNIX domain socket to a PF_INET domain socket\n");

  sock = ::socket(AF_INET,SOCK_STREAM,0);
  if (sock < 0)
	return false;
  
  if ( !init_sockaddr( _host, _port) )
	{
	  ::close( sock );
	  sock = -1;
	  return false;
	}

  int opt=fcntl(sock,F_GETFL,0);	
  debug("opt: %d",opt);
	opt=(opt|O_NONBLOCK);
  debug("fnctl: %d, %d, %d",fcntl(sock,F_SETFL,opt), opt, O_NONBLOCK);

  errno = 0;
  if (::connect(sock, (struct sockaddr*)(&server_name), sizeof(server_name))){
      if(errno != EINPROGRESS && errno != EWOULDBLOCK){
          ::close( sock );
          sock = -1;
          return false;
      }
  }else
      return true;
  debug("error: %s, %d, %d",strerror(errno),errno, EINPROGRESS);
/*
  fd_set rd, wr;
  struct timeval timeout;
  int ret = 0, n;

  n = timeOut*40;
  FD_ZERO(&rd);
  FD_ZERO(&wr);
  FD_SET(sock, &rd);
  FD_SET(sock, &wr);
  debug("timeout=%d\n", n);
  while(n--){
      timeout.tv_usec = 25000;
      timeout.tv_sec = 0;

//      struct rlimit rlp;
//      getrlimit(RLIMIT_NOFILE, &rlp); // getdtablesize() equivalent. David Faure.

      ret = select(sock+1, (fd_set *)&rd, (fd_set *)&wr, (fd_set *)0,
                   (struct timeval *)&timeout);
      // if(ret)
      //    return(true);
      // debug("ret: %d",ret);
      switch (ret)
      {
    	  case 0: break; // Timeout
    	  case 1: case 2: return(true); // Success
	      default: // Error
  	      ::close(sock);
  	      sock = -1;
	        return false;
      }

      qApp->processEvents();
      qApp->flushX();
  }
  warning("Timeout connecting socket...\n");
  ::close( sock );
  sock = -1;
  return false;
  */
  return true;
}

unsigned long kxSocket::getAddr()
{
  if ( domain != PF_INET )
    return 0;
  
  struct sockaddr_in name; ksize_t len = sizeof(name);
  getsockname(sock, (struct sockaddr *) &name, &len);
  return ntohl(name.sin_addr.s_addr);
}

kxSocket::~kxSocket()
{
    if ( readNotifier )
    {
	delete readNotifier;
    }
    if ( writeNotifier )
	delete writeNotifier; 
  
    ::close( sock ); 
}
kxServerSocket::kxServerSocket( const char *_path ) :
  notifier( 0L), sock( -1 )
{
  domain = PF_UNIX;
  
  if ( !init ( _path ) )
  {
    // fatal("Error constructing PF_UNIX domain server socket\n");
    return;
  }
    
  notifier = new QSocketNotifier( sock, QSocketNotifier::Read );
  connect( notifier, SIGNAL( activated(int) ), this, SLOT( slotAccept(int) ) );
}

kxServerSocket::kxServerSocket( int _port ) :
  notifier( 0L ), sock( -1 )
{
  domain = PF_INET;

  if ( !init ( _port ) )
  {
    // fatal("Error constructing\n");
    return;
  }
    
  notifier = new QSocketNotifier( sock, QSocketNotifier::Read );
  connect( notifier, SIGNAL( activated(int) ), this, SLOT( slotAccept(int) ) );
}

bool kxServerSocket::init( const char *_path )
{
  if ( domain != PF_UNIX )
    return false;
  
  int l = strlen( _path );
  if ( l > UNIX_PATH_MAX - 1 )
  {      
    warning( "Too long PF_UNIX domain name '%s'\n",_path);
    return false;
  }  
    
  sock = ::socket( PF_UNIX, SOCK_STREAM, 0 );
  if (sock < 0)
  {
    warning( "Could not create socket\n");
    return false;
  }

  unlink(_path);   

  struct sockaddr_un name;
  name.sun_family = AF_UNIX;
  strcpy( name.sun_path, _path );
    
  if ( bind( sock, (struct sockaddr*) &name,sizeof( name ) ) < 0 )
  {
    warning("Could not bind to socket\n");
    ::close( sock );
    sock = -1;
    return false;
  }
  
  if ( chmod( _path, 0600) < 0 )
  {
    warning("Could not setupt premissions for server socket\n");
    ::close( sock );
    sock = -1;
    return false;
  }
               
  if ( listen( sock, SOMAXCONN ) < 0 )
  {
    warning("Error listening on socket\n");
    ::close( sock );
    sock = -1;
    return false;
  }

  return true;
}

bool kxServerSocket::init( unsigned short int _port )
{
  if ( domain != PF_INET )
    return false;
  
  sock = ::socket( PF_INET, SOCK_STREAM, 0 );
  if (sock < 0)
  {
    warning( "Could not create socket\n");
    return false;
  }

  struct sockaddr_in name;
    
  name.sin_family = AF_INET;
  name.sin_port = htons( _port );
  name.sin_addr.s_addr = htonl(INADDR_ANY);
    
  if ( bind( sock, (struct sockaddr*) &name,sizeof( name ) ) < 0 )
  {
	  warning("Could not bind to socket\n");
	  ::close( sock );
	  sock = -1;
	  return false;
  }
    
  if ( listen( sock, SOMAXCONN ) < 0 )
  {
	  warning("Error listening on socket\n");
	  ::close( sock );
	  sock = -1;
	  return false;
  }

  struct sockaddr_in m_sLocalAddr=name;

  ksize_t  sizeofSockaddr = sizeof(struct sockaddr_in);                                                                                                                        
  if (getsockname(sock, (struct sockaddr *)&m_sLocalAddr, &sizeofSockaddr) < 0)                                                                                        
  {                                                                                                                                                                             
    return false;
  }                                                                                                                                                                             
  if (m_sLocalAddr.sin_addr.s_addr == INADDR_ANY)                                                                                                                               
  {                                                                                                                                                                             
    char szHostName[256];                                                                                                                                                       
    if (gethostname(szHostName, 256) == -1)                                                                                                                                     
    {                                                                                                                                                                           
      strcpy(szHostName, "localhost");                                                                                                                                          
      return false;
    }                                                                                                                                                                           
    struct hostent sLocalHost;                                                                                                                                                  
    h_errno = gethostbyname_r_portable(szHostName, &sLocalHost);                                                                                                                
    if (h_errno != 0)                                                                                                                                                           
    {                                                                                                                                                                           
      return false;
    }                                                                                                                                                                           
    m_sLocalAddr.sin_addr.s_addr = *((unsigned long *)sLocalHost.h_addr);
    name.sin_addr.s_addr = *((unsigned long *)sLocalHost.h_addr);    
  }
  
  Q_UINT32 ip=name.sin_addr.s_addr;    
  Q_UINT8 p1, p2, p3, p4;
	p1=ip;
	ip>>= 8;
	p2=ip;
	ip>>=8;
	p3=ip;
	ip>>=8;
	p4=ip;
	debug("IP1: [%d] %02d.%02d.%02d.%02d",ip, p1, p2, p3, p4);		
  our_ip=name.sin_addr.s_addr;
  ENGINE->ourIntIP=our_ip;

  return true;
}

unsigned short kxServerSocket::getPort()
{
  if ( domain != PF_INET )
    return false;

  struct sockaddr_in name; ksize_t len = sizeof(name);
  getsockname(sock, (struct sockaddr *) &name, &len);
  return ntohs(name.sin_port);
}

unsigned long kxServerSocket::getAddr()
{
//  if ( domain != PF_INET )
//    return false;

//  struct sockaddr_in name; ksize_t len = sizeof(name);
//  getsockname(sock, (struct sockaddr *) &name, &len);
// ntohl(name.sin_addr.s_addr);  
  return our_ip;
}

void kxServerSocket::slotAccept( int )
{
  if ( domain == PF_INET )
  {      
    struct sockaddr_in clientname;
    int new_sock;
    
    ksize_t size = sizeof(clientname);
    
    if ((new_sock = accept (sock, (struct sockaddr *) &clientname, &size)) < 0)
    {
      warning("Error accepting\n");
      return;
    }

    emit accepted( new kxSocket( new_sock ) );
  }
  else if ( domain == PF_UNIX )
  {      
    struct sockaddr_un clientname;
    int new_sock;
    
    ksize_t size = sizeof(clientname);
    
    if ((new_sock = accept (sock, (struct sockaddr *) &clientname, &size)) < 0)
    {
      warning("Error accepting\n");
      return;
    }

    emit accepted( new kxSocket( new_sock ) );
  }
}

kxServerSocket::~kxServerSocket()
{
  if ( notifier )
	delete notifier; 
  struct sockaddr_un name; ksize_t len = sizeof(name);
  getsockname(sock, (struct sockaddr *) &name, &len);
  close( sock );
  unlink(name.sun_path);                                                       
}

