/***************************************************************************
                          skymap.cpp  -  K Desktop Planetarium
                             -------------------
    begin                : Sat Feb 10 2001
    copyright            : (C) 2001 by Jason Harris
    email                : jharris@30doradus.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <kapp.h>
#include <kconfig.h>
#include <klocale.h>
#include <kurl.h>
#include <kiconloader.h>

#include <qpopupmenu.h>
#include <qlabel.h>
#include <qcursor.h>
#include <qpointarray.h>
#include <qarray.h>
#include <qfont.h>
#include <qtextstream.h>

#include <math.h>
#include <stdlib.h>
#include <stream.h>
#include <unistd.h>

#include "kstars.h"
#include "skymap.h"
#include "imageviewer.h"

SkyMap::SkyMap(QWidget *parent, const char *name )
 : QWidget (parent,name), downloads (0), computeSkymap (true)
{
	kapp->config()->setGroup( "View" );
	ZoomLevel = kapp->config()->readNumEntry( "ZoomLevel", 3 );
	focus.setRA( kapp->config()->readDoubleNumEntry( "FocusRA", 5.3 ) );
	focus.setDec( kapp->config()->readDoubleNumEntry( "FocusDec", 0.0 ) );
  oldfocus.set( focus.getRA(), focus.getDec() );

  //initialize pixelScale array, will be indexed by ZoomLevel
	for ( int i=0; i<13; ++i ) {
		pixelScale[i] = int( 256.*pow(sqrt(2.0),i) );
		Range[i] = 75.0*256.0/pixelScale[i];
	}
		
	setBackgroundColor( QColor( ((KStars *)parent)->GetOptions()->colorSky ) );	
	setBackgroundMode( QWidget::NoBackground );
	setFocusPolicy( QWidget::StrongFocus );
	setMouseTracking (true); //Generate MouseMove events!
	mouseButtonDown = false;
	slewing = false;

	sky = new QPixmap();
	initPopupMenus();
}

void SkyMap::initPopupMenus( void ) {
	pmStar = new QPopupMenu();
	pmStarTitle = new QLabel( i18n( "nothing" ), pmStar );
	pmStarTitle->setAlignment( AlignCenter );
	pmStar->insertItem( pmStarTitle );
	pmStar->insertSeparator();
	pmStar->insertItem( i18n( "Center on this" ), this, SLOT( slotCenter() ) );
	pmStar->insertSeparator();
	pmStar->insertItem( i18n( "Show 1st-Gen DSS image" ), this, SLOT( slotDSS() ) );
	pmStar->insertItem( i18n( "Show 2nd-Gen DSS image" ), this, SLOT( slotDSS2() ) );

	pmSolarSys = new QPopupMenu();
	pmSolTitle = new QLabel( i18n( "nothing" ), pmSolarSys );
	pmSolTitle->setAlignment( AlignCenter );
	pmSolarSys->insertItem( pmSolTitle );
	pmSolarSys->insertSeparator();
	pmSolarSys->insertItem( i18n( "Center on this" ), this, SLOT( slotCenter() ) );
	pmSolarSys->insertSeparator();
	idSolInfo = pmSolarSys->insertItem( i18n( "Show SEDS Information Page" ), this, SLOT( slotInfo() ) );

	pmMoon = new QPopupMenu();
	pmMoonTitle = new QLabel( i18n( "The Moon" ), pmMoon );
	pmMoonTitle->setAlignment( AlignCenter );
	pmMoon->insertItem( pmMoonTitle );
	pmMoon->insertSeparator();
	pmMoon->insertItem( i18n( "Center on this" ), this, SLOT( slotCenter() ) );
	pmMoon->insertSeparator();
	idMoonInfo = pmMoon->insertItem( i18n( "Show SEDS Information Page" ), this, SLOT( slotInfo() ) );
  idMoonImages = pmMoon->insertItem( i18n( "About Moon Images" ), this, SLOT( slotMoonPage() ) );

	pmMess = new QPopupMenu();
	pmMessTitle = new QLabel( i18n( "Messier Object" ), pmMess );
	pmMessTitle->setAlignment( AlignCenter );
	pmMessTitle2 = new QLabel( " ", pmMess );
	pmMessTitle2->setAlignment( AlignCenter );
	pmMessType = new QLabel( " ", pmMess );
	pmMessType->setAlignment( AlignCenter );
	pmMess->insertItem( pmMessTitle );
	pmMess->insertItem( pmMessTitle2 );
	pmMess->insertItem( pmMessType );
	pmMess->insertSeparator();
	pmMess->insertItem( i18n( "Center on this" ), this, SLOT( slotCenter() ) );
	pmMess->insertSeparator();
	idMessHST = pmMess->insertItem( i18n( "Show HST image" ), this, SLOT( slotHST() ) );
	pmMess->insertItem( i18n( "Show SEDS image" ), this, SLOT( slotSEDS() ) );
	pmMess->insertItem( i18n( "Show 1st-Gen DSS image" ), this, SLOT( slotDSS() ) );
	pmMess->insertItem( i18n( "Show 2nd-Gen DSS image" ), this, SLOT( slotDSS2() ) );
	pmMess->insertSeparator();
	idMessInfo = pmMess->insertItem( i18n( "Show SEDS Information Page" ), this, SLOT( slotInfo() ) );
	pmNGC = new QPopupMenu();
	pmNGCTitle = new QLabel( i18n( "NGC/IC Object" ), pmMess );
	pmNGCTitle->setAlignment( AlignCenter );
	pmNGCTitle2 = new QLabel( " ", pmMess );
	pmNGCTitle2->setAlignment( AlignCenter );
	pmNGCType = new QLabel( " ", pmMess );
	pmNGCType->setAlignment( AlignCenter );
	pmNGC->insertItem( pmNGCTitle );
	pmNGC->insertItem( pmNGCTitle2 );
	pmNGC->insertItem( pmNGCType );
	pmNGC->insertSeparator();
	pmNGC->insertItem( i18n( "Center on this" ), this, SLOT( slotCenter() ) );
	pmNGC->insertSeparator();
	idNGCHST = pmNGC->insertItem( i18n( "Show HST image" ), this, SLOT( slotHST() ) );
	pmNGC->insertItem( i18n( "Show 1st-Gen DSS image" ), this, SLOT( slotDSS() ) );
	pmNGC->insertItem( i18n( "Show 2nd-Gen DSS image" ), this, SLOT( slotDSS2() ) );
}

void SkyMap::setGeometry( int x, int y, int w, int h ) {
	QWidget::setGeometry( x, y, w, h );
	sky->resize( w, h );
}

void SkyMap::setGeometry( const QRect &r ) {
	QWidget::setGeometry( r );
	sky->resize( r.width(), r.height() );
}

void SkyMap::slotCenter( void ) {
	KStars *ksw = (KStars *)parent();
	focus.set( clickedPoint.getRA(), clickedPoint.getDec() );
	focus.RADecToAltAz( LSTh, ksw->geo->lat() );
	
	oldfocus.set( focus.getRA(), focus.getDec() );
	oldfocus.setAz( focus.getAz() );
	oldfocus.setAlt( focus.getAlt() );

	double dHA = LSTh.getH() - focus.getRA().getH();
  while ( dHA < 0.0 ) dHA += 24.0;
	HourAngle.setH( dHA );
	
	Update();  // must be new computed
}	

void SkyMap::slotDSS( void ) {
	QString URLprefix( "http://archive.stsci.edu/cgi-bin/dss_search?v=1" );
	QString URLsuffix( "&e=J2000&h=15.0&w=15.0&f=gif&c=none&fov=NONE" );
	QString RAString, DecString;
	char decsgn;
	RAString = RAString.sprintf( "&r=%02d+%02d+%02d", clickedPoint.getRA().getHour(),
																								 clickedPoint.getRA().getHMin(),
																								 clickedPoint.getRA().getHSec() );
	decsgn = '+';
	if (clickedPoint.getDec().getD() < 0.0) decsgn = '-';
	int dd = abs( clickedPoint.getDec().getDeg() );
	int dm = abs( clickedPoint.getDec().getArcMin() );
	int ds = abs( clickedPoint.getDec().getArcSec() );

	DecString = DecString.sprintf( "&d=%c%02d+%02d+%02d", decsgn, dd, dm, ds );

	//concat all the segments into the kview command line:
	KURL url (URLprefix + RAString + DecString + URLsuffix);
	new ImageViewer (&url, this);
}

void SkyMap::slotDSS2( void ) {
	QString URLprefix( "http://archive.stsci.edu/cgi-bin/dss_search?v=2r" );
	QString URLsuffix( "&e=J2000&h=15.0&w=15.0&f=gif&c=none&fov=NONE" );
	QString RAString, DecString;
	char decsgn;
	RAString = RAString.sprintf( "&r=%02d+%02d+%02d", clickedPoint.getRA().getHour(),
																								 clickedPoint.getRA().getHMin(),
																								 clickedPoint.getRA().getHSec() );
	decsgn = '+';
	if (clickedPoint.getDec().getD() < 0.0) decsgn = '-';
	int dd = abs( clickedPoint.getDec().getDeg() );
	int dm = abs( clickedPoint.getDec().getArcMin() );
	int ds = abs( clickedPoint.getDec().getArcSec() );

	DecString = DecString.sprintf( "&d=%c%02d+%02d+%02d", decsgn, dd, dm, ds );

	//concat all the segments into the kview command line:
	KURL url (URLprefix + RAString + DecString + URLsuffix);
	new ImageViewer (&url, this);
}

void SkyMap::slotSEDS( void ) {
	QString URLprefix( "http://www.seds.org/messier/Jpg/m" );
	QString URLsuffix( ".jpg" );
	QString id;
	id = foundObject->name.mid( 2, foundObject->name.length()-2 );
	KURL url (URLprefix + id + URLsuffix);
	new ImageViewer (&url, this);
}

void SkyMap::slotInfo( void ) {
	system( "konqueror " + foundObject->infoURL + " &" );
}

void SkyMap::slotMoonPage( void ) {
	system( "konqueror http://www.astrosurf.com/cidadao/ &" );
}

void SkyMap::slotHST( void ) {
	KURL url (foundObject->hstURL);
	if (!url.isEmpty())
		new ImageViewer (&url, this);
}
				
void SkyMap::keyReleaseEvent( QKeyEvent *e ) {
	slewing = false;
//	update();
	Update();	// Need a full update to draw faint objects that are not drawn while slewing.
}

void SkyMap::keyPressEvent( QKeyEvent *e ) {
	KStars *ksw = (KStars *)parent();
	QString s;

	switch ( e->key() ) {
		case Key_Left :
			if ( ksw->GetOptions()->useAltAz ) {
				focus.setAz( focus.getAz().getD() - 2.0*pixelScale[0]/pixelScale[ZoomLevel] );
				focus.AltAzToRADec( LSTh, ksw->geo->lat() );
			} else {
				focus.setRA( focus.getRA().getH() + 0.1 * pixelScale[0]/pixelScale[ZoomLevel] );
				focus.RADecToAltAz( LSTh, ksw->geo->lat() );
			}

			slewing = true;
			break;

		case Key_Right :
			if ( ksw->GetOptions()->useAltAz ) {
				focus.setAz( focus.getAz().getD() + 2.0*pixelScale[0]/pixelScale[ZoomLevel] );
				focus.AltAzToRADec( LSTh, ksw->geo->lat() );
			} else {
				focus.setRA( focus.getRA().getH() - 0.1 * pixelScale[0]/pixelScale[ZoomLevel] );
				focus.RADecToAltAz( LSTh, ksw->geo->lat() );
			}


			slewing = true;
			break;

		case Key_Up :
			if ( ksw->GetOptions()->useAltAz ) {
				focus.setAlt( focus.getAlt().getD() + 2.0*pixelScale[0]/pixelScale[ZoomLevel] );
				if ( focus.getAlt().getD() > 90.0 ) focus.setAlt( 90.0 );
				focus.AltAzToRADec( LSTh, ksw->geo->lat() );
			} else {
				focus.setDec( focus.getDec().getD() + 2.0 * pixelScale[0]/pixelScale[ZoomLevel] );
				if (focus.getDec().getD() > 90.0) focus.setDec( 90.0 );
				focus.RADecToAltAz( LSTh, ksw->geo->lat() );
			}

			slewing = true;
			break;

		case Key_Down:
			if ( ksw->GetOptions()->useAltAz ) {
				focus.setAlt( focus.getAlt().getD() - 2.0*pixelScale[0]/pixelScale[ZoomLevel] );
				if ( focus.getAlt().getD() < -90.0 ) focus.setAlt( -90.0 );
				focus.AltAzToRADec( LSTh, ksw->geo->lat() );
			} else {
				focus.setDec( focus.getDec().getD() - 2.0 * pixelScale[0]/pixelScale[ZoomLevel] );
				if (focus.getDec().getD() < -90.0) focus.setDec( -90.0 );
				focus.RADecToAltAz( LSTh, ksw->geo->lat() );
			}

			slewing = true;
			break;

		case Key_Plus:   //Zoom in
		case Key_Equal:
			ksw->mZoomIn();
			break;

		case Key_Minus:  //Zoom out
		case Key_Underscore:
			ksw->mZoomOut();
			break;
	}

	oldfocus.set( focus.getRA(), focus.getDec() );
	oldfocus.setAz( focus.getAz() );
	oldfocus.setAlt( focus.getAlt() );

	double dHA = LSTh.getH() - focus.getRA().getH();
  while ( dHA < 0.0 ) dHA += 24.0;
	HourAngle.setH( dHA );
	
	if (slewing) {
		ksw->GetOptions()->isTracking = false;
    ksw->actTrack->setIconSet( BarIcon( "unlock" ) );
	}

//	update();	// try only repaint, but be careful this might be wrong (every connected slot updates the skymap so I think this is OK)
	Update(); //need a total update, or slewing with the arrow keys doesn't work.
}

void SkyMap::mouseMoveEvent( QMouseEvent *e ) {
	KStars *ksw = (KStars *)parent();
	double dx = ( 0.5*width()  - e->x() )/pixelScale[ZoomLevel];
	double dy = ( 0.5*height() - e->y() )/pixelScale[ZoomLevel];

	//determine RA, Dec of mouse pointer
	MousePoint = dXdYToRaDec( dx, dy, ksw->GetOptions()->useAltAz, LSTh, ksw->geo->lat() );

	if ( mouseButtonDown ) {
		//Update focus such that the sky coords at mouse cursor remain approximately constant
		dms dRA = MousePoint.getRA().getD() - clickedPoint.getRA().getD();	
		dms dDec = MousePoint.getDec().getD() - clickedPoint.getDec().getD();

		focus.setRA( focus.getRA().getH() - dRA.getH() ); //move focus in opposite direction
		focus.setDec( focus.getDec().getD() - dDec.getD() );
		if ( focus.getDec().getD() >90.0 ) focus.setDec( 90.0 );
		if ( focus.getDec().getD() <-90.0 ) focus.setDec( -90.0 );
		focus.RADecToAltAz( LSTh, ksw->geo->lat() );
		oldfocus.set( focus.getRA(), focus.getDec() );
//		oldfocus.setAz( focus.getAz() );
//		oldfocus.setAlt( focus.getAlt() );

		double dHA = LSTh.getH() - focus.getRA().getH();
		while ( dHA < 0.0 ) dHA += 24.0;
		HourAngle.setH( dHA );

		//redetermine RA, Dec of mouse pointer, using new focus
		MousePoint = dXdYToRaDec( dx, dy, ksw->GetOptions()->useAltAz, LSTh, ksw->geo->lat() );
    clickedPoint.set( MousePoint.getRA(), MousePoint.getDec() );
		Update();  // must be new computed
	} else {
	
		QString dummy;
		int dd,dm,ds;
		char dsgn = '+';
		if ( MousePoint.getDec().getD() < 0.0 ) dsgn = '-';
		dd = abs( MousePoint.getDec().getDeg() );
		dm = abs( MousePoint.getDec().getArcMin() );
		ds = abs( MousePoint.getDec().getArcSec() );
		
		ksw->CurrentPosition = dummy.sprintf( " %02d:%02d:%02d, %c%02d:%02d:%02d ",
					MousePoint.getRA().getHour(), MousePoint.getRA().getHMin(),  MousePoint.getRA().getHSec(),
					dsgn, dd, dm, ds );
		ksw->statusBar()->changeItem( ksw->CurrentPosition, 1 );
	}
}

void SkyMap::mouseReleaseEvent( QMouseEvent *e ) {
	mouseButtonDown = false;
	slewing = false;
	Update();	// is needed because after moving the sky not all stars are shown
}

void SkyMap::mousePressEvent( QMouseEvent *e ) {
	KStars *ksw = (KStars *)parent();

	if ( mouseButtonDown == false ) {
		if ( e->button()==LeftButton ) {
			mouseButtonDown = true;
			slewing = true;
			ksw->GetOptions()->isTracking = false;
			ksw->actTrack->setIconSet( BarIcon( "unlock" ) );
		}

		double dx = ( 0.5*width()  - e->x() )/pixelScale[ZoomLevel];
		double dy = ( 0.5*height() - e->y() )/pixelScale[ZoomLevel];

		//determine RA, Dec of mouse pointer
		MousePoint = dXdYToRaDec( dx, dy, ksw->GetOptions()->useAltAz, LSTh, ksw->geo->lat() );
		clickedPoint.set( MousePoint.getRA(), MousePoint.getDec() );
	
		double rmin;
		double rstar_min = 10000000.0;
		int istar_min = -1;
	
		//Search stars database for nearby object.		
		for ( unsigned int i=0; i<ksw->GetData()->starList.count(); ++i ) {
			//test RA and dec to see if this star is roughly nearby
			SkyObject *test = (SkyObject *)ksw->GetData()->starList.at(i);
			double dRA = test->getRA().getH() - clickedPoint.getRA().getH();
			double dDec = test->getDec().getD() - clickedPoint.getDec().getD();
			//determine angular distance between this object and mouse cursor
			double f = 15.0*cos( test->getDec().radians() );
			double r = f*f*dRA*dRA + dDec*dDec; //no need to take sqrt, we just want to ID smallest value.
			if (r < rstar_min) {
				istar_min = i;
				rstar_min = r;
			}
		}
		
		rmin = rstar_min;
		int icat = 0;
		
		double rsolar_min = 10000000.0;
		int isolar_min = -1;
		
		//Sun
		double dRA = ksw->GetData()->Sun->getRA().getH() - clickedPoint.getRA().getH();
		double dDec = ksw->GetData()->Sun->getDec().getD() - clickedPoint.getDec().getD();
		double f = 15.0*cos( ksw->GetData()->Sun->getDec().radians() );
		double r = f*f*dRA*dRA + dDec*dDec; //no need to take sqrt, we just want to ID smallest value.
		if (r < rsolar_min) {
			isolar_min = 0;
			rsolar_min = r;
		}
		
		//Moon
		dRA = ksw->GetData()->Moon->getRA().getH() - clickedPoint.getRA().getH();
		dDec = ksw->GetData()->Moon->getDec().getD() - clickedPoint.getDec().getD();
		f = 15.0*cos( ksw->GetData()->Moon->getDec().radians() );
		r = f*f*dRA*dRA + dDec*dDec; //no need to take sqrt, we just want to ID smallest value.
		if (r < rsolar_min) {
			isolar_min = 1;
			rsolar_min = r;
		}
		
		//Mercury
		dRA = ksw->GetData()->Mercury->getRA().getH() - clickedPoint.getRA().getH();
		dDec = ksw->GetData()->Mercury->getDec().getD() - clickedPoint.getDec().getD();
		f = 15.0*cos( ksw->GetData()->Mercury->getDec().radians() );
		r = f*f*dRA*dRA + dDec*dDec; //no need to take sqrt, we just want to ID smallest value.
		if (r < rsolar_min) {
			isolar_min = 2;
			rsolar_min = r;
		}
		
		//Venus
		dRA = ksw->GetData()->Venus->getRA().getH() - clickedPoint.getRA().getH();
		dDec = ksw->GetData()->Venus->getDec().getD() - clickedPoint.getDec().getD();
		f = 15.0*cos( ksw->GetData()->Venus->getDec().radians() );
		r = f*f*dRA*dRA + dDec*dDec; //no need to take sqrt, we just want to ID smallest value.
		if (r < rsolar_min) {
			isolar_min = 3;
			rsolar_min = r;
		}
		
		//Mars
		dRA = ksw->GetData()->Mars->getRA().getH() - clickedPoint.getRA().getH();
		dDec = ksw->GetData()->Mars->getDec().getD() - clickedPoint.getDec().getD();
		f = 15.0*cos( ksw->GetData()->Mars->getDec().radians() );
		r = f*f*dRA*dRA + dDec*dDec; //no need to take sqrt, we just want to ID smallest value.
		if (r < rsolar_min) {
			isolar_min = 4;
			rsolar_min = r;
		}
		
		//Jupiter
		dRA = ksw->GetData()->Jupiter->getRA().getH() - clickedPoint.getRA().getH();
		dDec = ksw->GetData()->Jupiter->getDec().getD() - clickedPoint.getDec().getD();
		f = 15.0*cos( ksw->GetData()->Jupiter->getDec().radians() );
		r = f*f*dRA*dRA + dDec*dDec; //no need to take sqrt, we just want to ID smallest value.
		if (r < rsolar_min) {
			isolar_min = 5;
			rsolar_min = r;
		}
		
		//Saturn
		dRA = ksw->GetData()->Saturn->getRA().getH() - clickedPoint.getRA().getH();
		dDec = ksw->GetData()->Saturn->getDec().getD() - clickedPoint.getDec().getD();
		f = 15.0*cos( ksw->GetData()->Saturn->getDec().radians() );
		r = f*f*dRA*dRA + dDec*dDec; //no need to take sqrt, we just want to ID smallest value.
		if (r < rsolar_min) {
			isolar_min = 6;
			rsolar_min = r;
		}
		
		//Uranus
		dRA = ksw->GetData()->Uranus->getRA().getH() - clickedPoint.getRA().getH();
		dDec = ksw->GetData()->Uranus->getDec().getD() - clickedPoint.getDec().getD();
		f = 15.0*cos( ksw->GetData()->Uranus->getDec().radians() );
		r = f*f*dRA*dRA + dDec*dDec; //no need to take sqrt, we just want to ID smallest value.
		if (r < rsolar_min) {
			isolar_min = 7;
			rsolar_min = r;
		}
		
		//Neptune
		dRA = ksw->GetData()->Neptune->getRA().getH() - clickedPoint.getRA().getH();
		dDec = ksw->GetData()->Neptune->getDec().getD() - clickedPoint.getDec().getD();
		f = 15.0*cos( ksw->GetData()->Neptune->getDec().radians() );
		r = f*f*dRA*dRA + dDec*dDec; //no need to take sqrt, we just want to ID smallest value.
		if (r < rsolar_min) {
			isolar_min = 8;
			rsolar_min = r;
		}
		
		if (rsolar_min < rmin) {
			rmin = rsolar_min;
			icat = -1;
		}
		
		double rmess_min = 10000000.0;
		int imess_min = -1;
	
		//Search Messier database for nearby object.
		for ( unsigned int i=0; i<ksw->GetData()->messList.count(); ++i ) {
			//test RA and dec to see if this star is roughly nearby
			SkyObject *test = (SkyObject *)ksw->GetData()->messList.at(i);
			double dRA = test->getRA().getH()-clickedPoint.getRA().getH();
			double dDec = test->getDec().getD()-clickedPoint.getDec().getD();
				double f = 15.0*cos( test->getDec().radians() );
			  double r = f*f*dRA*dRA + dDec*dDec; //no need to take sqrt, we just want to ID smallest value.
				if (r < rmess_min) {
					imess_min = i;
					rmess_min = r;
				}
		}
	
		if (rmess_min < rmin) {
			rmin = rmess_min;
			icat = 1;
		}
		
		double rngc_min = 10000000.0;
		int ingc_min = -1;
	
		//Search NGC database for nearby object.
		for ( unsigned int i=0; i<ksw->GetData()->ngcList.count(); ++i ) {
			//test RA and dec to see if this star is roughly nearby
			SkyObject *test = (SkyObject *)ksw->GetData()->ngcList.at(i);
			double dRA = test->getRA().getH()-clickedPoint.getRA().getH();
			double dDec = test->getDec().getD()-clickedPoint.getDec().getD();
				double f = 15.0*cos( test->getDec().radians() );
			  double r = f*f*dRA*dRA + dDec*dDec; //no need to take sqrt, we just want to ID smallest value.
				if (r < rngc_min) {
					ingc_min = i;
					rngc_min = r;
				}
		}
	
		if (rngc_min < rmin) {
			rmin = rngc_min;
			icat = 2;
		}
		
		double ric_min = 10000000.0;
		int iic_min = -1;
	
		//Search NGC database for nearby object.
		for ( unsigned int i=0; i<ksw->GetData()->icList.count(); ++i ) {
			//test RA and dec to see if this star is roughly nearby
			SkyObject *test = (SkyObject *)ksw->GetData()->icList.at(i);
			double dRA = test->getRA().getH()-clickedPoint.getRA().getH();
			double dDec = test->getDec().getD()-clickedPoint.getDec().getD();
				double f = 15.0*cos( test->getDec().radians() );
			  double r = f*f*dRA*dRA + dDec*dDec; //no need to take sqrt, we just want to ID smallest value.
				if (r < ric_min) {
					iic_min = i;
					ric_min = r;
				}
		}

		if (ric_min < rmin) {
			rmin = ric_min;
			icat = 3;
		}
		
		if ( rmin < 80.0/pixelScale[ZoomLevel] ) {
			switch (icat) {
				case -1: //solar system object
					switch( isolar_min ) {
						case 0: //sun
							foundObject = (SkyObject *)ksw->GetData()->Sun;
							break;
						case 1: //moon
							foundObject = (SkyObject *)ksw->GetData()->Moon;
							break;
						case 2: //mercury
							foundObject = (SkyObject *)ksw->GetData()->Mercury;
							break;
						case 3: //venus
							foundObject = (SkyObject *)ksw->GetData()->Venus;
							break;
						case 4: //mars
							foundObject = (SkyObject *)ksw->GetData()->Mars;
							break;
						case 5: //jupiter
							foundObject = (SkyObject *)ksw->GetData()->Jupiter;
							break;
						case 6: //saturn
							foundObject = (SkyObject *)ksw->GetData()->Saturn;
							break;
						case 7: //uranus
							foundObject = (SkyObject *)ksw->GetData()->Uranus;
							break;
						case 8: //neptune
							foundObject = (SkyObject *)ksw->GetData()->Neptune;
							break;
					}
					
					if ( foundObject != NULL ) clickedPoint.set( foundObject->getRA(), foundObject->getDec() );
					
					switch (e->button()) {
						case LeftButton:
							ksw->statusBar()->changeItem( foundObject->name, 0 );
							break;
						case RightButton:
							if ( foundObject->name == i18n( "Moon" ) ) {
								pmMoon->setItemEnabled( idSolInfo, !foundObject->infoURL.isEmpty() );
								pmMoon->popup( QCursor::pos() );
							} else {
								pmSolTitle->setText( foundObject->name );
								pmSolarSys->setItemEnabled( idSolInfo, !foundObject->infoURL.isEmpty() );
								pmSolarSys->popup( QCursor::pos() );
							}				
							break;
					}
					break;
				case 0: //star
					foundObject = (SkyObject *)ksw->GetData()->starList.at(istar_min);
					clickedPoint.set( foundObject->getRA(), foundObject->getDec() );	
					
					switch (e->button()) {
						case LeftButton:
							ksw->statusBar()->changeItem( foundObject->name, 0 );
							break;
						case RightButton:
							pmStarTitle->setText( foundObject->name );
							pmStar->popup( QCursor::pos() );
							break;
						default:
							break;
					}
					break;
				case 1: //Messier Object
					foundObject = (SkyObject *)ksw->GetData()->messList.at(imess_min);
					clickedPoint.set( foundObject->getRA(), foundObject->getDec() );
					
					switch (e->button()) {
						case LeftButton:
							ksw->statusBar()->changeItem( foundObject->name, 0 );
							break;
						case RightButton:
							pmMessTitle->setText( foundObject->name );
							if (foundObject->longname != QString::null) {
								pmMessTitle2->setText( foundObject->longname );
							} else {
								pmMessTitle2->setText( foundObject->name2 );
							}
							pmMessType->setText( ksw->TypeName[foundObject->type] );
							pmMess->setItemEnabled( idMessHST, !foundObject->hstURL.isEmpty() );
							pmMess->setItemEnabled( idMessInfo, !foundObject->infoURL.isEmpty() );
							pmMess->popup( QCursor::pos() );
							break;
						default:
							break;
						}
					break;
				case 2: //NGC Object
					foundObject = (SkyObject *)ksw->GetData()->ngcList.at(ingc_min);
					clickedPoint.set( foundObject->getRA(), foundObject->getDec() );
	
					switch (e->button()) {
						case LeftButton:
							ksw->statusBar()->changeItem( foundObject->name, 0 );
							break;
						case RightButton:
							pmNGCTitle->setText( foundObject->name );
							pmNGCTitle2->setText( foundObject->longname );
							pmNGCType->setText( ksw->TypeName[foundObject->type] );
							pmNGC->setItemEnabled( idNGCHST, !foundObject->hstURL.isEmpty() );
							pmNGC->popup( QCursor::pos() );
							break;
						default:
							break;
					}
					break;
				case 3: //IC Object
					foundObject = (SkyObject *)ksw->GetData()->icList.at(iic_min);
					clickedPoint.set( foundObject->getRA(), foundObject->getDec() );
	
					switch (e->button()) {
						case LeftButton:
							ksw->statusBar()->changeItem( foundObject->name, 0 );
							break;
						case RightButton:
							pmNGCTitle->setText( foundObject->name );
							pmNGCTitle2->setText( foundObject->longname );
							pmNGCType->setText( ksw->TypeName[foundObject->type] );
							pmNGC->setItemEnabled( idNGCHST, !foundObject->hstURL.isEmpty() );
							pmNGC->popup( QCursor::pos() );
							break;
						default:
							break;
					}
					break;
			}
		} else {
			//Empty sky selected.  If left-click, display "nothing" in the status bar.
	    //If right-click, open pmStar with "nothing" as star name.
			foundObject = NULL;
	
			switch (e->button()) {
				case LeftButton:
					ksw->statusBar()->changeItem( i18n( "nothing" ), 0 );
					break;
				case RightButton:
					pmStarTitle->setText( "nothing" );
					pmStar->popup( QCursor::pos() );
					break;
				default:
					break;
			}
		}
	}
}
	
void SkyMap::mouseDoubleClickEvent( QMouseEvent *e ) {
	KStars *ksw = (KStars *)parent();

	double dx = ( 0.5*width()  - e->x() )/pixelScale[ZoomLevel];
	double dy = ( 0.5*height() - e->y() )/pixelScale[ZoomLevel];

	//determine RA, Dec of mouse pointer
	MousePoint = dXdYToRaDec( dx, dy, ksw->GetOptions()->useAltAz, LSTh, ksw->geo->lat() );

	QString dummy;
	ksw->CurrentPosition = dummy.sprintf( " %02d:%02d:%02d, %02d:%02d:%02d ",
				MousePoint.getRA().getHour(), MousePoint.getRA().getHMin(),  MousePoint.getRA().getHSec(),
				MousePoint.getDec().getDeg(), MousePoint.getDec().getArcMin(), MousePoint.getDec().getArcSec() );
	ksw->statusBar()->changeItem( ksw->CurrentPosition, 1 );

	if (dx != 0.0 || dy != 0.0) {  //clicked on a new position?
		focus.set( MousePoint.getRA(), MousePoint.getDec() );
		focus.RADecToAltAz( LSTh, ksw->geo->lat() );
		oldfocus.set( MousePoint.getRA(), MousePoint.getDec() );
		oldfocus.setAz( focus.getAz() );
		oldfocus.setAlt( focus.getAlt() );

		double dHA = LSTh.getH() - focus.getRA().getH();
	  while ( dHA < 0.0 ) dHA += 24.0;
		HourAngle.setH( dHA );
		
		update();	// must be new computed
	}
}

void SkyMap::paintEvent( QPaintEvent *e ) {
// if the skymap should be only repainted and constellations need not to be new computed; call this with update() (default)
	if (!computeSkymap)
	{
		bitBlt( this, 0, 0, sky );
		return ; // exit because the pixmap is repainted and that's all what we want
	}

// if the constallations should be new computed (this is not every paintEvent call needed, explicite call with Update())
	QPainter psky;
	KStars *ksw = (KStars *)parent();

	float FOV = Range[ZoomLevel]*width()/600.;
	float dY;
  bool isPoleVisible = false;
	if ( ksw->GetOptions()->useAltAz ) {
		dY = fabs( focus.getAlt().getD() ) + FOV;
	} else {
		dY = fabs( focus.getDec().getD() ) + FOV;
	}
	if ( dY >= 90. ) isPoleVisible = true;

	psky.begin( sky );
	psky.fillRect( 0, 0, width(), height(), QBrush( ksw->GetOptions()->colorSky ) );
	
	QFont stdFont = psky.font();
	QFont smallFont = psky.font();
	smallFont.setPointSize( stdFont.pointSize() - 2 );
	
	//Draw Equator
	if ( ksw->GetOptions()->drawEquator ) {
		psky.setPen( QColor( ksw->GetOptions()->colorEq ) );

		QPoint o = getXY( ksw->GetData()->Equator.at(0), ksw->GetOptions()->useAltAz, LSTh, ksw->geo->lat() );
		QPoint o1 = o;
		psky.moveTo( o.x(), o.y() );

		for ( unsigned int i=1; i<ksw->GetData()->Equator.count(); ++i ) {
			o = getXY( ksw->GetData()->Equator.at(i), ksw->GetOptions()->useAltAz, LSTh, ksw->geo->lat() );

			int dx = psky.pos().x() - o.x();
		
			if ( abs(dx) < 250 ) {
				psky.lineTo( o.x(), o.y() );
			} else {
				psky.moveTo( o.x(), o.y() );	
			}
		}
		//connect the final segment
		int dx = psky.pos().x() - o1.x();
		if (abs(dx) < 250 ) {
			psky.lineTo( o1.x(), o1.y() );
		} else {
			psky.moveTo( o1.x(), o1.y() );	
		}
  }

	//Draw Ecliptic
	if ( ksw->GetOptions()->drawEcliptic ) {
		psky.setPen( QColor( ksw->GetOptions()->colorEcl ) );
		
		QPoint o = getXY( ksw->GetData()->Ecliptic.at(0), ksw->GetOptions()->useAltAz, LSTh, ksw->geo->lat() );
		QPoint o1 = o;
		psky.moveTo( o.x(), o.y() );

		for ( unsigned int i=1; i<ksw->GetData()->Ecliptic.count(); ++i ) {
			o = getXY( ksw->GetData()->Ecliptic.at(i), ksw->GetOptions()->useAltAz, LSTh, ksw->geo->lat() );
			int dx = psky.pos().x() - o.x();
		
			if ( abs(dx) < 250 ) {
				psky.lineTo( o.x(), o.y() );
			} else {
				psky.moveTo( o.x(), o.y() );	
			}
		}
		//connect the final segment
		int dx = psky.pos().x() - o1.x();
		if (abs(dx) < 250 ) {
			psky.lineTo( o1.x(), o1.y() );
		} else {
			psky.moveTo( o1.x(), o1.y() );	
		}
  }

	//Draw Constellation Lines
	if ( ksw->GetOptions()->drawConstellLines ) {
		psky.setPen( QColor( ksw->GetOptions()->colorCLine ) );
		int iLast = 0;

		for ( unsigned int i=0; i < ksw->GetData()->clineList.count(); ++i ) {
			QPoint o = getXY( ksw->GetData()->clineList.at(i), ksw->GetOptions()->useAltAz, LSTh, ksw->geo->lat() );

			if ( ( o.x() >= -1000 && o.x() <= width()+1000 && o.y() >=-1000 && o.y() <=height()+1000 ) &&
					 ( o.x() >= -1000 && o.x() <= width()+1000 && o.y() >=-1000 && o.y() <=height()+1000 ) ) {
				if ( ksw->GetData()->clineModeList.at(i)->latin1()=='M' ) {
					psky.moveTo( o.x(), o.y() );
				} else if ( ksw->GetData()->clineModeList.at(i)->latin1()=='D' ) {
					if ( i==iLast+1 ) {
						psky.lineTo( o.x(), o.y() );
					} else {
						psky.moveTo( o.x(), o.y() );
  	      }
				}
				iLast = i;
			}
		}
  }

	//Draw Constellation Names
	//Don't draw names if slewing
	if ( !slewing && ksw->GetOptions()->drawConstellNames ) {
		psky.setFont( stdFont );
		psky.setPen( QColor( ksw->GetOptions()->colorCName ) );
		for ( unsigned int i=0; i < ksw->GetData()->cnameList.count(); ++i ) {
			if ( checkVisibility( ksw->GetData()->cnameList.at(i)->pos(), FOV, ksw->GetOptions()->useAltAz, isPoleVisible ) ) {
				QPoint o = getXY( ksw->GetData()->cnameList.at(i)->pos(), ksw->GetOptions()->useAltAz, LSTh, ksw->geo->lat() );
				int dx = 5*ksw->GetData()->cnameList.at(i)->name.length();
				if (o.x() >= 0 && o.x() <= width() && o.y() >=0 && o.y() <=height() ) {
					psky.drawText( o.x()-dx, o.y(), ksw->GetData()->cnameList.at(i)->name );
				}
			}
		}
  }
	
	//Draw Stars
  //Only draw bright stars if slewing
	if ( ksw->GetOptions()->drawBSC ) {
		psky.setPen( QColor( ksw->GetOptions()->colorSky ) );
		psky.setBrush( QColor( ksw->GetOptions()->colorStar ) );
		if ( ZoomLevel < 6 ) {
			psky.setFont( smallFont );
		} else {
			psky.setFont( stdFont );
		}
		unsigned int numStars = ksw->GetData()->starList.count();
		if ( slewing ) numStars = ksw->GetData()->starCount1;

		// draw a limited amount of stars
		for ( unsigned int i=0; i < numStars; ++i ) {
			SkyObject* curStar = ksw->GetData()->starList.at(i);
			if ( checkVisibility( curStar->pos(), FOV, ksw->GetOptions()->useAltAz, isPoleVisible ) ) {
				QPoint o = getXY( curStar->pos(), ksw->GetOptions()->useAltAz, LSTh, ksw->geo->lat() );
			
				// draw star if currently on screen
				if (o.x() >= 0 && o.x() <= width() && o.y() >=0 && o.y() <=height() ) {
					// but only if the star is bright enough.				
					if ( curStar->mag <= ksw->GetOptions()->magLimitDrawStar ) {
						float mag = curStar->mag;
						int size = int( 9.0 - mag ) + 1;
						if ( ZoomLevel < 2 ) {
							size = int( 7.5 - mag ) + 1;
						}
						psky.setPen( QColor( ksw->GetOptions()->colorSky ) );
						drawSymbol( psky, curStar->type, o.x(), o.y(), size );
						// now that we have drawn the star, we can display some extra info
						if ( !slewing && (curStar->mag <= ksw->GetOptions()->magLimitDrawStarInfo )
								&& (ksw->GetOptions()->drawStarName || ksw->GetOptions()->drawStarMagnitude ) )
						{
							// collect info text
							QString sTmp = "";
							if ( ksw->GetOptions()->drawStarName ) {
								if (curStar->name != "star") sTmp = curStar->name + " ";
							}
							if ( ksw->GetOptions()->drawStarMagnitude ) {
								sTmp += QString().sprintf("%.1f", curStar->mag );
							}
  	    	 	  int offset = int( 7.5 - mag ) + 5;
							psky.setPen( QColor( ksw->GetOptions()->colorSName ) );
							psky.drawText( o.x()+offset, o.y()+offset, sTmp );
  	    	 	}
					}
				}
			}
		}
  }

	//Draw Messier Objects
	if ( /*!slewing &&*/ ksw->GetOptions()->drawMessier ) {
		psky.setBrush( QColor( ksw->GetOptions()->colorSky ) );
		psky.setPen( QColor( ksw->GetOptions()->colorMess ) );

		for ( unsigned int i=0; i < ksw->GetData()->messList.count(); ++i ) {
			if ( checkVisibility( ksw->GetData()->messList.at(i)->pos(), FOV, ksw->GetOptions()->useAltAz, isPoleVisible ) ) {
				QPoint o = getXY( ksw->GetData()->messList.at(i)->pos(), ksw->GetOptions()->useAltAz, LSTh, ksw->geo->lat() );
				if (o.x() >= 0 && o.x() <= width() && o.y() >=0 && o.y() <=height() ) {
					int size = pixelScale[ZoomLevel]/pixelScale[0];
					if (size > 16) size = 16;

					drawSymbol( psky, ksw->GetData()->messList.at(i)->type, o.x(), o.y(), size );
			
					//Draw symbol with new color if it's a Hubble shot
					if ( !ksw->GetData()->messList.at(i)->hstURL.isEmpty() ) {
						psky.setPen( QColor( ksw->GetOptions()->colorHST ) );
						drawSymbol( psky, ksw->GetData()->messList.at(i)->type, o.x(), o.y(), size );
						psky.setPen( QColor( ksw->GetOptions()->colorMess ) );
					}
				}
			}
		}
  }

  //Draw NGC Objects
  if ( !slewing && ksw->GetOptions()->drawNGC ) {
		psky.setBrush( QColor( ksw->GetOptions()->colorSky ) );
		psky.setPen( QColor( ksw->GetOptions()->colorNGC ) );
		
		for ( unsigned int i=0; i < ksw->GetData()->ngcList.count(); ++i ) {
			if ( checkVisibility( ksw->GetData()->ngcList.at(i)->pos(), FOV, ksw->GetOptions()->useAltAz, isPoleVisible ) ) {
				QPoint o = getXY( ksw->GetData()->ngcList.at(i)->pos(), ksw->GetOptions()->useAltAz, LSTh, ksw->geo->lat() );
				if ( o.x() >= 0 && o.x() <= width() && o.y() >= 0 && o.y() <= height() ) {
					int size = pixelScale[ZoomLevel]/pixelScale[0];
					if (size>8) size = 8;
				
					drawSymbol( psky, ksw->GetData()->ngcList.at(i)->type, o.x(), o.y(), size );
				
					//Draw symbol with new color if its a Hubble shot
					if ( !ksw->GetData()->ngcList.at(i)->hstURL.isEmpty() ) {
						psky.setPen( QColor( ksw->GetOptions()->colorHST ) );
						drawSymbol( psky, ksw->GetData()->ngcList.at(i)->type, o.x(), o.y(), size );
						psky.setPen( QColor( ksw->GetOptions()->colorNGC ) );
					}
				}
			}
		}
	}

  //Draw IC Objects
  if ( !slewing && ksw->GetOptions()->drawIC ) {
		psky.setBrush( QColor( ksw->GetOptions()->colorSky ) );
		psky.setPen( QColor( ksw->GetOptions()->colorIC ) );
		
		for ( unsigned int i=0; i < ksw->GetData()->icList.count(); ++i ) {
			if ( checkVisibility( ksw->GetData()->icList.at(i)->pos(), FOV, ksw->GetOptions()->useAltAz, isPoleVisible ) ) {
				QPoint o = getXY( ksw->GetData()->icList.at(i)->pos(), ksw->GetOptions()->useAltAz, LSTh, ksw->geo->lat() );
				if ( o.x() >= 0 && o.x() <= width() && o.y() >= 0 && o.y() <= height() ) {
					int size = pixelScale[ZoomLevel]/pixelScale[0];
					if (size>8) size = 8;
				
					drawSymbol( psky, ksw->GetData()->icList.at(i)->type, o.x(), o.y(), size );
				
					//Draw a red "H" where hubble shots exist
					if ( !ksw->GetData()->icList.at(i)->hstURL.isEmpty() ) {
						psky.setPen( QColor( ksw->GetOptions()->colorHST ) );
						drawSymbol( psky, ksw->GetData()->icList.at(i)->type, o.x(), o.y(), size );
						psky.setPen( QColor( ksw->GetOptions()->colorIC ) );
					}
				}
			}
		}
	}

	QImage ScaledImage;
	
	//Draw Sun
  if ( ksw->GetOptions()->drawSun ) {
		psky.setPen( QColor( "Yellow" ) );
		psky.setBrush( QColor( "Yellow" ) );
		QPoint o = getXY( ksw->GetData()->Sun->pos(), ksw->GetOptions()->useAltAz, LSTh, ksw->geo->lat() );
		if ( o.x() >= 0 && o.x() <= width() && o.y() >= 0 && o.y() <= height() ) {
			int size = 8*pixelScale[ZoomLevel]/pixelScale[0];
			int x1 = o.x() - size/2;
			int y1 = o.y() - size/2;
			if ( !ksw->GetData()->Sun->image.isNull() ) {
				ScaledImage = ksw->GetData()->Sun->image.smoothScale( size, size );
				psky.drawImage( x1, y1, ScaledImage );
			} else {
				psky.drawEllipse( x1, y1, size, size );
			}
		}
	}
	
	//Draw Moon
  if ( ksw->GetOptions()->drawMoon ) {
		psky.setPen( QColor( "White" ) );
		psky.setBrush( QColor( "White" ) );
		QPoint o = getXY( ksw->GetData()->Moon->pos(), ksw->GetOptions()->useAltAz, LSTh, ksw->geo->lat() );
		if ( o.x() >= 0 && o.x() <= width() && o.y() >= 0 && o.y() <= height() ) {
			int size = 8*pixelScale[ZoomLevel]/pixelScale[0];
			int x1 = o.x() - size/2;
			int y1 = o.y() - size/2;
			if ( !ksw->GetData()->Moon->image.isNull() ) {
				ScaledImage = ksw->GetData()->Moon->image.smoothScale( size, size );
				psky.drawImage( x1, y1, ScaledImage );
			} else {
				psky.drawEllipse( x1, y1, size, size );
			}
		}
	}
	
	//Draw Mercury
  if ( ksw->GetOptions()->drawMercury ) {
		psky.setPen( QColor( "SlateBlue1" ) );
		psky.setBrush( QColor( "SlateBlue1" ) );
		QPoint o = getXY( ksw->GetData()->Mercury->pos(), ksw->GetOptions()->useAltAz, LSTh, ksw->geo->lat() );
		if ( o.x() >= 0 && o.x() <= width() && o.y() >= 0 && o.y() <= height() ) {
			int size = 4*pixelScale[ZoomLevel]/pixelScale[0];
			int x1 = o.x() - size/2;
			int y1 = o.y() - size/2;
			if ( ZoomLevel>2 && !ksw->GetData()->Mercury->image.isNull() ) {
				ScaledImage = ksw->GetData()->Mercury->image.smoothScale( size, size );
				psky.drawImage( x1, y1, ScaledImage );
			} else {
				psky.drawEllipse( x1, y1, size, size );
			}
		}
	}
				
	//Draw Venus
  if ( ksw->GetOptions()->drawVenus ) {
		psky.setPen( QColor( "LightGreen" ) );
		psky.setBrush( QColor( "LightGreen" ) );
		QPoint o = getXY( ksw->GetData()->Venus->pos(), ksw->GetOptions()->useAltAz, LSTh, ksw->geo->lat() );
		if ( o.x() >= 0 && o.x() <= width() && o.y() >= 0 && o.y() <= height() ) {
			int size = 4*pixelScale[ZoomLevel]/pixelScale[0];
			int x1 = o.x() - size/2;
			int y1 = o.y() - size/2;
			if ( ZoomLevel>2 && !ksw->GetData()->Venus->image.isNull() ) {
				ScaledImage = ksw->GetData()->Venus->image.smoothScale( size, size );
				psky.drawImage( x1, y1, ScaledImage );
			} else {
				psky.drawEllipse( x1, y1, size, size );
			}
		}
	}
			
	//Draw Mars
  if ( ksw->GetOptions()->drawMars ) {
		psky.setPen( QColor( "Red" ) );
		psky.setBrush( QColor( "Red" ) );
		QPoint o = getXY( ksw->GetData()->Mars->pos(), ksw->GetOptions()->useAltAz, LSTh, ksw->geo->lat() );
		if ( o.x() >= 0 && o.x() <= width() && o.y() >= 0 && o.y() <= height() ) {
			int size = 4*pixelScale[ZoomLevel]/pixelScale[0];
			int x1 = o.x() - size/2;
			int y1 = o.y() - size/2;
			if ( ZoomLevel>2 && !ksw->GetData()->Mars->image.isNull() ) {
				ScaledImage = ksw->GetData()->Mars->image.smoothScale( size, size );
				psky.drawImage( x1, y1, ScaledImage );
			} else {
				psky.drawEllipse( x1, y1, size, size );
			}
		}
	}	
			
	//Draw Jupiter
  if ( ksw->GetOptions()->drawJupiter ) {
		psky.setPen( QColor( "Goldenrod" ) );
		psky.setBrush( QColor( "Goldenrod" ) );
		QPoint o = getXY( ksw->GetData()->Jupiter->pos(), ksw->GetOptions()->useAltAz, LSTh, ksw->geo->lat() );
		if ( o.x() >= 0 && o.x() <= width() && o.y() >= 0 && o.y() <= height() ) {
			int size = 4*pixelScale[ZoomLevel]/pixelScale[0];
			int x1 = o.x() - size/2;
			int y1 = o.y() - size/2;
			if ( ZoomLevel>2 && !ksw->GetData()->Jupiter->image.isNull() ) {
				ScaledImage = ksw->GetData()->Jupiter->image.smoothScale( size, size );
				psky.drawImage( x1, y1, ScaledImage );
			} else {
				psky.drawEllipse( x1, y1, size, size );
			}
		}
	}

	//Draw Saturn
  if ( ksw->GetOptions()->drawSaturn ) {
		psky.setPen( QColor( "LightYellow2" ) );
		psky.setBrush( QColor( "LightYellow2" ) );
		QPoint o = getXY( ksw->GetData()->Saturn->pos(), ksw->GetOptions()->useAltAz, LSTh, ksw->geo->lat() );
		if ( o.x() >= 0 && o.x() <= width() && o.y() >= 0 && o.y() <= height() ) {
			int size = 4*pixelScale[ZoomLevel]/pixelScale[0];
			int x1 = o.x() - size/2;
			int y1 = o.y() - size/2;
			if ( ZoomLevel>2 && !ksw->GetData()->Saturn->image.isNull() ) {
				ScaledImage = ksw->GetData()->Saturn->image.smoothScale( size, size );
				psky.drawImage( x1, y1, ScaledImage );
			} else {
				psky.drawEllipse( x1, y1, size, size );
			}
		}
	}	
		
	//Draw Uranus
  if ( ksw->GetOptions()->drawUranus ) {
		psky.setPen( QColor( "LightSeaGreen" ) );
		psky.setBrush( QColor( "LightSeaGreen" ) );
		QPoint o = getXY( ksw->GetData()->Uranus->pos(), ksw->GetOptions()->useAltAz, LSTh, ksw->geo->lat() );
		if ( o.x() >= 0 && o.x() <= width() && o.y() >= 0 && o.y() <= height() ) {
			int size = 4*pixelScale[ZoomLevel]/pixelScale[0];
			int x1 = o.x() - size/2;
			int y1 = o.y() - size/2;
			if ( ZoomLevel>2 && !ksw->GetData()->Uranus->image.isNull() ) {
				ScaledImage = ksw->GetData()->Uranus->image.smoothScale( size, size );
				psky.drawImage( x1, y1, ScaledImage );
			} else {
				psky.drawEllipse( x1, y1, size, size );
			}
		}
	}	
		
	//Draw Neptune
  if ( ksw->GetOptions()->drawNeptune ) {
		psky.setPen( QColor( "SkyBlue" ) );
		psky.setBrush( QColor( "SkyBlue" ) );
		QPoint o = getXY( ksw->GetData()->Neptune->pos(), ksw->GetOptions()->useAltAz, LSTh, ksw->geo->lat() );
		if ( o.x() >= 0 && o.x() <= width() && o.y() >= 0 && o.y() <= height() ) {
			int size = 4*pixelScale[ZoomLevel]/pixelScale[0];
			int x1 = o.x() - size/2;
			int y1 = o.y() - size/2;
			if ( ZoomLevel>2 && !ksw->GetData()->Neptune->image.isNull() ) {
				ScaledImage = ksw->GetData()->Neptune->image.smoothScale( size, size );
				psky.drawImage( x1, y1, ScaledImage );
			} else {
				psky.drawEllipse( x1, y1, size, size );
			}
		}
	}		
	
	//Draw Horizon

	QPointArray pts;
	QList<QPoint> points;
	int ptsCount = 0;

	if (ksw->GetOptions()->drawHorizon || ksw->GetOptions()->drawGround) {
		psky.setPen( QColor ( ksw->GetOptions()->colorHorz ) );

		psky.setBrush( QColor ( ksw->GetOptions()->colorHorz ) );
		pts.resize( 720 );

		for ( unsigned int i=0; i<ksw->GetData()->Horizon.count(); ++i ) {
			QPoint *o = new QPoint();
			*o = getXY( ksw->GetData()->Horizon.at(i), ksw->GetOptions()->useAltAz, LSTh, ksw->geo->lat() );
      bool found = false;

			//Use the QList of points to pre-sort visible horizon points
			if ( o->x() > -pixelScale[ZoomLevel]/4 && o->x() < width() + pixelScale[ZoomLevel]/4 ) {

				if ( ksw->GetOptions()->useAltAz ) {
	        unsigned int j;
					for ( j=0; j<points.count(); ++j ) {
						if ( ksw->GetOptions()->useAltAz && o->x() < points.at(j)->x() ) {
							found = true;
							break;
						}
					}
					if ( found ) {
						points.insert( j, o );
					} else {
						points.append( o );
					}
				} else {
					points.append( o );
				}
			}
		}

//	Fill the pts array with sorted horizon points, Draw Horizon Line
		pts.setPoint( 0, points.at(0)->x(), points.at(0)->y() );
		if ( ksw->GetOptions()->drawHorizon ) psky.moveTo( points.at(0)->x(), points.at(0)->y() );

		for ( unsigned int i=1; i<points.count(); ++i ) {
			pts.setPoint( i, points.at(i)->x(), points.at(i)->y() );
			if ( ksw->GetOptions()->drawHorizon ) {
				if ( !ksw->GetOptions()->useAltAz && ( abs( points.at(i)->x() - psky.pos().x() ) > 250 ||
							abs( points.at(i)->y() - psky.pos().y() ) > 250 ) ) {
					psky.moveTo( points.at(i)->x(), points.at(i)->y() );
				} else {
					psky.lineTo( points.at(i)->x(), points.at(i)->y() );
				}
			}
		}

  }

//  Finish the polygon by adding a square bottom edge, offscreen
	if ( ksw->GetOptions()->useAltAz && ksw->GetOptions()->drawGround ) {
		ptsCount = points.count();
		pts.setPoint( ptsCount++, width()+10, height()+10 );
		pts.setPoint( ptsCount++, -10, height()+10 );
		pts.resize( ptsCount );

		psky.drawPolygon( pts );

//  remove all items in points list

		for ( unsigned int i=0; i<points.count(); ++i ) {
			points.remove(i);
		}

//	Draw compass heading labels along horizon
		SkyPoint *c = new SkyPoint;
		QPoint cpoint;
		psky.setPen( QColor ( ksw->GetOptions()->colorSky ) );
		psky.setFont( stdFont );
		
		//North
		c->setAz( 359.99 );
		c->setAlt( 0.0 );
		if ( !ksw->GetOptions()->useAltAz ) c->AltAzToRADec( LSTh, ksw->geo->lat() );
		cpoint = getXY( c, ksw->GetOptions()->useAltAz, LSTh, ksw->geo->lat() );
		cpoint.setY( cpoint.y() + 20 );
		if (cpoint.x() > 0 && cpoint.x() < width() && cpoint.y() > 0 && cpoint.y() < height() ) {
			psky.drawText( cpoint.x(), cpoint.y(), i18n( "N" ) );
		}

		//NorthEast
		c->setAz( 45.0 );
		c->setAlt( 0.0 );
		if ( !ksw->GetOptions()->useAltAz ) c->AltAzToRADec( LSTh, ksw->geo->lat() );
		cpoint = getXY( c, ksw->GetOptions()->useAltAz, LSTh, ksw->geo->lat() );
		cpoint.setY( cpoint.y() + 20 );
		if (cpoint.x() > 0 && cpoint.x() < width() && cpoint.y() > 0 && cpoint.y() < height() ) {
			psky.drawText( cpoint.x(), cpoint.y(), i18n( "NE" ) );
		}

		//East
		c->setAz( 90.0 );
		c->setAlt( 0.0 );
		if ( !ksw->GetOptions()->useAltAz ) c->AltAzToRADec( LSTh, ksw->geo->lat() );
		cpoint = getXY( c, ksw->GetOptions()->useAltAz, LSTh, ksw->geo->lat() );
		cpoint.setY( cpoint.y() + 20 );
		if (cpoint.x() > 0 && cpoint.x() < width() && cpoint.y() > 0 && cpoint.y() < height() ) {
			psky.drawText( cpoint.x(), cpoint.y(), i18n( "E" ) );
		}

		//SouthEast
		c->setAz( 135.0 );
		c->setAlt( 0.0 );
		if ( !ksw->GetOptions()->useAltAz ) c->AltAzToRADec( LSTh, ksw->geo->lat() );
		cpoint = getXY( c, ksw->GetOptions()->useAltAz, LSTh, ksw->geo->lat() );
		cpoint.setY( cpoint.y() + 20 );
		if (cpoint.x() > 0 && cpoint.x() < width() && cpoint.y() > 0 && cpoint.y() < height() ) {
			psky.drawText( cpoint.x(), cpoint.y(), i18n( "SE" ) );
		}

		//South
		c->setAz( 179.99 );
		c->setAlt( 0.0 );
		if ( !ksw->GetOptions()->useAltAz ) c->AltAzToRADec( LSTh, ksw->geo->lat() );
		cpoint = getXY( c, ksw->GetOptions()->useAltAz, LSTh, ksw->geo->lat() );
		cpoint.setY( cpoint.y() + 20 );
		if (cpoint.x() > 0 && cpoint.x() < width() && cpoint.y() > 0 && cpoint.y() < height() ) {
			psky.drawText( cpoint.x(), cpoint.y(), i18n( "S" ) );
		}

		//SouthWest
		c->setAz( 225.0 );
		c->setAlt( 0.0 );
		if ( !ksw->GetOptions()->useAltAz ) c->AltAzToRADec( LSTh, ksw->geo->lat() );
		cpoint = getXY( c, ksw->GetOptions()->useAltAz, LSTh, ksw->geo->lat() );
		cpoint.setY( cpoint.y() + 20 );
		if (cpoint.x() > 0 && cpoint.x() < width() && cpoint.y() > 0 && cpoint.y() < height() ) {
			psky.drawText( cpoint.x(), cpoint.y(), i18n( "SW" ) );
		}

		//West
		c->setAz( 270.0 );
		c->setAlt( 0.0 );
		if ( !ksw->GetOptions()->useAltAz ) c->AltAzToRADec( LSTh, ksw->geo->lat() );
		cpoint = getXY( c, ksw->GetOptions()->useAltAz, LSTh, ksw->geo->lat() );
		cpoint.setY( cpoint.y() + 20 );
		if (cpoint.x() > 0 && cpoint.x() < width() && cpoint.y() > 0 && cpoint.y() < height() ) {
			psky.drawText( cpoint.x(), cpoint.y(), i18n( "W" ) );
		}

		//NorthWest
		c->setAz( 315.0 );
		c->setAlt( 0.0 );
		if ( !ksw->GetOptions()->useAltAz ) c->AltAzToRADec( LSTh, ksw->geo->lat() );
		cpoint = getXY( c, ksw->GetOptions()->useAltAz, LSTh, ksw->geo->lat() );
		cpoint.setY( cpoint.y() + 20 );
		if (cpoint.x() > 0 && cpoint.x() < width() && cpoint.y() > 0 && cpoint.y() < height() ) {
			psky.drawText( cpoint.x(), cpoint.y(), i18n( "NW" ) );
		}

		delete c;
	}

	psky.end();
	bitBlt( this, 0, 0, sky );
	computeSkymap = false;	// use Update() to compute new skymap else old pixmap will be shown
}

void SkyMap::drawSymbol( QPainter &psky, int type, int x, int y, int size ) {
	int x1 = x - size/2;
	int x2 = x + size/2;
	int y1 = y - size/2;
	int y2 = y + size/2;

	int xa = x - size/4;
	int xb = x + size/4;
	int ya = y - size/4;
	int yb = y + size/4;

	switch (type) {
		case 0: //star
		case 1: //double star
			if (size > 1 ) {
				psky.drawEllipse( x1, y1, size, size );
			} else if (size==1) {
				psky.drawPoint( x, y );
			}
			break;
		case 2: //Planet
			break;
		case 3: //Open cluster
			psky.drawEllipse( xa, y1, 2, 2 ); // draw circle of points
			psky.drawEllipse( xb, y1, 2, 2 );
			psky.drawEllipse( xa, y2, 2, 2 );
			psky.drawEllipse( xb, y2, 2, 2 );
			psky.drawEllipse( x1, ya, 2, 2 );
			psky.drawEllipse( x1, yb, 2, 2 );
			psky.drawEllipse( x2, ya, 2, 2 );
			psky.drawEllipse( x2, yb, 2, 2 );
			break;
		case 4: //Globular Cluster
			if (size<2) size = 2;
			psky.drawEllipse( x1, y1, size, size );
			psky.moveTo( x, y1 );
			psky.lineTo( x, y2 );
			psky.moveTo( x1, y );
			psky.lineTo( x2, y );
			break;
		case 5: //Gaseous Nebula
			if (size <2) size = 2;
			psky.drawLine( x1, y1, x2, y1 );
			psky.drawLine( x2, y1, x2, y2 );
			psky.drawLine( x2, y2, x1, y2 );
			psky.drawLine( x1, y2, x1, y1 );
			break;
		case 6: //Planetary Nebula
			if (size<2) size = 2;
			psky.drawEllipse( x1, y1, size, size );
			psky.moveTo( x, y1 );
			psky.lineTo( x, y1 - size/2 );
			psky.moveTo( x, y2 );
			psky.lineTo( x, y2 + size/2 );
			psky.moveTo( x1, y );
			psky.lineTo( x1 - size/2, y );
			psky.moveTo( x2, y );
			psky.lineTo( x2 + size/2, y );
			break;
		case 7: //Supernova remnant
			if (size<2) size = 2;
			psky.moveTo( x, y1 );
			psky.lineTo( x2, y );
			psky.lineTo( x, y2 );
			psky.lineTo( x1, y );
			psky.lineTo( x, y1 );
			break;
		case 8: //Galaxy
			if ( size>2 ) {
				psky.drawEllipse( x-size, y1, 2*size, size );
			} else {
				psky.drawPoint( x, y );
			}
			break;
	}
}
//---------------------------------------------------------------------------

QPoint SkyMap::getXY( SkyPoint *o, bool Horiz, dms LSTh, dms lat ) {
	QPoint p;
	dms X, Y, X0, dX;
	double sindX, cosdX, sinY, cosY, sinY0, cosY0;

	if ( Horiz ) {
//		focus.RADecToAltAz( LSTh, lat );
//		o->RADecToAltAz( LSTh, lat );
		X0 = focus.getAz();

		X = o->getAz();
		Y = o->getAlt();

		if ( X0.getD() > 270.0 && X.getD() < 90.0 ) {
			dX = 360.0 + X0.getD() - X.getD();
		} else {
			dX = X0.getD() - X.getD();
		}
		
		focus.getAlt().SinCos( sinY0, cosY0 );

  } else {
		if (focus.getRA().getH() > 18.0 && o->getRA().getH() < 6.0) {
			dX = o->getRA().getD() + 360.0 - focus.getRA().getD();
		} else {
			dX = o->getRA().getD() - focus.getRA().getD();
	  }
    Y = o->getDec().getD();
		focus.getDec().SinCos( sinY0, cosY0 );
  }

	//Convert dX, Y coords to screen pixel coords.
	dX.SinCos( sindX, cosdX );
	Y.SinCos( sinY, cosY );
	
	double c = sinY0*sinY + cosY0*cosY*cosdX;

	if ( c < 0.0 ) { //Object is on "back side" of the celestial sphere; don't plot it.
		p.setX( -100000 );
		p.setY( -100000 );
		return p;
	}

	double k = sqrt( 2.0/( 1 + c ) );

	p.setX( int( 0.5*width()  - pixelScale[ZoomLevel]*k*cosY*sindX ) );
	p.setY( int( 0.5*height() - pixelScale[ZoomLevel]*k*( cosY0*sinY - sinY0*cosY*cosdX ) ) );

	return p;
}
//---------------------------------------------------------------------------

SkyPoint SkyMap::dXdYToRaDec( double dx, double dy, bool useAltAz, dms LSTh, dms lat ) {
	//Determine RA and Dec of a point, given (dx, dy): it's pixel
	//coordinates in the SkyMap with the center of the map as the origin.
	
	SkyPoint MousePoint;
	double sinDec, cosDec, sinDec0, cosDec0, sinc, cosc, sinlat, coslat;
	double xx, yy;

	double r  = sqrt( dx*dx + dy*dy );
	dms centerAngle;
	centerAngle.setRadians( 2.0*asin(0.5*r) );
	
	focus.getDec().SinCos( sinDec0, cosDec0 );
	centerAngle.SinCos( sinc, cosc );

	if ( useAltAz ) {
		dms HA;
		dms Dec, alt, az, alt0, az0;
		double A;
		double sinAlt, cosAlt, sinAlt0, cosAlt0, sinAz, cosAz;
//		double HA0 = LSTh - focus.getRA();
		az0 = focus.getAz();
		alt0 = focus.getAlt();
		alt0.SinCos( sinAlt0, cosAlt0 );

		dx = -dx; //Flip East-west (Az goes in opposite direction of RA)
		yy = dx*sinc;
		xx = r*cosAlt0*cosc - dy*sinAlt0*sinc;
		
		A = atan( yy/xx );
		//resolve ambiguity of atan():
		if ( xx<0 ) A = A + PI();
//		if ( xx>0 && yy<0 ) A = A + 2.0*PI();

		dms deltaAz;
		deltaAz.setRadians( A );
		az = focus.getAz().getD() + deltaAz.getD();
		alt.setRadians( asin( cosc*sinAlt0 + ( dy*sinc*cosAlt0 )/r ) );
		
		az.SinCos( sinAz, cosAz );
		alt.SinCos( sinAlt, cosAlt );
		lat.SinCos( sinlat, coslat );

		Dec.setRadians( asin( sinAlt*sinlat + cosAlt*coslat*cosAz ) );
		Dec.SinCos( sinDec, cosDec );
		
		HA.setRadians( acos( ( sinAlt - sinlat*sinDec )/( coslat*cosDec ) ) );
		if ( sinAz > 0.0 ) HA.setH( 24.0 - HA.getH() );
		
		MousePoint.setRA( LSTh.getH() - HA.getH() );
		MousePoint.setRA( MousePoint.getRA().reduce() );
		MousePoint.setDec( Dec.getD() );

		return MousePoint;

  } else {
		yy = dx*sinc;
		xx = r*cosDec0*cosc - dy*sinDec0*sinc;
		
		double RARad = ( atan( yy / xx ) );
		//resolve ambiguity of atan():
		if ( xx<0 ) RARad = RARad + PI();
//		if ( xx>0 && yy<0 ) RARad = RARad + 2.0*PI();
		
		dms deltaRA, Dec;
		deltaRA.setRadians( RARad );
		Dec.setRadians( asin( cosc*sinDec0 + (dy*sinc*cosDec0)/r ) );
		
		MousePoint.setRA( focus.getRA().getH() + deltaRA.getH() );
		MousePoint.setRA( MousePoint.getRA().reduce() );
		MousePoint.setDec( Dec.getD() );

		return MousePoint;
	}
}
//---------------------------------------------------------------------------

void SkyMap::setUpDownloads()
{
	downloads++;
}

void SkyMap::setDownDownloads()
{
	downloads--;
}

// force new compute of the skymap (used instead of update())
void SkyMap::Update()
{
	computeSkymap = true;
	update();
}

void SkyMap::resizeEvent( QResizeEvent * e)
{
    computeSkymap = true;	// skymap must be new computed
    if ( testWState(WState_AutoMask) )
		updateMask();
}

bool SkyMap::checkVisibility( SkyPoint *p, float FOV, bool useAltAz, bool isPoleVisible ) {
	double dX, dY;

	if ( useAltAz ) {
		dY = fabs( p->getAlt().getD() - focus.getAlt().getD() );
	} else {
		dY = fabs( p->getDec().getD() - focus.getDec().getD() );
	}
	if ( dY > FOV ) return false;

	if ( useAltAz ) {
		dX = fabs( p->getAz().getD() - focus.getAz().getD() );
	} else {
		dX = fabs( p->getRA().getD() - focus.getRA().getD() );
	}
	if ( dX > 180.0 ) dX = 360.0 - dX; // take shorter distance around sky
	if ( isPoleVisible || dX < 1.2*FOV/cos( focus.getDec().radians() ) ) {
		return true;
	} else {
		return false;
	}
}
