/***************************************************************************
                          kssun.cpp  -  K Desktop Planetarium
                             -------------------
    begin                : Sun Jul 22 2001
    copyright            : (C) 2001 by Jason Harris
    email                : jharris@30doradus.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <klocale.h>
#include <kstars.h>
#include <math.h>
#include "kssun.h"

KSSun::KSSun() {
	KSSun( J2000 );
}

KSSun::KSSun( long double Epoch ) : KSPlanet( i18n( "Sun" ) ) {
	JD0 = 2447892.5; //Jan 1, 1990
	eclong0 = 279.403303; //mean ecliptic longitude at JD0
	plong0 = 282.768422; //longitude of sun at perigee for JD0
	e0 = 0.016713; //eccentricity of Earth's orbit at JD0
	findPosition( Epoch );
}

KSSun::~KSSun(){
}

void KSSun::findPosition( long double jd ) {
	int DaysSinceEpoch = int( jd - JD0 );

	//Calculate Mean Anomaly
	double dMeanAnomaly = DaysSinceEpoch*360.0/365.242191 + eclong0 - plong0;
	while ( dMeanAnomaly < 0.0 ) { dMeanAnomaly += 360.0; }
	while ( dMeanAnomaly > 360.0 ) { dMeanAnomaly -= 360.0; }
	dms MeanAnomaly = dMeanAnomaly;

	//Find iterative numerical solution for the Eccentric Anomaly
	double dE;
	double E = MeanAnomaly.radians(); //first guess, in Radians
	double delta = 1.0;
	
	while ( fabs( delta ) > 1e-6 ) {
		delta = E - e0*sin( E ) - MeanAnomaly.radians();
		dE = delta/( 1 - e0*cos( E ) );
		E = E - dE;
	}
	
	//Find the True Anomaly (still in radians)
	//tan( TA/2 ) = sqrt[(1+e0)/(1-e0)]*( tan( E/2 ) )
	double TanHalfTrueAnomaly = sqrt((1.0+e0)/(1.0-e0))*tan( 0.5*E );
	dms TrueAnomaly;
	TrueAnomaly.setRadians( 2.0*atan( TanHalfTrueAnomaly ) );
	double dEcLong = TrueAnomaly.getD() + plong0; //ecliptic longitude, in deg
	while ( dEcLong < 0.0 ) { dEcLong += 360.0; }
	while ( dEcLong > 360.0 ) { dEcLong -= 360.0; }
	EcLong.setD( dEcLong ); // convert to deg/min/sec
	EcLat.setD( 0.0 ); //The sun always has Ecliptic Latitude=0.0, by definition.
	
	//Finally, convert Ecliptic coords to Ra, Dec.  Ecliptic latitude is zero, by definition
	EclipticToEquatorial( jd );	
}
