/***************************************************************************
                          setupdlg.cpp  -  The KountDown setup dialog
                             -------------------
    begin                : Mon Apr 15 2002
    copyright            : (C) 2002 by Erik Johansson
    email                : kountdown@erre.user.lysator.liu.se
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// Qt includes
#include <qcheckbox.h>
#include <qbuttongroup.h>
#include <qtabwidget.h>
#include <qfont.h>
#include <qdatetime.h>
#include <qmultilineedit.h>
#include <qslider.h>
#include <qspinbox.h>
#include <qstring.h>
#include <qvbox.h>
#include <qcombobox.h>

// KDE includes
#include <kcolorbutton.h>
#include <kconfig.h>
#include <kglobal.h>
#include <klocale.h>
#include <klineedit.h>
#include <kfontdialog.h>
#include <kmessagebox.h>
#include <kiconloader.h>
#include <kaboutapplication.h>
#include <kaboutdata.h>

// Local includes
#include "setupdlg.h"
#include "setupstart.h"
#include "setuphalf.h"
#include "setupfinish.h"
#include "timewidget.h"

/*
 * Set up the setup dialog by creating a KDialogBase->IconList and adding the widgets start, half and finish.
 * They are auto generated by uic and the setup*.ui files.
 */
SetupDlg::SetupDlg() :
KDialogBase( IconList, i18n("Setup the Kountdown screensaver"), Ok|Apply|Cancel|Default, Ok, 0, "setupdlg", true, false )
{
	setButtonText( Default, i18n( "About" ) );
	QVBox *startLayout = addVBoxPage( i18n("Start"), i18n("Start settings"), BarIcon("cookie", KIcon::SizeMedium) );
	start = new SetupStart( startLayout );
	connect( start->startFont, SIGNAL(clicked()), this, SLOT(slotFont()) );
	
	QVBox *halfLayout = addVBoxPage( i18n("Halfway"), i18n("Halfway settings"), BarIcon("misc", KIcon::SizeMedium) );
	half = new SetupHalf( halfLayout );
	connect( half->halfFont, SIGNAL(clicked()), this, SLOT(slotFont()) );
	
	QVBox *finishLayout = addVBoxPage( i18n("Finish"), i18n("Finish settings"), BarIcon("exit", KIcon::SizeMedium) );
	finish = new SetupFinish( finishLayout );		
	connect( finish->finFont, SIGNAL(clicked()), this, SLOT(slotFont()) );
	
	readSettings();
}

SetupDlg::~SetupDlg()
{
	delete start;
	start = 0;
	delete half;
	half = 0;
	delete finish;
	finish = 0;
}

// ReadCommonSettings macro
#define RCS(X) \
	X##->displayText->setText( config->readEntry( "Text", i18n( "It's now %D days and %S seconds left to %F - %T" ) ) ); \
	X##->backCol->setColor( config->readColorEntry( "BackCol", &Qt::black ) ); \
	X##->fontCol->setColor( config->readColorEntry( "FontCol", &Qt::red ) ); \
	X##->backPattern->setCurrentItem( config->readNumEntry( "BackPattern", 0) ); \
	X##->rndBackCol->setChecked( config->readBoolEntry( "RandomBackCol", false ) ); \
	X##->rndFontCol->setChecked( config->readBoolEntry( "RandomFontCol", false ) ); \
	X##->rndFont->setChecked( config->readBoolEntry( "RandomFont", false ) ); \
	X##->effects->setButton( config->readNumEntry( "Text effect", 0 ) ); \
	X##->speed->setValue( config->readNumEntry( "Speed", 50 ) );

// WriteCommonSettings macro
#define WCS(X) \
	config->writeEntry( "Text", X##->displayText->text() ); \
	config->writeEntry( "BackCol", X##->backCol->color() ); \
	config->writeEntry( "FontCol", X##->fontCol->color() ); \
	config->writeEntry( "BackPattern", X##->backPattern->currentItem() ); \
	config->writeEntry( "RandomBackCol", X##->rndBackCol->isChecked() ); \
	config->writeEntry( "RandomFontCol", X##->rndFontCol->isChecked() ); \
	config->writeEntry( "RandomFont", X##->rndFont->isChecked() ); \
	config->writeEntry( "Text effect", X##->effects->id( X##->effects->selected() ) ); \
	config->writeEntry( "Speed", X##->speed->value() );

// ReadFormat macro
#define RFORMAT(X) \
	format = config->readNumEntry( "Display", 9 ); \
	X##->years->setChecked( calcFormat( format, YEARS ) ); \
	X##->months->setChecked( calcFormat( format, MONTHS ) ); \
	X##->weeks->setChecked( calcFormat( format, WEEKS ) ); \
	X##->days->setChecked( calcFormat( format, DAYS ) ); \
	X##->hours->setChecked( calcFormat( format, HOURS ) ); \
	X##->min->setChecked( calcFormat( format, MINS ) ); \
	X##->sec->setChecked( calcFormat( format, SECS ) ); \

// WriteFormat macro
#define WFORMAT(X) \
	config->writeEntry( "Display", calcFormat( \
		X##->years->isChecked(), X##->months->isChecked(), X##->weeks->isChecked(), \
		X##->days->isChecked(), X##->hours->isChecked(), X##->min->isChecked(), \
		X##->sec->isChecked() ) );

void SetupDlg::readSettings()
{
	int format;
	KConfig *config = KGlobal::config();

	// Setup Start tab
	config->setGroup( "Start settings" );
		font[0] = config->readFontEntry("Font");
		RCS(start)
		RFORMAT(start)

	// Setup Half way tab
	config->setGroup( "HalfWay settings" );
		font[1] = config->readFontEntry("Font");
		RCS(half)
		RFORMAT(half)
		
	// Setup Finish tab
	config->setGroup("Finish settings");
		font[2] = config->readFontEntry("Font");
		RCS(finish)

	// Setup Global settings
	config->setGroup( "Global settings" );
		if ( config->readBoolEntry( "UseHalfWay", true ) )
			half->halfOn->setChecked( true );
		else
			half->halfTabs->setEnabled( false );
		QDateTime temp = QDateTime::currentDateTime().addDays(10);
		QDateTime fin = config->readDateTimeEntry( "FinishDateTime", &temp );
		temp = QDateTime::currentDateTime().addDays(5);
		QDateTime halfw = config->readDateTimeEntry( "HalfDateTime", &temp );

		ULONG y, o, w, d, h, m, s;		
		calcHalfTime	( halfw, fin, y, o, w, d, h, m, s );

		finish->finDate->setDateTime( fin );
		half->switchYears->setValue( y );
		half->switchMonths->setValue( o );
		half->switchWeeks->setValue( w );
		half->switchDays->setValue( d );
		half->switchHours->setValue( h );
		half->switchMin->setValue( m );
		half->switchSec->setValue( s );
}

void SetupDlg::slotOk()
{
	closeDialog = true;
	slotApply();
	if ( closeDialog )
		accept();
}

void SetupDlg::slotApply()
{
	// Make sure the QSpinBox loses focus, or theire value won't be updated
	if ( focusWidget() )
		focusWidget()->clearFocus();

	// Date should be in the future, if not complain.
	if ( finish->finDate->getDateTime() < QDateTime::currentDateTime() )
	{
		int ans = KMessageBox::warningYesNoCancel( this, i18n( "The finish date you has entered is in the past.\n"
			"Do you want me to set it to %1" ).arg(
			QDateTime( QDate::currentDate().addDays( 1 ), finish->finDate->getDateTime().time() ).toString() ),
			i18n( "Selected date not in the future." ) );

		if ( ans == KMessageBox::Yes )
				finish->finDate->setDateTime( QDateTime( QDate::currentDate().addDays( 1 ), finish->finDate->getDateTime().time() ) );
		if ( ans == KMessageBox::Cancel )
			closeDialog = false;
	}
	KConfig *config = KGlobal::config();

	// Save Start tab
	config->setGroup( "Start settings" );
		config->writeEntry( "Font", font[0] );
		WCS(start)
		WFORMAT(start)

	// Save half way tab
	config->setGroup( "HalfWay settings" );
		config->writeEntry( "Font", font[1] );
		WCS(half)
		WFORMAT(half)
		
	// Save finish tab
	config->setGroup( "Finish settings" );
		config->writeEntry( "Font", font[2] );
		WCS(finish)
		
	// Save global settings
	config->setGroup( "Global settings" );
		config->writeEntry( "FinishDateTime", finish->finDate->getDateTime() );
		config->writeEntry( "HalfDateTime", calcHalfTime(
			finish->finDate->getDateTime(), half->switchYears->value(), half->switchMonths->value(),
			half->switchWeeks->value(), half->switchDays->value(), half->switchHours->value(),
			half->switchMin->value(), half->switchSec->value() ) );
		config->writeEntry( "UseHalfWay", half->halfOn->isChecked() );

	config->sync();
}

void SetupDlg::slotDefault()
{
  KAboutData aboutData( "kountdown.kss", "Kountdown", VERSION,
    I18N_NOOP( "A screensaver that counts down\nto selected date" ), KAboutData::License_GPL_V2,
    "(c) 2002-2003, Erik Johansson", 0, "http://www.lysator.liu.se/~erre/", "kountdown@erre.user.lysator.liu.se");
  aboutData.addAuthor("Erik Johansson", "Author and maintainer", "kountdown@erre.user.lysator.liu.se", "http://www.lysator.liu.se/~erre/" );
  aboutData.addCredit( "Florian Hacker", "Suggestions and bugfixes", 0, 0 );
  aboutData.addCredit( "Thomas Kosch", "Found a nasty bug", 0, 0 );
  KAboutApplication about(&aboutData);
  about.exec();
}

void SetupDlg::slotFont()
{
	KFontDialog::getFont( font[activePageIndex()] );
}

// Return the halfWay date. Called when asked to save settings.
QDateTime SetupDlg::calcHalfTime( const QDateTime fin, const ULONG y, const ULONG o, const ULONG w,
	const ULONG d, const ULONG h, const ULONG m, const ULONG s )
{
	ULONG temp;
	temp = (y * 365) + (o * 30) + (w * 7) + d;
	QDateTime half = fin.addDays( -temp );
	temp = (h * 3600) + (m * 60) + s;
	return half.addSecs( -temp );
}

// Split the halfway date in years, months, weeks, days, hours, minutes and seconds and set the vars.
// Called when asked to load settings.
void SetupDlg::calcHalfTime( const QDateTime half, const QDateTime fin, ULONG &y, ULONG &o, ULONG &w,
	ULONG &d, ULONG &h, ULONG &m, ULONG &s )
{
	ULONG temp;
	if ( half.time() <= fin.time() )
		temp = half.daysTo(fin);
	else
		temp = ( half.daysTo(fin) - 1 >= 0 ) ? half.daysTo(fin) - 1 : 0;
	ULONG addDays = temp;
	
	y = (temp - (temp % 365)) / 365;
	temp = temp - (y * 365);
	o = (temp - (temp % 30)) / 30;
	temp = temp - (o * 30);
	w = (temp - (temp % 7)) / 7;
	temp = temp - (w * 7);
	d = temp;
	QDateTime calc = half.addDays( addDays );
	
	temp = calc.secsTo(fin);
	h = (temp - (temp % 3600)) / 3600;
	temp = temp - (h * 3600);
	m = (temp - (temp % 60)) / 60;
	s = temp - (m * 60);
}

// Calc the format i.e. what to display
// Called when asked to save settings.
int SetupDlg::calcFormat( bool y, bool m, bool w, bool d, bool h, bool min, bool s )
{
	int format = 0;
	if ( y )
		format += 64;
	if ( m )
		format += 32;
	if ( w )
		format += 16;
	if ( d )
		format += 8;
	if ( h )
		format += 4;
	if ( min )
		format += 2;
	if ( s )
		format += 1;
	return format;
}

// What format do we want. Called when asked to load settings.
bool SetupDlg::calcFormat( int &format, NAME name )
{
	switch ( name ) {
	case YEARS:
		if ( format >= 64 ) {
			format -= 64;
			return true;
		}
		break;
	case MONTHS:
		if ( format >= 32 ) {
			format -= 32;
			return true;
		}
		break;
	case WEEKS:
		if ( format >= 16 ) {
			format -= 16;
			return true;
		}
		break;
	case DAYS:
		if ( format >= 8 ) {
			format -= 8;
			return true;
		}
		break;
	case HOURS:
		if ( format >= 4 ) {
			format -= 4;
			return true;
		}
		break;
	case MINS:
		if ( format >= 2 ) {
			format -= 2;
			return true;
		}
		break;
	case SECS:
		if ( format >= 1 ) {
			format -= 1;
			return true;
		}
		break;
	}
	return false;
}

