/***************************************************************************
                          kdtimer.cpp  -  description
                             -------------------
    begin                : Tue Apr 2 2002
    copyright            : (C) 2002 by Erik Johansson
    email                : kountdown@erre.user.lysator.liu.se
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// Qt includes
#include <qdatetime.h>
#include <qtimer.h>

// Local includes
#include "kdtimer.h"

/*
* Takes finish date and finish format.
* Disables halfway.
*/
KDTimer::KDTimer( const QDateTime &fdate, short int ff )
{
	init( ff );
	*finishDate = fdate;
	calcFormat[FINISH] = ff;

	// No halfway date
	delete halfDate;
	halfDate = 0;
	useHalfWay = false;
	calcFormat[HALF] = ff;
}

KDTimer::KDTimer( const QDateTime &fdate, short int ff, const QDateTime &hdate, short int hf )
{
	init( ff, hf );
	*finishDate = fdate;
	calcFormat[FINISH] = ff;

	// Use halfway date and settings
	*halfDate = hdate;
	useHalfWay = true;
	calcFormat[HALF] = hf;
}

KDTimer::~KDTimer()
{
	stop();
	delete finishDate;
	finishDate = 0;
	delete halfDate;
	halfDate = 0;
	delete calcDateTime;
	calcDateTime = 0;
	delete updateTimer;
	updateTimer = 0;
}

/**
* KDTimer::init( *format, *format = 0 )
* Author: Erik Johansson 2002-04-03
* Init some member vars and get memory for the dates
* Make sure the format (ff and hf) is within boundaries (KD::NONE (0) - KD::ALL (127) ) else set it to KD::ALL
* hf are only checked if != 0
**/
void KDTimer::init( short ff, short hf )
{
	running = false;
	// Reserve some memory
	finishDate = new QDateTime();
	halfDate = new QDateTime();
	calcDateTime = new QDateTime();	
	
	updateTimer = new QTimer();
	connect( updateTimer, SIGNAL( timeout() ), this, SLOT( slotUpdateTimeLeft() ) );
	
	// Check boundaries
	if ( ff > KD::ALL || ff < KD::NIL )
		ff = KD::ALL;

	if ( hf != 0 ) {
		if ( hf > KD::ALL || hf < KD::NIL )
			hf = KD::ALL;
	}
	setUpFnPtrList( ff );
}

/*
* Set up fnPtrList by adding all functions that should be called to
* update timeLeft to it. Set fnPtrNr to the number of functions to call.
*/
void KDTimer::setUpFnPtrList( short format )
{
	fnPtrNr = 0;
	if ( format >= KD::YEARS ) {
		fnPtrList[fnPtrNr++] = &KDTimer::years;
		format -= KD::YEARS;
	}
	if ( format >= KD::MONTHS ) {
		fnPtrList[fnPtrNr++] = &KDTimer::months;
		format -= KD::MONTHS;
	}
	if ( format >= KD::WEEKS ) {
		fnPtrList[fnPtrNr++] = &KDTimer::weeks;
		format -= KD::WEEKS;
	}
	if ( format >= KD::DAYS ) {
		fnPtrList[fnPtrNr++] = &KDTimer::days;
		format -= KD::DAYS;
	}
	if ( format >= KD::HOURS ) {
		fnPtrList[fnPtrNr++] = &KDTimer::hours;
		format -= KD::HOURS;
	}
	if ( format >= KD::MINS ) {
		fnPtrList[fnPtrNr++] = &KDTimer::mins;
		format -= KD::MINS;
	}
	if ( format >= KD::SECS ) {
		fnPtrList[fnPtrNr++] = &KDTimer::secs;
		format -= KD::SECS;
	}
}

void KDTimer::start()
{
	if (!running) {
		running = true;
		calcFormat[CURRENT] = calcFormat[FINISH];
		slotUpdateTimeLeft();
	}
}

void KDTimer::stop()
{
	if (running) {
		emit timeLeftChanged(0);
		updateTimer->stop();
		running = false;
	}
}

void KDTimer::slotUpdateTimeLeft()
{
	unsigned int days = 0;
	*calcDateTime = QDateTime::currentDateTime();

	// If we have reached to or past finish
	if ( !timeLeft.reachedFinish && ( *calcDateTime >= *finishDate ) )
	{
		emit reachedMilestone(2);
		updateTimer->stop();
		timeLeft.setDateValue(0);
		timeLeft.setTimeValue(0);
		timeLeft.reachedFinish = true;
		emit timeLeftChanged( &timeLeft );
		return;
	}

	// If we have reached to or past half way	
	if ( useHalfWay && !timeLeft.reachedHalfWay && !timeLeft.reachedFinish && ( *calcDateTime >= *halfDate ) )
	{
		calcFormat[CURRENT] = calcFormat[HALF];
		setUpFnPtrList( (KD::TIME)calcFormat[HALF] );
		emit reachedMilestone(1);
		timeLeft.reachedHalfWay = true;
	}

	timeLeft.setDateValue( 0 );  // Reset
	timeLeft.setTimeValue( 0 );  // Reset
	
	for ( int i = 0; i < fnPtrNr - 1; i++ )
		(this->*fnPtrList[i])( KD::NONE, true );
	(this->*fnPtrList[fnPtrNr - 1])( KD::UP, true );
	emit timeLeftChanged( &timeLeft );

	if ( calcFormat[CURRENT] & KD::SECS || calcFormat[CURRENT] & KD::MINS ) {
		updateTimer->start( 1000, true );
		return;
	}
	
	if ( useHalfWay && !timeLeft.reachedHalfWay )
		days = QDateTime::currentDateTime().daysTo(*halfDate);
	else
		days = QDateTime::currentDateTime().daysTo(*finishDate);

	if ( days >= 1 ) 
		updateTimer->start( 60000, true ); // 1 min
	else
		updateTimer->start( 1000, true ); // 1 s
}

/*
* KDTimer::[years|months|...|secs]()
* Calculates the number of [years|...|secs] between calcDate and finishDate,
* rounds of according to round (KD::NONE or KD::UP) and then updates
* calcDate. If update is true, update timeLeft.[years|...|secs] and return 0.
* If false just return the number of [years|...|secs].
*
* Erik Johansson [updated: 2003-03-20]
*/
unsigned int KDTimer::years( KD::ROUND round, bool update )
{
	const QDate cDate = calcDateTime->date();
	const QDate fDate = finishDate->date();
	unsigned int years = 0;
	
	// Get the number of whole years left.
	if ( cDate.month() < fDate.month() )
		years = fDate.year() - cDate.year();
	else if ( cDate.month() > fDate.month() )
		years = ( (fDate.year() - cDate.year() - 1) >= 0 ) ? fDate.year() - cDate.year() - 1 : 0;
	else if ( cDate.month() == fDate.month() ) {
		if ( cDate.day() == fDate.day() ) {
			if ( calcDateTime->time() <= finishDate->time() )
				years = fDate.year() - cDate.year();
			else if ( calcDateTime->time() > finishDate->time()  )
				years = ( (fDate.year() - cDate.year() - 1) >= 0 ) ? fDate.year() - cDate.year() - 1 : 0;
		}
		else if ( cDate.day() < fDate.day() )
			years = fDate.year() - cDate.year();
		else if ( cDate.day() > fDate.day() )
			years = ( (fDate.year() - cDate.year() - 1) >= 0 ) ? fDate.year() - cDate.year() - 1 : 0;
	}

	// Update calcDateTime by adding years to it.
	*calcDateTime = calcDateTime->addYears( years );

	// Round of years according to round
	if ( round == KD::UP && calcDateTime->date() != finishDate->date() )
		years++;
	
	if ( update ) {
		timeLeft.years = years;
		return 0;
	}

	return years;
}

unsigned int KDTimer::months( KD::ROUND round, bool update )
{
	const QDate fDate = finishDate->date();
	unsigned int months = 0;
	unsigned int ymonths = 0;
	bool sameYear = false;
	
	// Have years been calculated?
	if ( calcFormat[CURRENT] < KD::YEARS )
		ymonths = 12 * years( KD::NONE, false);

	// This needs to be here, because calcDateTime may be updated.
	const QDate cDate = calcDateTime->date();

	if ( cDate.year() == fDate.year() )
		sameYear = true;
	
	// Calc whole months left
	if ( cDate.day() < fDate.day() ) {
		if ( sameYear )
			months = fDate.month() - cDate.month();
		else
			months = 12 - cDate.month() + fDate.month();
	}
	else if ( cDate.day() > fDate.day() ) {
		if ( sameYear )
			months = ( ( fDate.month() - cDate.month() - 1 ) >= 0 ) ? fDate.month() - cDate.month() - 1 : 0;
		else
			months = ( ( 11 - cDate.month() + fDate.month() ) >= 0 ) ? 11 - cDate.month() + fDate.month() : 0;
	}
	else if ( cDate.day() == fDate.day() ) {
		if ( calcDateTime->time() <= finishDate->time() ) {
			if ( sameYear )
				months = fDate.month() - cDate.month();
			else
				months = 12 - cDate.month() + fDate.month();
		}
		else if ( calcDateTime->time() > finishDate->time() ){
			if ( sameYear )
				months = ( ( fDate.month() - cDate.month() - 1 ) >= 0 ) ? fDate.month() - cDate.month() - 1 : 0;
			else
				months = ( ( 11 - cDate.month() + fDate.month() ) >= 0 ) ? 11 - cDate.month() + fDate.month() : 0;
		}
	}

	// Update calcDateTime
	*calcDateTime = calcDateTime->addMonths( months );

	// Add the "year-months"
	months += ymonths;

	// Round off months according to round
	if ( round == KD::UP && calcDateTime->date() != finishDate->date() )
		months++;

	if ( update ) {
		timeLeft.months = months;
		return 0;
	}
	return months;
}

unsigned int KDTimer::weeks( KD::ROUND round, bool update )
{
	unsigned int days = calcDateTime->daysTo( *finishDate );
	unsigned int weeks = 0;
	if ( calcDateTime->time() > finishDate->time() )
		days = ( days == 0 ) ? 0 : days - 1;

	weeks = days / 7;
	*calcDateTime = calcDateTime->addDays( weeks * 7 );
	
	if ( round == KD::UP && calcDateTime->date() != finishDate->date() )
		weeks++;

	if ( update ) {
		timeLeft.weeks = weeks;
		return 0;
	}
	return weeks;
}

unsigned int KDTimer::days( KD::ROUND round, bool update )
{
	unsigned int days = calcDateTime->daysTo( *finishDate );
	if ( calcDateTime->time() > finishDate->time() )
		days = ( days == 0 ) ? 0 : days - 1;
	
	*calcDateTime = calcDateTime->addDays( days );

	if ( round == KD::UP && calcDateTime->date() != finishDate->date() )
		days++;

	if ( update ) {
		timeLeft.days = days;
		return 0;
	}
	return days;
}

unsigned int KDTimer::hours( KD::ROUND round, bool update )
{
	unsigned int hours = 0;
	unsigned int day = 0;

	// We haven't calculated anything yet, get number of days
	if ( calcFormat[CURRENT] < KD::DAYS )
		day = days( KD::NONE, false );
	
	hours = calcDateTime->secsTo( *finishDate ) / 3600;

	// Add hours to calcDateTime	
	*calcDateTime = calcDateTime->addSecs( hours * 3600 );

	// Add "day-hours" to hours
	hours += 24 * day;

	if ( round == KD::UP && *calcDateTime != *finishDate )
		hours++;

	if ( update ) {
		timeLeft.hours = hours;
		return 0;
	}
	return hours;
}

unsigned int KDTimer::mins( KD::ROUND round, bool update )
{
	unsigned int day = 0;
	unsigned int mins = 0;

	// We haven't calculated anything yet, get number of days
	if ( calcFormat[CURRENT] < KD::HOURS )
		day = days( KD::NONE, false );

	mins = calcDateTime->secsTo( *finishDate ) / 60;
	
	*calcDateTime = calcDateTime->addSecs( mins * 60 );

	// 1440 = 24 * 60
	mins += day * 1440;
	
	if ( round == KD::UP && *calcDateTime != *finishDate )
		mins++;

	if ( update ) {
		timeLeft.mins = mins;
		return 0;
	}
	return mins;
}

unsigned int KDTimer::secs( KD::ROUND, bool )
{
	unsigned int day = 0;
	if ( calcFormat[CURRENT] == KD::SECS )
			day = days( KD::NONE, false );
	
	timeLeft.secs = calcDateTime->secsTo( *finishDate ) + (day * 24 * 3600);
	return 0;
}
