/*  This file is part of the KDE project
    Copyright (C) 2001 Simon Hausmann <hausmann@kde.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

// !! The history code is (c) David Faure <faure@kde.org> !!

#ifndef __view_h__
#define __view_h__

#include <qguardedptr.h>
#include <qobject.h>

#include <kurl.h>
#include <kparts/browserextension.h>

class HTMLView;
class Run;

namespace KIO
{
    class Job;
};

class BrowserInterface;

class View : public QObject
{
    Q_OBJECT
public:
    View( const QString &frameName, QWidget *parentWidget, const char *widgetName,
          QObject *parent, const char *name );

    virtual ~View();

    void openURL( const KURL &url, const QString &mimeType, const KParts::URLArgs &args );

    bool canGoBack() { return m_history.at() > 0; }

    bool canGoForward() { return m_history.at() != (int)(m_history.count() - 1); }

    QString caption() const { return m_caption; }

    QString locationBarURL() const { return m_locationBarURL; }

    bool isStopEnabled() const { return m_stop; }

    QString findText() const { return m_findText; }

    bool isFindEnabled() const { return m_find; }

    HTMLView *document() const { return m_doc; }

    QString frameName() const { return m_frameName; }

    QStringList frameNames() const;

    bool canHandleFrame( const QString &name, KParts::BrowserHostExtension **hostExtension );

    // same as 'internal' goHistory, but delayed action by a singleshot timer, in order not
    // to fuck up the caller stack if the history change deletes the caller.
    void goHistoryDelayed( int steps );

    uint historyLength() const { return m_history.count(); }

    static QStringList childFrameNames( KParts::ReadOnlyPart *part );

    static KParts::BrowserHostExtension *hostExtension( KParts::ReadOnlyPart *part, const QString &name );
public slots:
    void openURL( const KURL &url, const KParts::URLArgs &args = KParts::URLArgs() );

    void setFindEnabled( bool enable );

    void setFindText( const QString &text );

    void back();

    void forward();

    void reload();

    void find();

    void findNext();

    void runError();

    void stop();

    void started( KIO::Job *job );

    void completed();

signals:
    void openURLRequest( View *view, const KURL &url, const KParts::URLArgs &args );

    void captionChanged( const QString &caption );

    void locationBarURLChanged( const QString &url );

    void stopStatusChanged( bool enable );

    void backStatusChanged( bool canGoBack );

    void forwardStatusChanged( bool canGoForward );

    void statusMessage( const QString &text );

private slots:
    void setCaption( const QString &caption );

    void setLocationBarURL( const QString &url );

    void setStopEnabled( bool enable );

    void infoMessage( KIO::Job *, const QString &text );

    void htmlViewDestroyed();

    void goHistoryFromBuffer();

    void openURLNotify();

private:
    void createHistoryEntry();

    void updateHistoryEntry();

    void goHistory( int steps );

    struct HistoryEntry
    {
        KURL m_url;
        QByteArray m_buffer;
    };

    QList<HistoryEntry> m_history;

    HTMLView *m_doc;

    QString m_oldLocationBarURL;

    QGuardedPtr<Run> m_run;

    QString m_caption;

    QString m_locationBarURL;

    bool m_stop;

    QString m_findText;

    bool m_find;

    QString m_frameName;

    BrowserInterface *m_browserIface;

    int m_goBuffer;
};

#endif
