/*  This file is part of the KDE project
    Copyright (C) 2001 Simon Hausmann <hausmann@kde.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    As a special exception this program may be linked with Qt non-commercial 
    edition, the resulting executable be distributed, without including the 
    source code for the Qt non-commercial edition in the source distribution.

*/

// !! The history code is (c) David Faure <faure@kde.org> !!

#include "view.h"
#include "run.h"
#include "htmlview.h"
#include "browseriface.h"

#include <qtimer.h>

#include <khtml_part.h>
#include <khtmlview.h>
#include <kio/job.h>

#include <assert.h>

View::View( const QString &frameName, QWidget *parentWidget, const char *widgetName,
            QObject *parent, const char *name )
    : QObject( parent, name ), m_frameName( frameName )
{
    m_history.setAutoDelete( true );

    // don't specify a parent here as the javascript stuff can do
    // window.close() which results in a delete part; which in turn
    // we trigger in htmlViewDestroyed() and destroy us. Our QObject
    // destructor attempts to delete the child objects and whooops, we would
    // end up in double deletion of the part
    m_doc = new HTMLView( parentWidget, widgetName, 0, "khtmlpart" ); // ### frame name

    connect( m_doc, SIGNAL( setStatusBarText( const QString & ) ),
             this, SIGNAL( statusMessage( const QString & ) ) );

    connect( m_doc->browserExtension(), SIGNAL( openURLRequest( const KURL &, const KParts::URLArgs & ) ),
             this, SLOT( openURL( const KURL &, const KParts::URLArgs & ) ) );

    connect( m_doc->browserExtension(), SIGNAL( openURLNotify() ),
             this, SLOT( openURLNotify() ) );

    connect( m_doc, SIGNAL( started( KIO::Job * ) ),
             this, SLOT( started( KIO::Job * ) ) );

    connect( m_doc, SIGNAL( completed() ),
             this, SLOT( completed() ) );

    connect( m_doc, SIGNAL( setWindowCaption( const QString & ) ),
             this, SLOT( setCaption( const QString & ) ) );

    connect( m_doc->browserExtension(), SIGNAL( setLocationBarURL( const QString & ) ),
             this, SLOT( setLocationBarURL( const QString & ) ) );

    m_browserIface = new BrowserInterface( this, "browseriface" );

    m_doc->browserExtension()->setBrowserInterface( m_browserIface );

    // javascript stuff can do window.close()
    connect( m_doc, SIGNAL( destroyed() ),
             this, SLOT( htmlViewDestroyed() ) );

    m_find = false;
    m_stop = false;

    m_goBuffer = 0;
}

View::~View()
{
    if ( m_run )
    {
        if ( m_run->job() )
            m_run->job()->kill();

        delete static_cast<Run *>( m_run );
    }
    delete m_doc;
}

void View::openURL( const KURL &url, const KParts::URLArgs &args )
{
    if ( !args.frameName.isEmpty() )
    {
        emit openURLRequest( this, url, args );
        return;
    }

    stop();

    m_run = new Run( this, url, args );
    connect( m_run, SIGNAL( error() ),
             this, SLOT( runError() ) );

    m_oldLocationBarURL = m_locationBarURL;
    setLocationBarURL( url.prettyURL() );

    setStopEnabled( true );
}

void View::openURLNotify()
{
    updateHistoryEntry();
    createHistoryEntry();
}

void View::runError()
{
    setLocationBarURL( m_oldLocationBarURL );
    setStopEnabled( false );
}

void View::openURL( const KURL &url, const QString &mimeType, const KParts::URLArgs &_args )
{
    createHistoryEntry();

    setLocationBarURL( url.prettyURL() );

    KParts::URLArgs args = _args;
    args.serviceType = mimeType;
    m_doc->browserExtension()->setURLArgs( args );

    m_doc->openURL( url );

    updateHistoryEntry();
}

void View::stop()
{
    if ( m_run )
    {
        setLocationBarURL( m_oldLocationBarURL );

        if ( m_run->job() )
            m_run->job()->kill();

        delete static_cast<Run *>( m_run );
    }

    m_doc->closeURL();

    setStopEnabled( false );

    if ( m_history.count() > 0 )
        updateHistoryEntry();
}

void View::started( KIO::Job *job )
{
    setStopEnabled( true );

    if ( job )
        connect( job, SIGNAL( infoMessage( KIO::Job *, const QString & ) ),
                 this, SLOT( infoMessage( KIO::Job *, const QString & ) ) );
}

void View::completed()
{
    setStopEnabled( false );

    updateHistoryEntry();

    emit backStatusChanged( canGoBack() );
    emit forwardStatusChanged( canGoForward() );
}

void View::infoMessage( KIO::Job *, const QString &text )
{
    emit statusMessage( text );
}

void View::htmlViewDestroyed()
{
    m_doc = 0;
    delete this;
}

void View::createHistoryEntry()
{
    HistoryEntry *current = m_history.current();

    if ( current )
    {
        m_history.at( m_history.count() - 1 );
        while ( m_history.current() != current )
            m_history.removeLast();
    }

    m_history.append( new HistoryEntry );
}

void View::updateHistoryEntry()
{
    HistoryEntry *current = m_history.current();

    assert( current );

    QDataStream stream( current->m_buffer, IO_WriteOnly );

    m_doc->browserExtension()->saveState( stream );

    current->m_url = m_doc->url();
}

void View::back()
{
    stop();

    goHistory( -1 );
}

void View::forward()
{
    stop();

    goHistory( 1 );
}

void View::goHistoryDelayed( int steps )
{
    m_goBuffer = steps;
    QTimer::singleShot( 0, this, SLOT( goHistoryFromBuffer() ) );
}

void View::goHistoryFromBuffer()
{
    if ( m_goBuffer == 0 )
        return;

    int steps = m_goBuffer;
    m_goBuffer = 0;

    goHistory( steps );
}

void View::goHistory( int steps )
{
    int newPos = m_history.at() + steps;

    HistoryEntry *current = m_history.at( newPos );

    assert( current );

    HistoryEntry h( *current );
    h.m_buffer.detach();

    QDataStream stream( h.m_buffer, IO_ReadOnly );

    m_doc->browserExtension()->restoreState( stream );

    emit backStatusChanged( canGoBack() );
    emit forwardStatusChanged( canGoForward() );
    setLocationBarURL( h.m_url.prettyURL() );
}

void View::reload()
{
    // ### FIXME: image/text doc support
    KURL url = m_doc->url();
    if ( !url.isEmpty() && !url.isMalformed() )
    {
        KParts::URLArgs args;
        args.reload = true;
        m_doc->browserExtension()->setURLArgs( args );
        m_doc->openURL( url );
    }
}

void View::setFindEnabled( bool enable )
{
    m_find = enable;
}

void View::setFindText( const QString &text )
{
    m_findText = text;
}

void View::find()
{
    m_doc->findTextBegin();
}

void View::findNext()
{
    m_doc->findTextNext( m_findText, true, false );
}

void View::setCaption( const QString &caption )
{
    m_caption = caption;
    emit captionChanged( m_caption );
}

void View::setLocationBarURL( const QString &url )
{
    m_locationBarURL = url;
    emit locationBarURLChanged( m_locationBarURL );
}

void View::setStopEnabled( bool enable )
{
    m_stop = enable;
    emit stopStatusChanged( m_stop );
}

QStringList View::frameNames() const
{
  return childFrameNames( m_doc );
}

bool View::canHandleFrame( const QString &name, KParts::BrowserHostExtension **hostExtension )
{
    if ( !m_frameName.isEmpty() && m_frameName == name )
    {
        *hostExtension = 0;
        return true;
    }

    if ( !frameNames().contains( name ) )
        return false;

    *hostExtension = View::hostExtension( m_doc, name );
    return true;
}

QStringList View::childFrameNames( KParts::ReadOnlyPart *part )
{
  QStringList res;

  KParts::BrowserHostExtension *hostExtension = KParts::BrowserHostExtension::childObject( part );

  if ( !hostExtension )
    return res;

  res += hostExtension->frameNames();

  const QList<KParts::ReadOnlyPart> children = hostExtension->frames();
  QListIterator<KParts::ReadOnlyPart> it( children );
  for (; it.current(); ++it )
    res += childFrameNames( it.current() );

  return res;
}

KParts::BrowserHostExtension* View::hostExtension( KParts::ReadOnlyPart *part, const QString &name )
{
    KParts::BrowserHostExtension *ext = KParts::BrowserHostExtension::childObject( part );

  if ( !ext )
    return 0;

  if ( ext->frameNames().contains( name ) )
    return ext;

  const QList<KParts::ReadOnlyPart> children = ext->frames();
  QListIterator<KParts::ReadOnlyPart> it( children );
  for (; it.current(); ++it )
  {
    KParts::BrowserHostExtension *childHost = hostExtension( it.current(), name );
    if ( childHost )
      return childHost;
  }

  return 0;
}

#include "view.moc"
